 /***************************************************************************
    File        : NoiseFloorMonitor.cpp
    Description : Implements the NoiseFloorMonitor
 ---------------------------------------------------------------------------
    Begin       : Tue March 30 2005
    Author(s)   : Roberto Grosso
 ***************************************************************************/


#include "NoiseFloorMonitor.h"


//===================================== generate the main routine
EXECDAT(NoiseFloorMonitor)

//===================================== Local help functions
bool ReadNotchFilterFreq(const std::string& file,const string& channelname,const double maxfreq,gwd::Vector& freq,gwd::Vector& freqbw);
inline void GetTrend(const double sigma,gwd::Vector& ts,double& maxVal,double& percentage);


//======================================  NoiseFloorMonitor constructor
NoiseFloorMonitor::NoiseFloorMonitor(int argc, const char *argv[]) : DatEnv(argc, argv), MonServer("NoiseFloorMon")
{
  //----------------------------------  Process arguments
  //--------------------------------------------------------------------
  std::string prFile;
  std::string fqFile;

  
  // Process input parameters
  for (int i=1 ; i<argc ; i++)
  {
    std::string argi = argv[i];
    if (argi == "-param") {
      prFile = argv[++i];
    } else if (argi == "-freq") {
      fqFile = argv[++i];
    } else if (argi == "-inlist") {
      ++i;
    } else if (argi == "-maxframes") {
      ++i;
    } else if (argi == "-debug") {
      ++i;
    } else {
      std::string strError = std::string("NoiseFloorMonitor() Unrecognized argument: ") + std::string(argi);
      gwd::Error(strError);
    }
  }

  //----------------------------------  init class variables
  //--------------------------------------------------------------------
  mCounterTimeStrides  = 0;
  mTStride             = 0.;
  mWindowSizeRunningMedian = 0.;
  mSampleRate              = 0.;

  //----------------------------------  read parameters
  //--------------------------------------------------------------------
  gwd::Parameter param;
  param.ReadParameters(prFile);

  //----------------------------------  copy parameters
  //--------------------------------------------------------------------
  mWindowSizeRunningMedian = param.WindowSizeRunningMedian();
  mDetectorSite    = param.DetectorSite();
  for (int ii = 0; ii < param.NoOfMainChannels(); ii++)
    mMainChannel.push_back(param.MainChannel(ii));
  mNoChannels      = param.NoChannels();
  mChannelName.resize(mNoChannels);
  mChannelSampleRate.resize(mNoChannels);
  for (int ii = 0; ii < (int)mNoChannels; ii++)
  {
    mChannelName[ii] = param.ChannelName(ii);
    mChannelSampleRate[ii] = param.ChannelSampleRate(ii);
  }
  
  //----------------------------------  check update whitening filter
  //----------------------------------  and number of channels
  //--------------------------------------------------------------------
  mUpdateWhiteFlt = param.UpdateWhiteFilter();
  if (mUpdateWhiteFlt <= mNoChannels)
  {
    std::string strError = std::string("NoiseFllorMonitor(): number of channels is larger than update of whitening filter");
    gwd::Error(strError);
  }
  
  // Needs cutoff frequency to compute the sampling rate, Nyquist.
  double upperCutoffFreq = param.UpperCutoffFreq();
  
  //---------Find out the number of frequency bands and the frequencies
  //--------------------------------------------------------------------
  mNoFreqBands = param.NoFreqBands();
  mFreqBand.resize(mNoFreqBands);
  for (int ii = 0; ii < mNoFreqBands; ii++) mFreqBand[ii] = param.FreqBand(ii);

  //----------------------------------  set the time stride.
  //--------------------------------------------------------------------
  mTStride = param.TimeStride();
  getDacc().setStride(param.TimeStride());
  mTStrideSize = mTStride.GetSecs();
  
  //----------------------------------  Specify channel to scan
  //--------------------------------------------------------------------
  for (int ii = 0; ii < (int)mNoChannels; ii++)
  {
    getDacc().addChannel(mChannelName[ii].c_str());
  }
  // Do not know excalty, what is th e function of this call.
  getDacc().setIgnoreMissingChannel(true);

  //----------------------------------  Set up the trender
  //--------------------------------------------------------------------
  std::string IFOfile(&mMainChannel[0][0],2);
  mTrend.setIFO(IFOfile.c_str());
  std::ostringstream trN;
  trN << "NoiseFloor_" << IFOfile;
  mTrend.setName(trN.str().c_str());    //  Set trend frame name
  mTrend.setFrameCount(1);          //  Set frames per file
  mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
  mTrend.setAutoUpdate(false);      //  Avoid automatic screwup.
  
  //----------------------------------  Create the Channels
  //--------------------------------------------------------------------
  mDetectorChannel.resize(mNoChannels);
  for (int ii = 0; ii < (int)mNoChannels; ii++)
  {
    //----------------------------------  read frequecies for Notch-Filter
    //--------------------------------------------------------------------
    gwd::Vector freq;
    gwd::Vector freqbw;
    if (!ReadNotchFilterFreq(fqFile,mChannelName[ii],upperCutoffFreq,freq,freqbw))
    {
      std::cout << "ERROR: no frequencies specified for line elimination, please check file" << fqFile << std::endl;
      exit(1);
    }
    else
    {
      mDetectorChannel[ii].Init(mChannelName[ii],mChannelSampleRate[ii],param,freq,freqbw);
    }
  }

  //----------------------------------  oversampling factor for resampling and decimation
  //-------------------------------------------------------------------------------------
  mOverSamplingFactor = param.OverSamplingFactor();

  //----------------------------------  sample rate is equal twice of cutoff frequency (Nyquist)
  //--------------------------------------------------------------------------------------------
  mSampleRate = 2.*upperCutoffFreq;

  //----------------------------------  assume signal sampling rate
  //--------------------------------------------------------------------
  mSignalSampleRate = double(16384);

  //----------------------------------  sample rate must be less than or equal to signal sampling rate
  //--------------------------------------------------------------------------------------------------
  if (mSampleRate > mSignalSampleRate)
    mSampleRate = mSignalSampleRate;

  //----------------------------------  estimate threshold with Gaussian
  //----------------------------------  White noise
  //--------------------------------------------------------------------
  EstimateThreshold();

  //---------------------------------- Specify time series names
  //--------------------------------------------------------------------
  std::ostringstream chNout;
  chNout << "NoiseFloor-" << mDetectorSite;
  setServerName(chNout.str().c_str());
  // The time series
  std::vector<int> freqs(mFreqBand.size()+1);
  freqs[0] = 0;
  for (int ii = 0; ii < (int)mFreqBand.size(); ii++) freqs[ii+1] = static_cast<int>(mFreqBand[ii]);
  for (int channel = 0; channel < (int)mNoChannels; channel++)
  {
    for (int fq = 0; fq < mDetectorChannel[channel].NoFreqBands(); fq++)
    {
      // The noise floor for the frequency bands
      std::ostringstream tsName;
      tsName << mChannelName[channel] << "-" << freqs[fq] << "-" << freqs[fq+1] << "-Hz";
      MonServer::serveData(tsName.str().c_str(), mDetectorChannel[channel].GetFrequencyBand(fq));
      // The 12hs plot
      std::ostringstream tsName12hs;
      tsName12hs << mChannelName[channel] << "-" << freqs[fq] << "-" << freqs[fq+1] << "-Hz-12hs";
      MonServer::serveData(tsName12hs.str().c_str(), mDetectorChannel[channel].GetFrequencyBand12hs(fq));
    }
  }

  //----------------------------------  Specify DMT Viewer channel names
  //--------------------------------------------------------------------
  mLowerThreshold.resize(mNoFreqBands);
  mUpperThreshold.resize(mNoFreqBands);
  mLower2SigmaThreshold.resize(mNoFreqBands);
  mUpper2SigmaThreshold.resize(mNoFreqBands);
  for (int fq = 0; fq < mNoFreqBands; fq++)
  {
    std::ostringstream thresName;
    thresName << "LowerThres-" << freqs[fq] << "-" << freqs[fq+1];
    MonServer::serveData(thresName.str().c_str(), &(mLowerThreshold[fq]));
    thresName.str("");
    thresName.seekp(0);
    thresName << "UpperThres-" << freqs[fq] << "-" << freqs[fq+1];
    MonServer::serveData(thresName.str().c_str(), &(mUpperThreshold[fq]));
    thresName.str("");
    thresName.seekp(0);
    thresName << "Lower2Sigma-" << freqs[fq] << "-" << freqs[fq+1];
    MonServer::serveData(thresName.str().c_str(), &(mLower2SigmaThreshold[fq]));
    thresName.str("");
    thresName.seekp(0);
    thresName << "Upper2Sgima-" << freqs[fq] << "-" << freqs[fq+1];
    MonServer::serveData(thresName.str().c_str(), &(mUpper2SigmaThreshold[fq]));
    thresName.str("");
    thresName.seekp(0);
  }

  //----------------------------------  Set Operation State Condition
  //--------------------------------------------------------------------
  mOSC = new OperStateCondList(getDacc());
  mOSC->setStride((int)mTStride.GetSecs());
  mOSC->setDebug(0);
  mOSC->ignoreAllExcept(mMainChannel[0].c_str());
  mOSC->readConfig("LockLoss.conf");
  mLockConditionName = mMainChannel[0].substr(0,2) + std::string(":Both_arms_locked_strict_cm");
  
  //---------------------------------- TREND data
  //----------------------------------  Specify trend channels
  //--------------------------------------------------------------------
  for (int mc = 0; mc < (int)mMainChannel.size(); mc++)
  {
    std::string mainchannel(&mMainChannel[mc][0],2);
    for (int chn = (int)mMainChannel.size(); chn < (int)mNoChannels; chn++)
    {
      std::string IFO(&mChannelName[chn][0],2);
      for (int ii = 0; ii < mNoFreqBands; ii++)
      {
        gwd::FrequencyBand* noisefloor = mDetectorChannel[chn].GetNoiseFloor(ii);
        std::ostringstream trNp;
        std::ostringstream trNm;
        std::string chname = mChannelName[chn].substr(3,mChannelName[chn].size()-3);
        for (int cc = 0; cc < (int)chname.size(); cc++)
        {
          if (chname[cc] == '-') chname[cc] = '_';
        }
        trNp << IFO << ":DMT-NFLM_" << chname << "_" << ((int)noisefloor->mLowFreq) << "_" << ((int)noisefloor->mHighFreq) << "_CROSSCORRL_" << mainchannel;
        mTrendNameCrossCorr.push_back(trNp.str());
        mTrend.addChannel(trNp.str().c_str());
        //std::cout << "Trend: " << trNp.str() << std::endl;
        trNp.str("");
        trNm.str("");
        trNp.seekp(0);
        trNm.seekp(0);
      }
    }
  }
  
  for (int chn = 0; chn < (int)mMainChannel.size(); chn++)
  {
    std::string IFO(&mChannelName[chn][0],2);
    for (int ii = 0; ii < mNoFreqBands; ii++)
    {
      gwd::FrequencyBand* noisefloor = mDetectorChannel[chn].GetNoiseFloor(ii);
      std::ostringstream trNp;
      std::ostringstream trNm;
      std::string chname = mChannelName[chn].substr(3,mChannelName[chn].size()-3);
      for (int cc = 0; cc < (int)chname.size(); cc++)
      {
        if (chname[cc] == '-') chname[cc] = '_';
      }
      trNp << IFO << ":DMT-NFLM_" << chname << "_" << ((int)noisefloor->mLowFreq) << "_" << ((int)noisefloor->mHighFreq) << "_MAXVALUE";
      mTrendNameNoiseFloor.push_back(trNp.str());
      mTrend.addChannel(trNp.str().c_str());
      //std::cout << "Trend: " << trNp.str() << std::endl;
      trNp.str("");
      trNm.str("");
      trNp.seekp(0);
      trNm.seekp(0);
    }
  }
  //----------------------------------  Set file names for html output
  //--------------------------------------------------------------------
  const char* thepath  = getenv("DMTHTMLOUT");
  std::string htmlPath;
  if (!thepath)
  {
    std::cerr << "ERROR: can't get html directory from environment variable DMTHTMLOUT" << std::endl;
    exit(1);
  }
  else
    htmlPath = std::string(thepath);
  // Check if environment variable is defined
  if (htmlPath.empty())
  {
    std::cerr << "ERROR: can't set html directory from environment variable DMTHTMLOUT" << std::endl;
    exit(1);
  }
  if (htmlPath.at(htmlPath.size()-1) != '/')
    htmlPath.append("/");

  // check if directory exists and file can open and written
  //mHTMLStatusFile = htmlPath + mChannelName.substr(0,2) + std::string("NFTMStatusPage.html");
  mHTMLStatusFile = htmlPath + std::string("NFTMStatusPage.html");
  mHTMLErrorFile  = htmlPath + std::string("NFTMErrorPage.html");
  std::ofstream outStatusF(mHTMLStatusFile.c_str());
  std::ofstream outErrorF(mHTMLErrorFile.c_str());
  if (!outStatusF || !outErrorF)
  {
    std::cerr << "ERROR: can't open html file for monitor status and error report!" << std::endl;
    exit(1);
  }
  else
  {
    outStatusF.close();
    outErrorF.close();
  }
  
} // Constructor

//======================================
//======================================  Skeleton object destructor.
NoiseFloorMonitor::~NoiseFloorMonitor()
{
  std::cout << "NoiseFloorMonitor is finished" << std::endl;
}

//=================================================================================================
//=================================================================================================
//======================================  Handle Messages
//=================================================================================================
void
NoiseFloorMonitor::Attention(void)
{
    MonServer::Attention();
}

//=================================================================================================
//=================================================================================================
//======================================  Frame processing function.
//=================================================================================================
void
NoiseFloorMonitor::ProcessData(void)
{
  //msr::Clock clk;
  //clk.start();

  //----------------------------------  clear error buffer
  //-------------------------------------------------------------------------------
  gwd::Singleton* single = gwd::Singleton::exemplar();
  single->ClearMessages();
  
  //----------------------------------  Check detector's operation state condition
  //------------------------------------------------------------------------------
  mLockConditionFlag = true;
  if (mOSC->satisfied(mLockConditionName.c_str()))
  {
    mLockConditionFlag = true;
  }
  else
  {
    mLockConditionFlag = false;
  }
  //----------------------------------  Compute the Noise Floor
  //--------------------------------------------------------------------
  for (int channel = 0; channel < (int) mNoChannels; channel++)
  {
    //----------------------------------  Get pointers to the current data
    //--------------------------------------------------------------------
    const TSeries* ts = getDacc().refData(mChannelName[channel].c_str());
    if (ts) mCurrentTime = ts->getStartTime();
    else mCurrentTime += mTStrideSize;

    //----------------------------------  Process Data
    //--------------------------------------------------------------------
    unsigned int locCounter = mCounterTimeStrides + mUpdateWhiteFlt - channel;
    if (!mDetectorChannel[channel].ProcessData(ts,mCurrentTime,mLockConditionFlag,locCounter))
    {
      std::string errStr = mChannelName[channel] + " NoiseFloorMonitor::ProcessData(): ";
      std::string message = errStr + " can't process channel";
      single->AppendMessage(message);
    }
    //if (mCounterTimeStrides > 20 && mCounterTimeStrides < 23)
      //mDetectorChannel[channel].WriteData((unsigned int)mCurrentTime.getS());
   
  }

  //----------------------------------  Plot threshold values
  //----------------------------------------------------------------------------------
  ThresholdPlot();
  
  //----------------------------------  Write trend values
  //----------------------------------------------------------------------------------
  SetNoiseFloorTrend();
  
  //----------------------------------  Write Monitor status in html
  //--------------------------------------------------------------------
  DumpHTMLStatusPage();

  //----------------------------------   Write Monitor processing errors in html
  //----------------------------------------------------------------------------
  DumpHTMLErrorPage();

  //----------------------------------  get processing time
  //--------------------------------------------------------------------
  //clk.stop();
  //clk.print(" ... processing time");
  
  //----------------------------------  update counter
  //--------------------------------------------------------------------
  mCounterTimeStrides++;
}


//=================================================================================================
//=================================================================================================
//====================================== Estimate Threshold from Monte Carlo simulation
//=================================================================================================
void
NoiseFloorMonitor::EstimateThreshold( )
{
  // Calculate threshold
  //========================================================================================
  mSampleRateFreqBand.resize(mNoFreqBands);
  mMeanValue.resize(mNoFreqBands);
  mStdDev.resize(mNoFreqBands);
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    // compute sampling rate
    double samplerate = 2.*mOverSamplingFactor*mFreqBand[ii];
    int    decimate  = (int)std::floor(mSampleRate/samplerate);
    if (decimate < 1) decimate = 1;
    samplerate = mSampleRate / static_cast<double>(decimate);

    // compute running median window size
    unsigned int rmwz = static_cast<unsigned int>(mWindowSizeRunningMedian*samplerate);
    unsigned int lcSz = static_cast<unsigned int>(mTStride.GetSecs()*samplerate);

    gwd::Vector noise(lcSz);
    gwd::NormalDistribution normal;

    for(unsigned int nn = 0; nn < lcSz; nn++)
    {
      noise[nn] = normal.Normal(0.,1.);
      noise[nn] = gwd::square(noise[nn]);
    }
    gwd::RunningMedian rm;
    gwd::Vector rmTS;
    rm.ComputeMedians(rmwz,noise,rmTS);

    // Compute mean and std. deviation
    double meanValue;
    double stdDev;
    Estimator(rmTS,meanValue,stdDev);
    mSampleRateFreqBand[ii] = samplerate;
    mMeanValue[ii] = meanValue;
    mStdDev[ii] = stdDev;
  }
}



//======================================
//====================================== DMT-viewer: threshold plot
void
NoiseFloorMonitor::ThresholdPlot()
{
  Time t0 = mCurrentTime;
  std::vector<float> ouTS(1000);
  Interval dt = mTStride / static_cast<double>(ouTS.size()-1);

  for (int fq = 0; fq < mNoFreqBands; fq++)
  {
    //----------------------------------  Band
    double locStdDev = mStdDev[fq];
    std::fill(ouTS.begin(),ouTS.end(),3.*locStdDev);
    mUpperThreshold[fq].setData(t0, dt, &ouTS[0], ouTS.size());
    std::fill(ouTS.begin(),ouTS.end(),-3.*locStdDev);
    mLowerThreshold[fq].setData(t0, dt, &ouTS[0], ouTS.size());
    std::fill(ouTS.begin(),ouTS.end(),2.*locStdDev);
    mUpper2SigmaThreshold[fq].setData(t0, dt, &ouTS[0], ouTS.size());
    std::fill(ouTS.begin(),ouTS.end(),-2.*locStdDev);
    mLower2SigmaThreshold[fq].setData(t0, dt, &ouTS[0], ouTS.size());
  }
}


//=================================================================================================
//=================================================================================================
//====================================== Set the noise floor trend and 12hs plot
//=================================================================================================
void
NoiseFloorMonitor::SetNoiseFloorTrend()
{
  // For the twelve hour plot use always
  // 12hs = 43200 sec
  Time t0 = mCurrentTime;
  Time t12 = mCurrentTime - Interval(43200.);
  Interval dt12;
  Interval stride(double(43200));
  // data buffer
  gwd::Vector ouTS;

  //----------------------------------  Process the 12hs plot
  //----------------------------------  secondary channels
  //--------------------------------------------------------------------
  //for (int chn = (int)mMainChannel.size(); chn < (int)mNoChannels; chn++)
  //{
  //  for (int ii = 0; ii < mNoFreqBands; ii++)
  //  {
  //    double maxVal = double(0);
  //    // Get frequency band
  //    gwd::FrequencyBand* noisefloor = mDetectorChannel[chn].GetNoiseFloor(ii);
  //    if (!mDetectorChannel[chn].empty())
  //    {
  //      //----------------------------------  set band data
  //      //double StdDev = noisefloor->mStdDev;
  //      double StdDev = mStdDev[ii];
  //      double percentage;
  //      GetTrend(StdDev,noisefloor->mNoiseFloor,maxVal,percentage);
  //    }
  //    noisefloor->m12hsPlot.pop_front();
  //    noisefloor->m12hsPlot.push_back(maxVal);
  //    ouTS.resize(noisefloor->m12hsPlot.size());
  //    std::copy(noisefloor->m12hsPlot.begin(),noisefloor->m12hsPlot.end(),ouTS.begin());
  //    dt12 = stride / static_cast<double>(ouTS.size()-1);
  //    noisefloor->mTS12hs.setData(t12, dt12, &ouTS[0], ouTS.size());
  //  }
  //}
  
  //---------------------------------- TREND data
  //----------------------------------  cross correlation
  //--------------------------------------------------------------------
  int counter = 0;
  for (int mc = 0; mc < (int)mMainChannel.size(); mc++)
  {
    for (int chn = (int)mMainChannel.size(); chn < (int)mNoChannels; chn++)
    {
      for (int ii = 0; ii < mNoFreqBands; ii++)
      {
        //if (!mTrend.exists(mTrendNameCrossCorr[counter].c_str()))
          //std::cout << "ERROR: trend channel not exists!" << std::endl;
        if (mDetectorChannel[mc].empty() || mDetectorChannel[chn].empty())
        {
          // write zeros here
          double crosscorr = double(0);
          //std::cout << "set ZERO trend for channel" << mChannelName[chn] << " max values: " <<  crosscorr << std::endl;
          mTrend.trendData(mTrendNameCrossCorr[counter].c_str(),t0,crosscorr);
        }
        else
        {
          gwd::FrequencyBand* mchnoise = mDetectorChannel[mc].GetNoiseFloor(ii);
          gwd::FrequencyBand* schnoise = mDetectorChannel[chn].GetNoiseFloor(ii);
          // compute the cross correlation
          gwd::Vector cc;
          const int lags = mchnoise->mNoiseFloor.size() / 10;
          gwd::Crosscorrelation(lags,mchnoise->mNoiseFloor,schnoise->mNoiseFloor,cc);
          double maxVal = double(0);
          for (int jj = 0; jj < (int)cc.size(); jj++)
          {
            if (maxVal < fabs(cc[jj])) maxVal = fabs(cc[jj]);
          }
          // set trend
          //std::cout << "set trend for channel" << mChannelName[chn] << " max values: " << maxVal << std::endl;
          mTrend.trendData(mTrendNameCrossCorr[counter].c_str(),t0,maxVal);
        }
        // update trend name counter
        //std::cout << "Trend write to: " << mTrendNameCrossCorr[counter] << std::endl;
        counter++;
      }
    }
  }
  //---------------------------------- TREND data
  //----------------------------------  main channels
  //--------------------------------------------------------------------
  counter = 0;
  for (int chn = 0; chn < (int)mMainChannel.size(); chn++)
  {
    for (int ii = 0; ii < mNoFreqBands; ii++)
    {
      double maxVal = double(0);
      // Get frequency band
      gwd::FrequencyBand* noisefloor = mDetectorChannel[chn].GetNoiseFloor(ii);
      if (!mDetectorChannel[chn].empty())
      {
        //----------------------------------  set band data
        //double StdDev = noisefloor->mStdDev;
        double StdDev = mStdDev[ii];
        double percentage;
        GetTrend(StdDev,noisefloor->mNoiseFloor,maxVal,percentage);
      }
      noisefloor->m12hsPlot.pop_front();
      noisefloor->m12hsPlot.push_back(maxVal);
      ouTS.resize(noisefloor->m12hsPlot.size());
      std::copy(noisefloor->m12hsPlot.begin(),noisefloor->m12hsPlot.end(),ouTS.begin());
      dt12 = stride / static_cast<double>(ouTS.size()-1);
      noisefloor->mTS12hs.setData(t12, dt12, &ouTS[0], ouTS.size());
      // set trend
      //std::cout << "set trend for channel" << mChannelName[chn] << std::endl;
      //if (!mTrend.exists(mTrendNameNoiseFloor[counter].c_str()))
          //std::cout << "ERROR: trend channel not exists!" << std::endl;
      mTrend.trendData(mTrendNameNoiseFloor[counter].c_str(),t0,maxVal);
      //std::cout << "Trend write to: " << mTrendNameNoiseFloor[counter] << "  max value: " << maxVal << std::endl;
      counter++;
    }
  }
  mTrend.Update();
}


//======================================
//======================================  Write html Monitor status document.
void
NoiseFloorMonitor::DumpHTMLStatusPage(void)
{
  // Create a link to main page
  html::link linkToMainPage("Link to Monitor's Main Page","index.html");
  
  // Status Page
  // Set the current time
  char cbuf[1024];
  LocalStr(Now(), cbuf, "%M %d, %Y %02H:%02N");

  // Create html document
  html::document doc("Noise Floor Monitor Status Page");
  doc.setBackgroundColor(html::color("wheat"));

  // Write title
  html::text myTitle(std::string("Status page of the Noise Floor Tracker Monitor"));
  myTitle.setSize(html::size(2));
  html::block cb01("center");
  cb01.addObject(myTitle);
  doc.addObject(cb01);

  html::hline horizontalline;
  doc.addObject(horizontalline);

  html::text ln01(string("Last Noise Floor statistic at: ") + cbuf);
  //ln01.setSize(html::size(2));
  html::block lb01("left");
  lb01.lineBreak();
  lb01.addObject(ln01);
  doc.addObject(lb01);

  if  (mLockConditionFlag)
  {
    html::text ln02(std::string("Detector Lock Status: locked"));
    html::text ln03(std::string("Detector Lock Condition: ") + mLockConditionName);
    html::block lb02("left");
    lb02.lineBreak();
    lb02.addObject(ln02);
    lb02.lineBreak();
    lb02.addObject(ln03);
    doc.addObject(lb02);
  }
  else
  {
    html::text ln02(std::string("Detector Lock Status: not locked"));
    html::text ln03(std::string("Detector Lock Condition: ") + mLockConditionName);
    html::block lb02("left");
    lb02.lineBreak();
    lb02.addObject(ln02);
    lb02.lineBreak();
    lb02.addObject(ln03);
    doc.addObject(lb02);
  }

  // write number and name of channels being processed
  std::ostringstream strChannel;
  strChannel << "Processing " << mNoChannels << " channels";
  html::text ln04(strChannel.str());
  html::block chnBlock("left");
  chnBlock.addObject(ln04);
  chnBlock.lineBreak();
  for (int ii = 0; ii < (int)mNoChannels; ii++)
  {
    std::ostringstream strChannelName;
    strChannelName << "Channel:  " << mChannelName[ii];
    html::text ln05(strChannelName.str());
    chnBlock.addObject(ln05);
    chnBlock.lineBreak();
    strChannelName.str("");
  }
  doc.addObject(horizontalline);
  doc.addObject(chnBlock);

  // write threshold values for frequency bands
  std::ostringstream thrFreq;
  thrFreq << "Processing " << mNoFreqBands << " frequency bands";
  html::text ln06(thrFreq.str());
  html::block thrBlock("left");
  thrBlock.addObject(ln06);
  thrBlock.lineBreak();
  std::vector<int> frequency(mNoFreqBands+1);
  frequency[0] = 0;
  for (int ii = 0; ii < mNoFreqBands; ii++)
  {
    frequency[ii+1] = static_cast<int>(mFreqBand[ii]);
    thrFreq.str("");
    thrFreq.seekp(0);
    thrFreq << "Threshold band " << frequency[ii] << "-" << frequency[ii+1] << ": " << mStdDev[ii];
    html::text ln07(thrFreq.str());
    thrBlock.addObject(ln07);
    thrBlock.lineBreak();
  }
  
  doc.addObject(horizontalline);
  doc.addObject(thrBlock);

  // Link back to main page
  html::block spb("left");
  spb.lineBreak();
  doc.addObject(spb);
  doc.addObject(horizontalline);
  doc.addObject(linkToMainPage);

  spb.lineBreak();

  // Write into file
  ofstream f(mHTMLStatusFile.c_str());
  html::writer w(f);
  doc.write(w);
}


//======================================
//======================================  Write html Monitor status document.
void
NoiseFloorMonitor::DumpHTMLErrorPage(void)
{
  // Auxiliary data
  char cbuf[1024];
  LocalStr(Now(), cbuf, "%M %d, %Y %02H:%02N");
  // Error Page
  // Create html document
  html::document doc("Noise Floor Monitor Error Page");
  doc.setBackgroundColor(html::color("wheat"));

  // Write title
  html::text myTitleErrPage(std::string("Error page of the Noise Floor Tracker Monitor"));
  myTitleErrPage.setSize(html::size(2));
  html::block cb01ErrPage("center");
  cb01ErrPage.addObject(myTitleErrPage);
  doc.addObject(cb01ErrPage);

   html::text timeLine(std::string("Last Noise Floor statistic at: ") + cbuf);
  //ln01.setSize(html::size(2));
  html::block timeBlock("left");
  timeBlock.lineBreak();
  timeBlock.addObject(timeLine);
  doc.addObject(timeBlock);
  
  
  // Add an horizontal line
  html::hline horizontalline;
  doc.addObject(horizontalline);

  // Write out the lock sate
  if  (mLockConditionFlag)
  {
    html::text ln01ErrPage(std::string("Detector Lock Satus: locked"));
    html::text ln02ErrPage(std::string("Detector Lock Condition: ") + mLockConditionName);
    html::block lb01ErrPage("left");
    lb01ErrPage.lineBreak();
    lb01ErrPage.addObject(ln01ErrPage);
    lb01ErrPage.lineBreak();
    lb01ErrPage.addObject(ln02ErrPage);
    doc.addObject(lb01ErrPage);
    lb01ErrPage.lineBreak();
    lb01ErrPage.lineBreak();
  }
  else
  {
    html::text ln01ErrPage(std::string("Detector Lock Status: not locked"));
    html::text ln02ErrPage(std::string("Detector Lock Condition: ") + mLockConditionName);
    html::block lb01ErrPage("left");
    lb01ErrPage.lineBreak();
    lb01ErrPage.addObject(ln01ErrPage);
    lb01ErrPage.lineBreak();
    lb01ErrPage.addObject(ln02ErrPage);
    doc.addObject(lb01ErrPage);
    lb01ErrPage.lineBreak();
    lb01ErrPage.lineBreak();
  }

  // Process Error messages
  doc.addObject(horizontalline);
  gwd::Singleton* single = gwd::Singleton::exemplar();
  if (single->empty())
  {
    html::text ln03ErrPage(std::string("No error messages to be reported at: ") + cbuf);
    html::block lb03ErrPage("left");
    lb03ErrPage.lineBreak();
    lb03ErrPage.addObject(ln03ErrPage);
    doc.addObject(lb03ErrPage);
  }
  else
  {
    html::block eBlock("left");
    gwd::Singleton::Iterator p;
    for (p = single->begin(); p != single->end(); ++p)
    {
      html::text line(*(*p));
      eBlock.addObject(line);
      eBlock.lineBreak();
    }
    doc.addObject(eBlock);
  }

  // Link back to main page
  html::block spb("left");
  doc.addObject(spb);
  doc.addObject(horizontalline);
  html::link linkToMainPage("Link to Monitor's Main Page","index.html");
  doc.addObject(linkToMainPage);

  // Write into file
  std::ofstream ferr(mHTMLErrorFile.c_str());
  html::writer werr(ferr);
  doc.write(werr);

  // Clear message buffer
  single->ClearMessages();
}


//==================================================================================================
//==================================================================================================
//====================================== Estimate mean value and standard deviation of a time series
void NoiseFloorMonitor::Estimator(const gwd::Vector& ts,double& mean,double& stdDev)
{
  const int lcSize = (int) ts.size();
  mean = double(0);
  for (int ii = 0; ii < lcSize; ii++)
  {
    mean += ts[ii];
  }

  mean /= static_cast<double>(lcSize);

  stdDev = double(0);
  for (int ii = 0; ii < lcSize; ii++)
  {
    stdDev += gwd::square(ts[ii]-mean);
  }

  stdDev /= static_cast<double>(lcSize-1);
  stdDev = std::sqrt(stdDev);

}



//##########################################################################################################
//
//===================================== Auxiliary functions
//
//##########################################################################################################

//======================================
//======================================  Read frequencies and band with for the Notch filter
bool ReadNotchFilterFreq(const std::string& file,const string& channelname,const double maxfreq,gwd::Vector& freq,gwd::Vector& freqbw)
{
  std::ifstream ifd(file.c_str());
  bool readFlag = false;
  double val1,val2;
  //double dummy;
  if (ifd.fail())
  {
    std::string errText = "can't open file " + file;
    std::cout << "ERROR: " << errText << std::endl;
    exit(1);
  }
  while (!ifd.eof())
  {
    std::string channel;
    ifd >> channel;
    if (channel == channelname)
    {
      readFlag = true;
      int noFreqs;
      ifd >> noFreqs;
      for (int ii = 0; ii < noFreqs; ii++)
      {
        ifd >> val1 >> val2;
        if (val1 < maxfreq)
        {
          freq.push_back(val1);
          freqbw.push_back(val2);
        }
      }
      break;
    }
  }
  ifd.close();

  return readFlag;
}


//=================================================================================================
//=================================================================================================
//=================================================================================================
//=================================================================================================
inline void GetTrend(const double sigma,gwd::Vector& ts,double& maxVal,double& percentage)
{
  const unsigned int size = static_cast<unsigned int>(ts.size());
  const double threshold = 3.*sigma;
  int counter = 0;
  maxVal = 0;

  for (unsigned int ii = 0; ii < size; ii++)
  {
    // Check fp numbers
    //if (isnand(ts[ii]))
      //std::cout << "IsNaN found!" << std::endl;
    //else if (isinf(ts[ii]))
      //std::cout << "isinf found!" << std::endl;
      
    // count values above and below threshold
    if (fabs(ts[ii]) > threshold) counter++;
    
    // get min. and max values
    maxVal = (maxVal > fabs(ts[ii])) ? maxVal : fabs(ts[ii]);
  }

  percentage = static_cast<double>(counter*100)/static_cast<double>(size);
}

