#ifndef MULTIVOLT_HH
#define MULTIVOLT_HH


#include "DatEnv.hh"
#include "MonServer.hh"
#include "VoltBase.hh"
#include <string>


   class VoltWriter;

/** @name MultiVolt monitor program
   
    MultiVolt monitors the power grid at the LIGO Observatories.
    Power line monitors are installed at each rack location in the
    LVEA and the mid and end stations. The monitor program determines
    frequency, rms, crest factor, total harmonic distortion, and
    the amplitudes of the first 17 harmonics. Typically, MultiVolt 
    will take a measurement every 10 second, average it over 6 points 
    and write a new minute trend point once every minute. It also
    has the capability to write a html summary page which is updated
    once a minute.
    
\begin{verbatim}
   usage: MultiVolt -c conf -f file [-w web] [-d dt] [-b len] [-a avg] [-moq]
          MultiVolt -c conf -t trend [-w web] [-d dt] [-b len] [-a avg] [-moiq]
          MultiVolt -c conf -T [-w web] [-d dt] [-b len] [-a avg] [-moiq]
              -c conf - configuration file
              -f file - ASCII output file, channels are stored in out.xx
              -t trend - trend file for output
	      -T - write trend to default location
              -w web - filename for web page summary information
              -d dt - number of measurements (default 0 for running indefinitely)
              -b len - number of seconds per average (default 10)
              -a avg - number of averages per measurement (default 6)
              -m - no monitor server
              -o - suppress harmonics in output
	      -i - make trend file index
              -q - quiet (no screen output)
\end{verbatim}

    A report describing in detail the layout, implementation and performance
    of the power lime monitors can be found in 
    \URL[T000104-00]{T000104-00.pdf}.

    @author Charlie Shapiro

*/

//@{

/** MultiVolt is the main object implementing the main program

    Source code:
    \URL[MultiVolt.hh]{source/MultiVolt.hh} and
    \URL[MultiVolt.cc]{source/MultiVolt.cc}

@memo	MultiVolt main program
@author Charlie Shapiro
*/
   class MultiVolt : public DatEnv, MonServer {
   public:
      // Note that these were added 9/27/01 by JGZ to make MultiVolt.cc 
      // compile with -D_POSIX_C_SOURCE=199506L -D_XOPEN_SOURCE=500 on Solaris
      typedef unsigned short u_short;
      typedef unsigned long  u_long;

      /// Constructor for MultiVolt
      MultiVolt (int argc, const char* argv[]);
      /// Destructor for MultiVolt
      virtual ~MultiVolt();
      /// Process data
      void ProcessData();
      /// Attention
      void Attention();
      /// Reset
      void Reset();
   
   protected:
      /// Error flag
      bool errflag;
      /// No output to screen
      bool quiet;
      /// is a monitor server?
      bool ismonserv;
      /// write trends?
      int usetrend;
      /// write harmonics
      bool useharmonics;
      /// make trend file index
      bool makeindex;
      /// name of configuration file
      std::string configfile;
      /// name of output file
      std::string filename;
      /// name of trend file
      std::string trendfile;
      /// name of HTML file
      std::string HTMLfile;
      /// Duration the monitor should run (0 of indefinite)
      u_short duration;
      /// Number of buffers used for a single measurement
      u_short bufSize;
      /// Number of averages use for a data point
      int avg;
   
      /// Volt Writer
      VoltWriter* VWriter;
      /// Volt Watchers
      VoltWatcher* array[maxChannels];
   
      /// Array of ID#s from config file
      int IDnums[maxChannels];
      /// Variable to hold channel ID#s
      int ID;
      /// array size
      int array_size;
      /// Number times Fill() was called
      u_long count;
      /// GPS time when Fill() was called
      u_long fillTime;
      /// Last GPS time when Fill() was called
      u_long oldTime;
      /// Index of VW array element to be analyzed
      u_short whosTurn;
      /// initial countdown
      int countdown;
   
      /// Print help message
      void HelpMessage();
      /// Get number of channels
      int GetNChannels ();
      /// Make volt watchers
      void MakeWatchers ();
   };

//@Include: VoltBase.hh
//@Include: VoltWatcher.hh
//@Include: VoltWriter.hh

//@}


#endif
