//
//    File LightMon.cc
//    K. Riles - kriles@umich.edu
//
//    Monitor light intensity in interferometer arms in order to
//    create data quality flags for
//
//       OUT_OF_LOCK
//
//       PRE_LOCKLOSS_10_SEC PRE_LOCKLOSS_30_SEC PRE_LOCKLOSS_60_SEC PRE_LOCKLOSS_120_SEC 
//
//       LIGHTDIP_01_PERCENT LIGHTDIP_02_PERCENT LIGHTDIP_03_PERCENT LIGHTDIP_04_PERCENT 
//       LIGHTDIP_05_PERCENT LIGHTDIP_06_PERCENT LIGHTDIP_07_PERCENT LIGHTDIP_08_PERCENT 
//       LIGHTDIP_09_PERCENT LIGHTDIP_10_PERCENT LIGHTDIP_12_PERCENT LIGHTDIP_15_PERCENT 
//       LIGHTDIP_20_PERCENT 
//    
//    Structure borrows heavily from Dave Chin's LockLoss program
//
//    $Id: LightMon.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $

#include "LightMon.hh"

#ifndef __CINT__

#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <cstring>
#include <vector>

#include "TSeries.hh"
#include "FSeries.hh"
#include "Dacc.hh"
#include "DatEnv.hh"
#include "TrigRslt.hh"

// The next three lines must appear, and in this sequence
#define  PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/LightMon/LightMon.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $"
#define  PIDTITLE  "Watch for loss and acquiring of lock"
#include "ProcIdent.hh"

//
// Generate the main routine.
//
EXECDAT(LightMon) ;
#endif // !def(__CINT__)

using namespace std;


LightMon::LightMon(int argc, const char *argv[]) 
    : DatEnv(argc, argv),
      mDebug(0),
      mMaxFrame(999999),
      mTotalRunTime(0,0),
      mOsclist(getDacc(), 0),
      mTstampFormat("%Y-%m-%d %H:%N:%S %Z"),
      mName("LightMon_"),
      mBothLock(-1,mMaxLength),
      mBothLock1min(-1,mMaxLength1min),
      mLockState(-1,mMaxLength1min),
      mBothLock6hr(-1,mMaxLength6hr),
      mBothLock12hr(-1,mMaxLength12hr5),
      mArmPower(-1,mMaxLength1hr),
      mArmPowerNoDip(-1,mMaxLength1hr)
{
    if (argc < 3) {
        cerr << "Must specify IFO name (H1, H2, or L1)" << endl;
        cerr << "Usage: LightMon -ifo ifoname [-debug n] " << endl;
        std::string eObj = "Must specify IFO name (H1, H2, or L1)";
        throw runtime_error(eObj);
    }

    // Set pre-lockloss time intervals
    
    mN_prelockloss = 6;
    mNsec_prelockloss[0] = 10;
    mNsec_prelockloss[1] = 30;
    mNsec_prelockloss[2] = 60;
    mNsec_prelockloss[3] = 120;
    mNsec_prelockloss[4] = 600;
    mNsec_prelockloss[5] = 1800;
    

    // Set lightdip loss fractions

    mN_lightdip = 13;
    mFract_lightdip[0] = 0.01;
    mFract_lightdip[1] = 0.02;
    mFract_lightdip[2] = 0.03;
    mFract_lightdip[3] = 0.04;
   mFract_lightdip[4] = 0.05;
    mFract_lightdip[5] = 0.06;
    mFract_lightdip[6] = 0.07;
    mFract_lightdip[7] = 0.08;
    mFract_lightdip[8] = 0.09;
    mFract_lightdip[9] = 0.10;
    mFract_lightdip[10] = 0.12;
    mFract_lightdip[11] = 0.15;
    mFract_lightdip[12] = 0.20;

    // Set OUT_OF_LOCK to false at start

    mOUT_OF_LOCK_already_true = false;

    // Disable sending of triggers if offline

    //    mSendTriggers = true;
    mSendTriggers = false;
 
    bool ifoset_p = false;
    
    for (int i = 0; i < argc; ++i) {
        if (strcmp(argv[i], "-debug") == 0) {
            ++i;
            mDebug = atoi(argv[i]);
        }

        if (strcmp(argv[i], "-ifo") == 0) {
            ++i;

            if (i > argc-1 || argv[i] == NULL) {
                std::string eObj = "Must specify IFO name (H1, H2, or L1)";
                cerr << eObj << endl;
                throw runtime_error(eObj);
            }

            mIFOname = argv[i];
            ifoset_p = true;
            
            if (mIFOname != "H1" && mIFOname != "H2" && mIFOname != "L1") {
                std::string eObj = "Must specify IFO name (H1, H2, or L1)";
                cerr << eObj << endl;
                throw runtime_error(eObj);
            }
        }
    }

    if (ifoset_p == false) {
        std::string eObj = "Must specify IFO name (H1, H2, or L1)";
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }

    mName += mIFOname;
    if (mDebug == 0) {
        MonServer::setServerName(mName.c_str());
    } else {
        mName += "_DEBUG";
        MonServer::setServerName(mName.c_str());
    }

    // set debug level on OSC list
    mOsclist.setDebug(0);
    
    // Write to $DMTHTMLOUT or $GDSAPACHE/monitor_reports/mName/
    char *htmldir = getenv("DMTHTMLOUT");
    if (!htmldir) {
        cerr << mName << ": warning -- environment variable DMTHTMLOUT "
             << "not set. Trying GDSAPACHE." << endl;
        
        htmldir = getenv("GDSAPACHE");
        if (!htmldir) {
            cerr << mName << ": warning -- environment variable GDSAPACHE "
                 << "not set. Writing log and duty cycle files to "
                 << "current directory" << endl;
        } else {
            cerr << mName << ": Writing log and duty cycle files to "
                 << "$GDSAPACHE/monitor_reports/" << mName << "/"
                 << endl;

            mHtmlDir  = htmldir;
            mHtmlDir += "/monitor_reports/" + mName + "/";
        }
    } else {
        cerr << mName << ": Writing log and duty cycle files to "
             << htmldir << endl;

        mHtmlDir = htmldir;
        mHtmlDir += "/";
    }

    //
    // Set the time stride
    //
    float tStep(1.0);
    getDacc().setStride(tStep);

    // Initialize frame counter
    mFrameCount = 0;

    // Read in configuration file (but first check we are right observatory)

    char *ifocstr = getenv("DMTIFOS");
    if (!ifocstr) {
        std::string eObj =
            "LightMon: Don't know if we're at Hanford or Livingston. ";
        eObj += "Set the environment variable DMTIFOS and restart.";
        std::cerr << eObj << std::endl;
        throw runtime_error(eObj);
    } else {
        std::string ifos(ifocstr);
        // std::string parPath(dmtparcstr);

        // Make sure requested IFO is available at this site
        if (ifos.find(mIFOname) == std::string::npos) {
            std::string eObj = "Specified IFO (";
            eObj += mIFOname;
            eObj += ") is not available at this site";
            std::cerr << eObj << std::endl;
            throw runtime_error(eObj);
        } else {
	    string osc_config;
	    if (getenv("DMTPARS")) {
	        osc_config = getenv("DMTPARS");
		osc_config += "/";
	    } else if (mDebug > 0) {
	        osc_config = "/export/home/ops/pars/";
	    }
	    // read config file from specified directory
	    osc_config += "LockLoss.conf";
	    mOsclist.readConfig(osc_config.c_str());
        }
    }

    // Dump all conditions that we don't care about
    if (mDebug > 0)
        cerr << mName << ": dumping unneeded IFOs" << endl;
   
    mOsclist.ignoreAllExcept(mIFOname);

    if (mDebug > 1)
        mOsclist.chanUtilInfo();

    // Once config file is read, query OSCs for channel names so we
    // know the IFO name with which to tag our triggers.  Do this
    // even though the mIFOname is set, i.e. don't trust it.

    std::string ifoname;
    ifoname = mOsclist[mIFOname + ":sgl_X_arm_thres"]->channel();

    if (mDebug > 2) {
        cerr << mName << ": ifoname = " << ifoname << endl;
        cerr << "          substr = " << ifoname.substr(0,2) << endl;
    }
    
    if (mIFOname != ifoname.substr(0,2)) {
        cerr << "IFO name specified on command line does not match IFO "
             << "name in config file" << endl;
        std::string eObj = "IFO name on cmdline does not match IFO name in config file";
        throw runtime_error(eObj);
    } 

    if (mDebug > 0) {
        cerr << mName << ": mIFOname = " << mIFOname << endl;
    }


    setTSWindowStuffMod();

    // once the filenames for the TSWindow objects have been set,
    // we can restore them
    restoreAllTSWindowsMod();
    
    //
    // Serve data to DMTviewer
    //
    serveData(mBothLock.name().c_str(), &mBts);
    serveData(mBothLock6hr.name().c_str(), &mBts6hr);
    serveData(mBothLock12hr.name().c_str(), &mBts12hr);
    serveData((mIFOname+":BothArmsLocked12hExport_LightMon").c_str(), &mBts12hrExp);
    serveData((mIFOname+":ArmPowerAll_LightMon").c_str(), &mArmPowerTS);
    serveData((mIFOname+":ArmPowerDipsExcluded_LightMon").c_str(), &mArmPowerNoDipTS);

    //
    // Set up trending
    //

    mTrendChanNames["Both"] = mIFOname + ":DMT-LGHT_BOTH_ARMS_LOCKED";
    mTrendChanNames["LockState"] = mIFOname + ":DMT-LGHT_STATE_VECTOR";

    char *tdir = getenv("DMTRENDOUT");
    std::string trendDir;
    if (!tdir) {
        trendDir = "/dmt/";
        trendDir += mName;
        trendDir += "/";
        cout << "trendDir now defined to be " << trendDir << endl;
    } else {
        trendDir = tdir;
        cout << "trendDir already defined to be " << trendDir << endl;
    }

    if (mDebug > 2) {
        std::cout << "trendDir = " << trendDir << std::endl;
    }
    
    // trend frame
    mTrend.setIFO(mIFOname.c_str());
    mTrend.setName(mName.c_str());
    mTrend.setType(Trend::kMinute);
    mTrend.setFrameCount(1);
    mTrend.setAutoUpdate(false);

    // add trend channels
    cout << "Adding Both trend channel: " << mTrendChanNames["Both"] << endl;
    mTrend.addChannel(mTrendChanNames["Both"].c_str());
    cout << "Adding State trend channel: " << mTrendChanNames["LockState"] << endl;
    mTrend.addChannel(mTrendChanNames["LockState"].c_str());

    // Write index file with trend channel names
    char indexfile[512];
    sprintf(indexfile, "%s/channel.cfg", trendDir.c_str());
    if (mDebug > 0)
        std::cout << "Writing index file " << indexfile << std::endl;
    mTrend.writeIndex(indexfile);

    // serve up trends to DMTviewer
    const std::string ntag = " min. trend - 1hr";
    
    std::string dmtviewerName = mTrendChanNames["Both"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["Both"].c_str()).refAvgSeries()),
              "LightMon: Both-arms lock status minute trend");

    dmtviewerName = mTrendChanNames["LockState"] + ntag;
    serveData(dmtviewerName.c_str(),
              &(mTrend.find(mTrendChanNames["LockState"].c_str()).refAvgSeries()),
	      "LightMon: Lock State minute trend");

    if (mDebug > 0) {
        std::cout << mName << ": OSC info" << std::endl;
        OperStateCondList::const_iterator iter = mOsclist.begin();
        for ( ; iter != mOsclist.end(); ++iter) {
            std::cout << (*iter).first << ": ";
            (*iter).second->printInfo();
        }

        cerr << mName << ": exiting constructor" << endl;
    }
}

LightMon::~LightMon() 
{
    mTrend.close();

    saveAllTSWindowsMod();
    
    Time now = getDacc().getFillTime();
    *mLog << "LightMon is finished at time: " << now << endl;
    *mLog << endl << "***********************************************" 
          << endl << endl;

    mLogfile.close();
}

//
// For the BothArmsLocked conditions, need to incorporate
// different values in the TSWindow for the different 
// Lock Modes, i.e. run mode, common mode, etc.
//
// Talk with Dave Barker, 2002-Aug-19
//   value            meaning
//   1                mode cleaner only locked (both-arms not locked)
//   2                both-arms locked
//   3                both-arms locked + common mode
//   4                both-arms locked + Op Go button on
//
//   The Op Go MEDM button is constrained by the state vector bits:
//   it can only be turned on if ALL state vector bits are on

//
// BothArmsLockedActions()
//
void LightMon::BothArmsLockedActions(const Time &now)
{
        
    mBothLock.append(1, now);
    mBothLock1min.append(1, now);

    // update the lock state trend every stride
    if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
        if (mDebug>1) cout << "Appending 4 to mLockState" << endl;
        mLockState.append(4, now);
    } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
        if (mDebug>1) cout << "Appending 3 to mLockState" << endl;
        mLockState.append(3, now);
    } else {
        if (mDebug>1) cout << "Appending 2 to mLockState" << endl;
        mLockState.append(2, now);
    }

    // update the 6hr series only every 5 seconds
    if (now.getS() % 5 == 0) {
        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
            mBothLock6hr.append(4, now);
            if (mDebug > 1)
                cout << "Appending 4" << endl;
        } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
            mBothLock6hr.append(3, now);
            if (mDebug > 1)
                cout << "Appending 3" << endl;
        } else {
            mBothLock6hr.append(2, now);
            if (mDebug > 1)
                cout << "Appending 2" << endl;
        } 
    }

    // update the 12hr series every 6 seconds
    if (now.getS() % 6 == 0) {
        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_op_go")) {
            mBothLock12hr.append(4, now);
        } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked_common_mode")) {
            mBothLock12hr.append(3, now);
        } else {
            mBothLock12hr.append(2, now);
        }
    }
        
    if (mFrameCount == 1)
        mBacquiredP = false;
} // END: BothArmsLockedActions()


//
// BothArmsNotLockedActions()
//
void LightMon::BothArmsNotLockedActions(const Time &now)
{

  mBothLock.append(0, now);
  mBothLock1min.append(0, now);
  
  // it could be that the ModeCleaner is in lock
  
  // update lock state every stride
  if (mOsclist.satisfied(mIFOname + ":Mode_Cleaner_locked")) {
    if (mDebug>1) cout << "Appending 1 to mLockState" << endl;
    mLockState.append(1, now);
  } else {
    if (mDebug>1) cout << "Appending 0 to mLockState" << endl;
    mLockState.append(0, now);
  }
  
  // update the 6hr series only every 5 seconds, and the 12hr series
  // every 6 seconds
  if (mOsclist.satisfied(mIFOname + ":Mode_Cleaner_locked")) {
    if (now.getS() % 5 == 0)
      mBothLock6hr.append(1, now);
    
    if (now.getS() % 6 == 0)
      mBothLock12hr.append(1, now);
  } else {
    if (now.getS() % 5 == 0)
      mBothLock6hr.append(0, now);
    
    if (now.getS() % 6 == 0)
      mBothLock12hr.append(0, now);
  }
} // END: BothArmsNotLockedActions()


//
// Main loop
//
void LightMon::ProcessData(void)
{
    // check stride alignment -- copied from PSLmon
    // this is attempt at fixing problem with zeros at start of trend frames
    Time tEnd = getDacc().getCurrentTime();
    int tSec = tEnd.getS();
    int iStride = int(getDacc().getStride());
    int version=1; // New convention: Online segments are version 1 jgz: 3/10/09

    if (mDebug > 2) {
        cerr << mName << ": tEnd = " << tEnd << endl;
        cerr << mName << ": tSec = " << tSec << endl;
        cerr << mName << ": iStride = " << iStride << endl;
    }

    if ((tSec % iStride) != 0 || tEnd.getN() != 0) {
        tSec += iStride - (tSec % iStride);
        *mLog << mName << ": Skipping to " << tSec << " GPS." << endl;
        getDacc().seek(Time(tSec));
        return;
    }

    bool printed_p = false;

    if (mDebug > 2) {
        cerr << mName << ": entered ProcessData()" << endl;
        mBothLock.printInfo();
        mBothLock6hr.printInfo();
    }
    
    ++mFrameCount;
    mTotalRunTime += getDacc().getStride();

    Time now = getDacc().getFillTime();

    //
    // Things to do on the first pass through ProcessData()
    //
    if (mFrameCount == 1) {

      mFirstNow = now;

      char Timestamp2[64];
      std::string TstampFormat2("%Y-%m-%d_%H:%N:%S_%Z");
      TimeStr(now, Timestamp2, TstampFormat2.c_str());
      mLogfileName   = mHtmlDir + "LightMon_Log_" + Timestamp2 + ".txt";
      mDispfileName  = mHtmlDir + "index.html";

      if (mDebug > 2) {
        cerr << mName << ": Log file -- " << mLogfileName << endl;
        cerr << mName << ": Display file -- " << mDispfileName << endl;
      }

      // Open log with append mode
      mLogfile.open(mLogfileName.c_str(), ios::out);

      if (!mLogfile) {
        cerr << mName << ": could not open logfile " << mLogfileName  << endl;
        cerr << "          reverting to stdout" << endl;
        mLog = &std::cout;
      } else {
        mLog = &mLogfile;
      }
      
      *mLog << "Starting LightMon 1st pass at " << now << " with log file = " << mLogfileName << endl;
      initTSWindowsMod(now);

      for (int i=0; i<mN_prelockloss; i++) {
	mLastFlushState_prelockloss[i] = -1;
	mLastFlushStart_prelockloss[i] = now;
	mLastFlushStop_prelockloss[i] = now;
      }

      for (int i=0; i<mN_lightdip; i++) {
	mLastFlushState_lightdip[i] = -1;
	mLastFlushStart_lightdip[i] = now;
	mLastFlushStop_lightdip[i] = now;
      }

      mLastFlushState_outoflock = -1;
      mLastFlushStart_outoflock = now;
      mLastFlushStop_outoflock = now;
    }

    // Form timestamp
    TimeStr(now, mTimestamp, mTstampFormat.c_str());

    if (mDebug > 0 || now.getS() % mLiveTimeUpdateInterval == 0) {
        *mLog << mName << ": Interval " << mFrameCount 
              << ", GPS = " << now << " (" << mTimestamp << ")" << endl;
        printed_p = true;
    }

    if (mDebug > 0) {

        if (mOsclist.satisfied(mIFOname + ":Both_arms_locked"))
            *mLog << mName << ": Both_arms_locked" << endl;

        printed_p = true;
    }

    ofstream dispfile(mDispfileName.c_str(), ios::out);
    if (!dispfile) {
      cerr << mName << ": could not open dispfile "
	   << mDispfileName  << endl;
      cerr << "          reverting to stdout" << endl;
      mDisplay = &std::cout;
    } else {
      mDisplay = &dispfile;
    }
    (*mDisplay).precision(3);

    htmlTopOfPage(now);

    // Do actions that depend on instantanous lock status
    // (including setting OUT_OF_LOCK flag)

    if (mOsclist.satisfied(mIFOname + ":Both_arms_locked") == true) {
      
      BothArmsLockedActions(now);
      if (mOUT_OF_LOCK_already_true) {
	if (mFrameCount != 1) {
	  if (mDebug>-1) {
	    *mLog << "Sending OUT_OF_LOCK segment with start/stop = " << mOUT_OF_LOCK_start << "/" << now << endl;
	  }
	  trig::Segment s_lock("DMT-OUT_OF_LOCK",version,mOUT_OF_LOCK_start,now);
	  s_lock.setIfos(mIFOname.c_str());
	  s_lock.setActivity(1);
	  sendSeg(s_lock,now);
	  mLastFlushStart_outoflock = mOUT_OF_LOCK_start;
	  mLastFlushStop_outoflock = now;
	  mLastFlushState_outoflock = 1;
	  mLastFlushStart_outoflock_on = mOUT_OF_LOCK_start;
	  mLastFlushStop_outoflock_on = now;
	}
	mIN_LOCK_start = now;
	if (mDebug>-1) {
	  *mLog << "Starting new In_LOCK segment at now=" << now << endl;
	}
      } else {
	if ( mFrameCount == 1) {
	  mIN_LOCK_start = now;
	  if (mDebug>-1) {
	    *mLog << "-> Starting new IN_LOCK segment at now=" << now << endl;
	  }
	} else {
	  if ( now-mIN_LOCK_start >= (Interval)60) {
	    if (mDebug>0) {
	      *mLog << "Flushing In_LOCK segment with start/stop = " << mIN_LOCK_start << "/" << now << endl;
	    }
	    trig::Segment s_lock("DMT-OUT_OF_LOCK",version,mIN_LOCK_start,now);
	    s_lock.setIfos(mIFOname.c_str());
	    s_lock.setActivity(0);
	    sendSeg(s_lock,now);
	    if (mDebug>0) {
	      *mLog << "-> Starting new IN_LOCK segment at now=" << now << endl;
	    }
	    mLastFlushStart_outoflock = mIN_LOCK_start;
	    mLastFlushStop_outoflock = now;
	    mLastFlushState_outoflock = 0;
	    mIN_LOCK_start = now;
	  }
	}
      }
      mOUT_OF_LOCK_already_true = false;
      
    } else {
      
      BothArmsNotLockedActions(now);
      if (!mOUT_OF_LOCK_already_true) {
	if (mFrameCount != 1) {
	  if (mDebug>-1) {
	    *mLog << "Sending In_LOCK segment with start/stop = " << mIN_LOCK_start << "/" << now << endl;
	  }
	  trig::Segment s_lock("DMT-OUT_OF_LOCK",version,mIN_LOCK_start,now);
	  s_lock.setIfos(mIFOname.c_str());
	  s_lock.setActivity(0);
	  sendSeg(s_lock,now);
	  mLastFlushStart_outoflock = mIN_LOCK_start;
	  mLastFlushStop_outoflock = now;
	  mLastFlushState_outoflock = 0;
	}
	mOUT_OF_LOCK_start = now;
	mOUT_OF_LOCK_already_true = true;
	if (mDebug>-1) {
	  *mLog << "Starting new OUT_OF_LOCK segment at now=" << now << endl;
	}
      } else {
	if ( mFrameCount == 1 ) {
	  mOUT_OF_LOCK_start = now;
	} else {
	  if ( now-mOUT_OF_LOCK_start >= (Interval)60) {
	    if (mDebug>0) {
	      *mLog << "Flushing OUT_OF_LOCK segment with start/stop = " << mOUT_OF_LOCK_start << "/" << now << endl;
	    }
	    trig::Segment s_lock("DMT-OUT_OF_LOCK",version,mOUT_OF_LOCK_start,now);
	    s_lock.setIfos(mIFOname.c_str());
	    s_lock.setActivity(1);
	    sendSeg(s_lock,now);
	    mLastFlushStart_outoflock = mOUT_OF_LOCK_start;
	    mLastFlushStop_outoflock = now;
	    mLastFlushState_outoflock = 1;
	    mLastFlushStart_outoflock_on = mOUT_OF_LOCK_start;
	    mLastFlushStop_outoflock_on = now;
	    mOUT_OF_LOCK_start = now;
	  }
	  if (mDebug>0) {
	    *mLog << "-> Starting new OUT_OF_LOCK segment at now=" << now << endl;
	  }
	}
      }
    }
      
    mTrend.trendData(mTrendChanNames["Both"].c_str(), now,
		       mBothLock1min.getLatest());

    int ivalue = mLockState.getLatest();
    double fvalue = (double)ivalue;
    if (mDebug > 1) {
      *mLog << "Appending fvalue= " << fvalue << " to LockState minute trend" << endl; 
    }
    mTrend.trendData(mTrendChanNames["LockState"].c_str(), now, 
                     fvalue);

    if (mDebug > 2) {
        cerr << "Both-Lock state = " << mBothLock1min.getLatest() << endl;
    }

    mBacquiredP =
        mOsclist.satisfied(mIFOname + ":Both_arms_lock_acquired");
    mBlostP     =
        mOsclist.satisfied(mIFOname + ":Both_arms_lock_lost");

    // Actions when lock is acquired:

    if (mBacquiredP) {

        sendTrig("Both_arms_lock_acquired", mIFOname, now);
        *mLog << mName << ": Both arms lock ACQUIRED" << endl;
        printed_p = true;

	// Flush out old prelockloss segments

	for (int i=0; i<mN_prelockloss; i++) {
    
	  int nsec = mNsec_prelockloss[i];
	  char groupname[256];
	  sprintf(groupname,"DMT-PRE_LOCKLOSS_%d_SEC",nsec);
	  Time tEnd = now;
	  Time tBegin = mLastFlushStop_prelockloss[i];
	  trig::Segment s(groupname,version,tBegin,tEnd);
	  s.setIfos(mIFOname.c_str());
	  s.setActivity(1);
	  char comment[256];
	  sprintf(comment,"Interval of %d seconds preceding lock loss",nsec);
	  if (mDebug>-1) *mLog << "Acquired lock - flushing " << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	  s.setComment(comment);
	  sendSeg(s,now);
	  mLastFlushStart_prelockloss[i] = tBegin;
	  mLastFlushStop_prelockloss[i] = tEnd;
	  mLastFlushState_prelockloss[i] = 1;
	  mLastFlushStart_prelockloss_on[i] = tBegin;
	  mLastFlushStop_prelockloss_on[i] = tEnd;
	}

    } else if (mBlostP) {

        sendTrig("Both_arms_lock_lost", mIFOname, now);
        *mLog << mName << ": Both arms lock LOST" << endl;
        *mLog << mName << ": Both Lock segment length = "
              << (float)(mBothTimeInLock.GetSecs())/60.
              << " mins" << endl;
        printed_p = true;

	// Write the prelockloss segments
  
	for (int i=0; i<mN_prelockloss; i++) {
    
	  int nsec = mNsec_prelockloss[i];
	  if ( now - (Interval)nsec > mLastFlushStop_prelockloss[i] ) {
	    char groupname1[256];
	    sprintf(groupname1,"DMT-PRE_LOCKLOSS_%d_SEC",nsec);
	    Time tEnd1 = now - (Interval)nsec;
	    Time tBegin1 = mLastFlushStop_prelockloss[i];
	    trig::Segment s1(groupname1,version,tBegin1,tEnd1);
	    s1.setIfos(mIFOname.c_str());
	    s1.setActivity(0);
	    char comment1[256];
	    sprintf(comment1,"Interval of %d seconds preceding lock loss",nsec);
	    if (mDebug>-1) *mLog << "Flushing (fragment) " << comment1 << "(segment: " << tBegin1 << "-" << tEnd1 << ")" << endl;
	    s1.setComment(comment1);
	    sendSeg(s1,now);
	    mLastFlushStart_prelockloss[i] = tBegin1;
	    mLastFlushStop_prelockloss[i] = tEnd1;
	    mLastFlushState_prelockloss[i] = 0;
	  }
	  char groupname[256];
	  sprintf(groupname,"DMT-PRE_LOCKLOSS_%d_SEC",nsec);
	  Time tEnd = now;
	  Time tBegin = mLastFlushStop_prelockloss[i];
	  trig::Segment s(groupname,version,tBegin,tEnd);
	  s.setIfos(mIFOname.c_str());
	  s.setActivity(1);
	  char comment[256];
	  sprintf(comment,"Interval of %d seconds preceding lock loss",nsec);
	  if (mDebug>-1) *mLog << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	  s.setComment(comment);
	  sendSeg(s,now);
	  mLastFlushStart_prelockloss[i] = tBegin;
	  mLastFlushStop_prelockloss[i] = tEnd;
	  mLastFlushState_prelockloss[i] = 1;
	  mLastFlushStart_prelockloss_on[i] = tBegin;
	  mLastFlushStop_prelockloss_on[i] = tEnd;
	}

    } else if (mOsclist.satisfied(mIFOname + ":Both_arms_locked") == true) {

	// Flush the (non) prelockloss segments
  
	for (int i=0; i<mN_prelockloss; i++) {
    
	  int nsec = mNsec_prelockloss[i];
	  if ( now - mLastFlushStop_prelockloss[i] - (Interval)nsec >= (Interval)60 ) {
	    char groupname[256];
	    sprintf(groupname,"DMT-PRE_LOCKLOSS_%d_SEC",nsec);
	    Time tEnd = now - (Interval)nsec;
	    Time tBegin = mLastFlushStop_prelockloss[i];
	    trig::Segment s(groupname,version,tBegin,tEnd);
	    s.setIfos(mIFOname.c_str());
	    s.setActivity(0);
	    char comment[256];
	    sprintf(comment,"Interval of %d seconds preceding lock loss",nsec);
	    if (mDebug>-1) *mLog << "Flushing(off) " << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	    s.setComment(comment);
	    sendSeg(s,now);
	    mLastFlushStart_prelockloss[i] = tBegin;
	    mLastFlushStop_prelockloss[i] = tEnd;
	    mLastFlushState_prelockloss[i] = 0;
	  }
	}

    } else {

	// Flush the prelockloss segments
  
	for (int i=0; i<mN_prelockloss; i++) {
    
	  int nsec = mNsec_prelockloss[i];
	  if ( now - mLastFlushStop_prelockloss[i] >= (Interval)60 ) {
	    char groupname[256];
	    sprintf(groupname,"DMT-PRE_LOCKLOSS_%d_SEC",nsec);
	    Time tEnd = now;
	    Time tBegin = mLastFlushStop_prelockloss[i];
	    trig::Segment s(groupname,version,tBegin,tEnd);
	    s.setIfos(mIFOname.c_str());
	    s.setActivity(1);
	    char comment[256];
	    sprintf(comment,"Interval of %d seconds preceding lock loss",nsec);
	    if (mDebug>-1) *mLog << "Flushing(on) " << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	    s.setComment(comment);
	    sendSeg(s,now);
	    mLastFlushStart_prelockloss[i] = tBegin;
	    mLastFlushStop_prelockloss[i] = tEnd;
	    mLastFlushState_prelockloss[i] = 1;
	    mLastFlushStart_prelockloss_on[i] = tBegin;
	    mLastFlushStop_prelockloss_on[i] = tEnd;
	  }
	}


    }

    // Get mean value of minimum arm power over previous 10 seconds
    // (using Peter Shawhan's offline algorithm - exclude seconds marked already as dips)

    double meanval_last10 = mArmPowerNoDip.getMeanPos(10);
    if (mDebug>0) *mLog << "Mean arm power over preceding 10 seconds = " << meanval_last10 << endl;

    // Get times series for X and Y arm normalized powers

    std::string Xname = mIFOname + ":LSC-LA_PTRX_NORM";
    Xnow = getDacc().refData(Xname.c_str());
    
    std::string Yname = mIFOname + ":LSC-LA_PTRY_NORM";
    Ynow = getDacc().refData(Yname.c_str());
    
    int dip = 0;
    if (meanval_last10>0. && mOsclist.satisfied(mIFOname + ":Both_arms_locked")) {
      
      // Get lowest value in X, Y arms for this second
      
      double Xmin = Xnow->getMinimum();
      double Ymin = Ynow->getMinimum();

      double fractdrop = (meanval_last10-min(Xmin,Ymin))/meanval_last10;
      if (mDebug>0) *mLog << "Current Xmin,Ymin= " << Xmin << "," << Ymin << " --> fractdrop = " << fractdrop << endl;
      for (int i=0; i<mN_lightdip; i++) {
	if (fractdrop>=mFract_lightdip[i]) {
	  dip = 1;
	  char groupname[256];
	  int percent = (int)(mFract_lightdip[i] * 100 + 0.0001);
	  sprintf(groupname,"DMT-LIGHTDIP_%d_PERCENT",percent);
	  char comment[256];
	  sprintf(comment,"Dip in interferometer arm power by %d percent",percent);

	  // First flush out inactive segment
	  if ( now-mLastFlushStop_lightdip[i] >= (Interval)1) {
	    Time tBegin = mLastFlushStop_lightdip[i];
	    Time tEnd = now;
	    trig::Segment s(groupname,version,tBegin,tEnd);
	    s.setIfos(mIFOname.c_str());
	    s.setActivity(0);
	    s.setComment(comment);
	    sendSeg(s,now);
	    if (mDebug>0) {
	      *mLog << "Flushing " << groupname << " segment with start/stop = " << tBegin << "/" << tEnd << endl;
	    }
	  }

	  Time tBegin = now;
	  Time tEnd = tBegin + (Interval)1;
	  trig::Segment s(groupname,version,tBegin,tEnd);
	  s.setIfos(mIFOname.c_str());
	  s.setActivity(1);
	  s.setComment(comment);
	  sendSeg(s,now);
	  if (mDebug>0) *mLog << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	  mLastFlushStart_lightdip[i] = tBegin;
	  mLastFlushStop_lightdip[i] = tEnd;
	  mLastFlushState_lightdip[i] = 1;
	  mLastFlushStart_lightdip_on[i] = tBegin;
	  mLastFlushStop_lightdip_on[i] = tEnd;
	} else {
	  if ( now-mLastFlushStop_lightdip[i] >= (Interval)60) {
	    char groupname[256];
	    int percent = (int)(mFract_lightdip[i] * 100 + 0.0001);
	    sprintf(groupname,"DMT-LIGHTDIP_%d_PERCENT",percent);
	    Time tBegin = mLastFlushStop_lightdip[i];
	    Time tEnd = now;
	    trig::Segment s(groupname,version,tBegin,tEnd);
	    s.setIfos(mIFOname.c_str());
	    char comment[256];
	    sprintf(comment,"Dip in interferometer arm power by %d percent",percent);
	    if (mDebug>0) *mLog << comment << "(segment: " << tBegin << "-" << tEnd << ")" << endl;
	    sprintf(comment,"Dip in interferometer arm power by %d percent",percent);
	    s.setComment(comment);
	    s.setActivity(0);
	    sendSeg(s,now);
	    if (mDebug>0) {
	      *mLog << "Flushing " << groupname << " segment with start/stop = " << tBegin << "/" << tEnd << endl;
	    }
	    mLastFlushStart_lightdip[i] = tBegin;
	    mLastFlushStop_lightdip[i] = tEnd;
	    mLastFlushState_lightdip[i] = 0;
	  }

	}
      }
      
    }

    // Add current minimum mean arm power to queue

    double Xmean = Xnow->getAverage();
    double Ymean = Ynow->getAverage();
    double ArmPower = min(Xmean,Ymean);
    mArmPower.append(ArmPower,now);

    double ArmPowerSigned = ArmPower;
    if (dip) ArmPowerSigned = -ArmPower;
    if (!mOsclist.satisfied(mIFOname + ":Both_arms_locked")) ArmPowerSigned = -1.;
    mArmPowerNoDip.append(ArmPowerSigned,now);

    if (mFrameCount == 1) {
        *mLog << "Starting LightMon with following lock states:" << endl;
        
        if (mOsclist[mIFOname + ":Both_arms_locked"]->satisfied())
            *mLog << "    Both arms locked" << endl;
        else
            *mLog << "    Both arms not locked" << endl;

        printed_p = true;
    }

    if (mDebug > 2) {
        cerr << "mFrameCount  = " << mFrameCount << endl;
        cerr << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
             << endl;
        *mLog << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
              << endl;
    }
        
    if (mFrameCount == 1) {
        *mLog << mName << ": GPS = " << now << " (" << mTimestamp << ")" 
              << endl;
        printed_p = true;
    }

    //
    // Update data for DMT viewer
    //
    mBothLock.fillTSeries(mBts);
    mBothLock6hr.fillTSeries(mBts6hr);
    mBothLock12hr.fillTSeries(mBts12hr);
    mBts12hrExp = mBts12hr.decimate(6);
    mArmPower.fillTSeries(mArmPowerTS);
    mArmPowerNoDip.fillTSeries(mArmPowerNoDipTS);

    if (mDebug > 2) {
        cerr << mName << ": mBothLock.startTime() = " << mBothLock.startTime()
             << endl;
        cerr << mName << ": mBts.getStartTime()   = " << mBts.getStartTime()
             << endl;
        cerr << mName << ": mBts.getNSample()     = " << mBts.getNSample()
             << endl;

        cerr << mName << ": mBothLock6hr.startTime() = "
             << mBothLock6hr.startTime()
             << endl;
        cerr << mName << ": mBts6hr.getStartTime()   = "
             << mBts6hr.getStartTime()
             << endl;
        cerr << mName << ": mBts6hr.getNSample()     = " << mBts6hr.getNSample()
             << endl;
    }

    if (printed_p == true) 
        *mLog << endl;

    // update trend data
    mTrend.Update();

    htmlFoot();

    // save on I/O time. since the saveAllTSWindows is also done in
    // the destructor, this is only for "checkpointing"
    if (mFrameCount > 10 && (now.getS() % 600) == 0)
        saveAllTSWindowsMod();

    return;
} /* END: ProcessData() */


void LightMon::sendTrig(string trigname, string ifoname, Time now)
{
    trig::TrigRslt tout("Lock transition",trigname.c_str(),0);
    tout.setIfos(ifoname.c_str());
    tout.setTime(now);

    if (mDebug > 0) {
        *mLog << mName << ": Debug trigger of subtype " << trigname
              << " at GPS " << now << endl;
    }
    tout.setDisposition(trig::d_metaDB + trig::d_alarm);
    *mLog << mName << ": Sending trigger of subtype " << trigname
	  << " at GPS " << now << endl;
    
    if (mSendTriggers) {
      lmsg::error_type rc = sendTrigger(tout);
    
      if (rc)
	*mLog << mName << ": Error sending trigger: error code "
	      << rc << endl;
      else
	*mLog << mName << ": Trigger successfully sent" << endl;
    } else {
      *mLog << mName << ": NOT sending trigger because disabled" << endl;
    }
      
}  /* END: sendTrig() */


void LightMon::sendSeg(Segment s, Time now)
{

  std::string groupname = s.getGroup();

  if (mDebug > 0) {
    
      Time tBegin = s.getStartTime();
      Time tEnd = s.getEndTime();
      std::string ifo = s.getIfos();
      int activity = s.getActivity();
      *mLog << mName << ": Dumping segment info for " << groupname << "(activity = " << activity << " ), IFO=" << ifo << ", begin=" << tBegin << ", end=" << tEnd << " at GPS=" << now << endl;
      
  }
    
  lmsg::error_type rc = sendSegment(s);
  
  if (rc)
    *mLog << mName << ": Error sending segment: error code "
	  << rc << endl;
  else
    if ( mDebug>1) *mLog << mName << ": Segment successfully sent" << endl;
  
}  /* END: sendSeg() */


void LightMon::htmlHead(void) const
{
    *mDisplay << "<!doctype html public \"-//W3C//DTD HTML 4.0 Transitional//EN\""
              << endl;
    *mDisplay << "\"http://www.w3.org/TR/REC-html40/loose.dtd\">"
              << endl;

    *mDisplay << "<html><head>" << endl;
    *mDisplay << "<!-- kriles@umich.edu -->" << endl;
    *mDisplay << "<meta http-equiv=\"Refresh\" content=\"" << mHTMLupdateInterval
              << "\">" << endl;
    *mDisplay << "<style type=\"text/css\">" << endl;
    *mDisplay << "   <!-- body { color: black; background: white; " << endl;
    *mDisplay << "               font-family: Helvetica, Arial, sans-serif; " << endl
              << "               font-weight: normal; } " << endl
              << "        h1   { font-weight: bold; }" << endl
              << "        h2   { font-weight: bold; }" << endl
              << "        h3   { font-weight: bold; }" << endl
              << "        h4   { font-weight: bold; }" << endl
              << "        h5   { font-weight: bold; }" << endl
              << "    -->" << endl;
    *mDisplay << "</style>" << endl;
    *mDisplay << "<title>LightMon update on " << mIFOname;
    if (mIFOname == "H1") {
        *mDisplay << " (Hanford 4k)";
    } else if (mIFOname == "H2") {
        *mDisplay << " (Hanford 2k)";
    } else if (mIFOname == "L1") {
        *mDisplay << " (Livingston 4k)";
    } else {
        // should never reach here since mIFOname is checked before
        std::string eObj = "Unknown IFO: ";
        eObj += mIFOname;
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }
    *mDisplay << "</title>" << endl;

    *mDisplay << "</head>" << endl;
    
    *mDisplay << "<body><h1>LightMon update on " << mIFOname;
    if (mIFOname == "H1") {
        *mDisplay << " (Hanford 4k)";
    } else if (mIFOname == "H2") {
        *mDisplay << " (Hanford 2k)";
    } else if (mIFOname == "L1") {
        *mDisplay << " (Livingston 4k)";
    } else {
        // should never reach here since mIFOname is checked before
        std::string eObj = "Unknown IFO: ";
        eObj += mIFOname;
        cerr << eObj << endl;
        throw runtime_error(eObj);
    }
    *mDisplay << "</h1>" << endl;

}

void LightMon::htmlFoot(void) const
{
    *mDisplay << "<p>&nbsp;</p><hr>" << endl;
    *mDisplay << "Contact: <a href=\"mailto:kriles@umich.edu\">Keith Riles <tt>&lt;kriles@umich.edu&gt;</tt> 734-764-4652</a>"
              << endl;
    *mDisplay << "</body></html>" << endl;
}

void LightMon::htmlTopOfPage(Time now)
{
    htmlHead();

    (*mDisplay).setf(ios::fixed, ios::floatfield);
    (*mDisplay).precision(2);
    
    *mDisplay << "<h3>Last update: " << mTimestamp << "</h3>" << endl;
    *mDisplay << "<p>LightMon (" << mIFOname << ") has been running "
              << "for " << mTotalRunTime.GetSecs() << " seconds ("
              << (double)(mTotalRunTime.GetSecs())/3600. << " hrs) (" 
              << (double)(mTotalRunTime.GetSecs())/86400. << " days)</p>" << endl;
    *mDisplay << "Process began running at GPS " << mFirstNow << endl;

    *mDisplay << "<p><font size=\"-1\"><em>Display updates automatically every 60 seconds ";
    //    *mDisplay << "($Revision: 7126 $)";
    *mDisplay << "</em></font></p>" << endl;

    // Print out table of current/recent arm power status

    *mDisplay << "<table cellpadding=5>";
    *mDisplay << "<tr><td>Current arm power is <td>" << mArmPower.getLatest() << endl;
    *mDisplay << "<tr><td>Average arm power over last 10 strides is <td>" << mArmPower.getMean(10) << endl;
    *mDisplay << "<tr><td>Average arm power over last 10 strides (excluding dips) is <td>" << mArmPowerNoDip.getMeanPos(10)<< endl;
    *mDisplay << "</table><p>" << endl;

    // Print out header for table of recently written segments

    *mDisplay << "<b>Table of most recent DQ segments written (any, on):</b><p><table border=1 cellpadding=5>" << endl;
    *mDisplay << "<tr><td align=center>Flag name<td align=center>Setting<sup>1</sup><td align=center>Flag on or off <br>Start-Stop (now=" << now.getS() << ")" << " <td align=center>Flag on<br>Start-Stop (now=" << now.getS() << ")" << endl;

    // Print out recently written OUT_OF_LOCK segment

    *mDisplay << "<tr><td>OUT_OF_LOCK<td align=center>" << mLastFlushState_outoflock << "<td align=center>" << mLastFlushStart_outoflock.getS() << "-" << mLastFlushStop_outoflock.getS() << "<td align=center>" << mLastFlushStart_outoflock_on.getS() << "-" << mLastFlushStop_outoflock_on.getS();

    // Print out recently written PRE_LOCKLOSS segments

    char groupname[256];
    int nsec;
    for (int i=0; i<mN_prelockloss; i++) {
      nsec = mNsec_prelockloss[i];
      sprintf(groupname,"PRE_LOCKLOSS_%d_SEC",nsec);
      *mDisplay << "<tr><td>" << groupname << "<td align=center>" << mLastFlushState_prelockloss[i] << "<td align=center>" << mLastFlushStart_prelockloss[i].getS() << "-" << mLastFlushStop_prelockloss[i].getS() << "<td align=center>" << mLastFlushStart_prelockloss_on[i].getS() << "-" << mLastFlushStop_prelockloss_on[i].getS() << endl;
    }

    // Print out recently written LIGHTDIP segments

    for (int i=0; i<mN_lightdip; i++) {
      int percent = (int)(mFract_lightdip[i] * 100 + 0.0001);
      sprintf(groupname,"LIGHTDIP_%d_PERCENT",percent);
      *mDisplay << "<tr><td>" << groupname << "<td align=center>" << mLastFlushState_lightdip[i] << "<td align=center>" << mLastFlushStart_lightdip[i].getS() << "-" << mLastFlushStop_lightdip[i].getS() << "<td align=center>" << mLastFlushStart_lightdip_on[i].getS() << "-" << mLastFlushStop_lightdip_on[i].getS() << endl;
    }

    *mDisplay << "</table><p><sup>1</sup>Setting values: (-1 = no segment written yet; 0 = flag off; 1 = flag on)<p>";
      
    *mDisplay << "<p><div style=\"font-weight: bold;\">Note:</div> "
              << "<table border=\"1\" cellpadding=5 width=\"60%\"><tr><td><b>DMTviewer State Vector "
              << "value</b></td>"
              << "<td><b>Meaning</b></td></tr> "
              << "<tr><td align=center>-1</td><td>No data available</td></tr>"
              << "<tr><td align=center>0</td><td>Mode Cleaner not locked</td></tr>"
              << "<tr><td align=center>1</td><td>Mode Cleaner locked, but not BothArms</td></tr>"
              << "<tr><td align=center>2</td><td>Mode Cleaner locked, BothArms locked</td></tr>"
              << "<tr><td align=center>3</td><td>Mode Cleaner locked, BothArms locked, High power</tr>"
              << "<tr><td align=center>4</td><td>Science Mode</tr></table>" << endl;

    *mDisplay << "<p>&nbsp;</p>" << endl;
}

void LightMon::setTSWindowStuffMod(void)
{
    
    //
    // Initialize and set TSWindow parameters 
    //
    Interval dt(1, 0);

    // strings for savefile names
    std::string chartstr = mHtmlDir + mIFOname + "_StripChart";
    std::string timeinlockstr = mHtmlDir + mIFOname + "_TimeInLock";

    if (mDebug > 0) {
        cout << "TSWindow: chartstr = " << chartstr << std::endl;
    }

    mArmPower.setDebug(mDebug);
    mArmPower.setDt(dt);
    mArmPower.setName(mIFOname + ":Arm_Power");
    mArmPower.setSaveFileName(chartstr + "_Arm_Power");

    mArmPowerNoDip.setDebug(mDebug);
    mArmPowerNoDip.setDt(dt);
    mArmPowerNoDip.setName(mIFOname + ":Arm_Power_No_Dip");
    mArmPowerNoDip.setSaveFileName(chartstr + "_Arm_Power_No_Dip");

    mBothLock.setDebug(mDebug);
    mBothLock.setDt(dt);
    mBothLock.setName(mIFOname + ":Both_arms_locked_LightMon");
    mBothLock.setSaveFileName(chartstr + "_Both_2hr.dat");

    // 5 second interval
    dt.SetS(5);
    dt.SetN(0);
    
    mBothLock6hr.setDebug(0);
    mBothLock6hr.setDt(dt);
    mBothLock6hr.setName(mIFOname + ":Both_arms_locked_6hr_LightMon");
    mBothLock6hr.setSaveFileName(chartstr + "_Both_6hr.dat");

    // 6 second interval
    dt.SetS(6);
    dt.SetN(0);
    
    mBothLock12hr.setDebug(0);
    mBothLock12hr.setDt(dt);
    mBothLock12hr.setName(mIFOname + ":Both_arms_locked_12hr_LightMon");
    mBothLock12hr.setSaveFileName(chartstr + "_Both_12hr.dat");
}

void LightMon::initTSWindowsMod(const Time & now)
{

    if (mDebug > 0) 
        std::cout << "LightMon::initTSWindowsMod() Entered: now = " 
                  << now << std::endl;
    // have to put the starting time back for each TSWindow

    mBothLock.fillMaybe(now-Interval(7200,0));

    mBothLock1min.fillMaybe(now-Interval(60,0));

    mLockState.fillMaybe(now-Interval(60,0));

    mBothLock6hr.fillMaybe(now-Interval(6*60*60,0));

    mBothLock12hr.fillMaybe(now-Interval(12*60*60,0));

}


void LightMon::saveAllTSWindowsMod(void)
{
    // I wish I had used a hash_map here, so I could iterate

    // Don't need to save the 1min TSWindows because those contain
    // data that's trended and written to Frames, anyway

    mBothLock.save();
    mBothLock6hr.save();
    mBothLock12hr.save();
    
}


void LightMon::restoreAllTSWindowsMod(void)
{
    // I wish I had used a hash_map here, so I could iterate

    mBothLock.restore();
    mBothLock6hr.restore();
    mBothLock12hr.restore();
}
