/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "HistCompr.hh"
#include "ParseLine.hh"
#include "TSeries.hh"
#include "DVector.hh"
#include "TrigRslt.hh"
#include <stdlib.h>
#include <fstream>
#include <iomanip>

//---------------------------------------  Supported filter classes
#include "BaseLine.hh"
#include "Blackman.hh"
#include "Difference.hh"
#include "FilterDesign.hh"
#include "Hamming.hh"
#include "Hanning.hh"
#include "MultiPipe.hh"
#include "IIRFilter.hh"

#ifdef  SHOWHISTO
#include "TApplication.h"
#include "TCanvas.h"
#include "TH1.h"
#include "TROOT.h"

//======================================  Convert Histogram1 to a TH1
inline TH1*
makeRootHist(const Histogram1& h) {
    int nBin = h.GetNBins();
    Histogram1::xbin_t lo = h.GetBinLowEdge(0);
    Histogram1::xbin_t hi = h.GetBinLowEdge(nBin+1);
    TH1D* rh = new TH1D(h.GetTitle(), h.GetTitle(), nBin, lo, hi);
    Histogram1::histdata_t* bins = new Histogram1::histdata_t[nBin+2];
    h.GetBinContents(bins);
    rh->SetContent(bins);
    delete[] bins;
    rh->SetEntries(h.GetNEntries());
    return rh;
}
#endif

using namespace std;

//======================================  Identify the process
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/HistCompr/HistCompr.cc 6619 2012-03-12 23:58:12Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Histogram comparison monitor"
#include "ProcIdent.hh"

//======================================  Generate the main function
#ifdef  SHOWHISTO
EXEROOT(HistCompr)
#else
EXECDAT(HistCompr)
#endif

//======================================  Histx constructor
Histx::Histx(const string& name, const string& channel, 
	     int nBin, double lo, double hi, double lim) 
  : mName(name),  mChannel(channel), mLimit(lim),
    mHisto(name.c_str(), nBin, lo, hi), 
    mRefer(name.c_str(), nBin, lo, hi),
    mStride(name.c_str(), nBin, lo, hi),
    mShort(name.c_str(), nBin, lo, hi),
    mLongAcc(name.c_str(), nBin, lo, hi),
    mLong(name.c_str(), nBin, lo, hi),
    mBaseLine(false), mFilterName("-")
{
}

//======================================  Histx destructor
Histx::~Histx(void) {
}

//======================================  Get the channel name
const char* 
Histx::getChannel(void) const {
    return mChannel.c_str();
}

//======================================  Get the histogram name
const char* 
Histx::getName(void) const {
    return mName.c_str();
}

//======================================  Get the histogram condition
const char* 
Histx::getCondition(void) const {
    return mCondition.c_str();
}

//======================================  Get a zero parameter chi-square
double 
Histx::getChiSquare(void) const {
    double ChiSq = 0;

    double hEntries = double(getNEntries());
    double rEntries = double(mRefer.GetNEntries());
    if (hEntries == 0 || rEntries == 0) return 0.0;

    int nBins = mShort.GetNBins();
    for (int i=0 ; i<nBins ; i++) {
        double hFrac = mShort.GetBinContent(i)/hEntries;
        double eps   = mRefer.GetBinContent(i)/rEntries;
	double sig2A = eps*(1.0-eps)/hEntries;
	if (eps == 0.0 && hFrac == 0.0) continue;
	else if (eps == 0.0) sig2A = 1.0/(rEntries*hEntries);
	double delta = (hFrac-eps)*(hFrac-eps)/sig2A;
	ChiSq += delta;
	// cout << "Bin/hFrac/eps/sig2A/delta=" << i << " " << hFrac 
	//      << " " << eps << " " << sig2A << " " << delta << endl;
    }
    return ChiSq;
}

//======================================  Get the number of entries
int 
Histx::getNEntries(void) const {
    return mHisto.GetNEntries();
}

//======================================  Clear all histograms
void 
Histx::clear(void) {
    clearStride();
    mHisto.Clear();
    mShort.Clear();
    mLongAcc.Clear();
    mLong.Clear();
}

//======================================  Clear the current histogram.
void 
Histx::clearStride(void) {
    mStride.Clear();
}

//======================================  Bump a single bin
void 
Histx::fillData(double x, double wt) {
    mHisto.Fill(x, wt);
}

//======================================  Bump a set of bins
void 
Histx::fillData(const TSeries& ts, double wt) {
    if (!mStride.GetTime()) mStride.SetTime(ts.getStartTime());
    int nPoint=ts.getNSample();
    if (!nPoint) return;

    if (!mPipe.null() || mBaseLine || 
	ts.refDVect()->getType() != DVector::t_double) {
        TSeries temp(ts);
	temp.Convert(DVector::t_double);
	if (!mPipe.null()) {
	    Time cTime = mPipe->getCurrentTime();
	    if (temp.getStartTime() != cTime && !(!cTime)) mPipe.reset();
	    temp  = mPipe(temp);
	}
	if (mBaseLine) temp -= temp.getAverage();
	mStride.FillN(nPoint, reinterpret_cast<double*>(temp.refData()));
    } else {
        mStride.FillN(nPoint, reinterpret_cast<const double*>(ts.refData()));
    }
}

//======================================  Get the reference histogram.
bool 
Histx::fetchReference(const string& fileName) {
    string title;
    Time::ulong_t sec, nSec;
    int nBin;
    Histogram1::xbin_t low, high;
    char x;

    fstream in(fileName.c_str(), ios::in);
    if (!in.good()) return false;
    in >> title;
    in >> sec; in.get(x); in >> nSec;
    in >> nBin >> low >> high;
    if (nBin != mRefer.GetNBins() ||
	low  != mRefer.GetBinLowEdge(0) ||
	high != mRefer.GetBinLowEdge(nBin+1)) return false;
    Histogram1::histdata_t* bins = new Histogram1::histdata_t[nBin+2];
    int nEntries = 0;
    for (int i=0 ; i<nBin+2 && in.good() ; i++) {
        in >> bins[i];
	nEntries += int(bins[i]);
    }
    if (!in.good()) return false;
    mRefer.SetTitle(title.c_str());
    mRefer.SetTime(Time(sec, nSec));
    mRefer.SetBinContents(bins);
    mRefer.SetNEntries(nEntries);
    delete[] bins;
    cout << mChannel << " reference set from: " << fileName 
	 << " GPS time: " << sec << " title: " << title << endl;
    cout << "N bins: " << nBin << " Low: " << low << " High: " << high 
	 << " N Entries: " << mRefer.GetNEntries() << " Average: " 
	 << mRefer.GetMean() << " Sigma: " << mRefer.GetSdev() << endl;
    return true;
}

//======================================  Increment all histos from stride
void
Histx::increment(void) {
    if (!mHisto.GetTime())   mHisto.SetTime(mStride.GetTime());
    mHisto += mStride;
    if (!mLongAcc.GetTime()) mLongAcc.SetTime(mStride.GetTime());
    mLongAcc += mStride;
}

//======================================  Print histogram statistics.
void
Histx::printHdr(ostream& out) {
    out.width(20);
    out.setf(ios::left);
    out << "Histo Name";
    out.width(20);
    out.setf(ios::left);
    out << "Channel Name";
    out.width(15);
    out.setf(ios::left);
    out << "Filter";
    out.width(20);
    out.setf(ios::left);
    out << "Condition";
    out.width(10);
    out.setf(ios::right);
    out << "# Bins";
    out.width(10);
    out << "Low Edge";
    out.width(10);
    out << "Hi Edge";
    out.width(10);
    out << "# Entries";
    out.width(10);
    out << "Mean";
    out.width(10);
    out << "Sigma";
    out.width(10);
    out << "Underflow";
    out.width(10);
    out << "Overflow";
    out.width(0);
    out << endl;
}

void
Histx::printStats(ostream& out) const {
    out.setf(ios::left);
    out << setw(20) << mName.c_str();
    out.setf(ios::left);
    out << setw(20) << mChannel.c_str();
    out.setf(ios::left);
    out << setw(15) << mFilterName.c_str();
    out.setf(ios::left);
    out << setw(20) << mCondition.c_str();
    int nBins = mHisto.GetNBins();
    out.setf(ios::right);
    out << setw(10) << nBins;
    out.setf(ios::right);
    out << setw(10) << mHisto.GetBinLowEdge(1);
    out.setf(ios::right);
    out << setw(10) << mHisto.GetBinLowEdge(nBins+1);
    out.setf(ios::right);
    out << setw(10) << mShort.GetNEntries();
    out.setf(ios::right);
    out << setw(10) << mShort.GetMean();
    out.setf(ios::right);
    out << setw(10) << mShort.GetSdev();
    out.setf(ios::right);
    out << setw(10) << mShort.GetBinContent(0);
    out.setf(ios::right);
    out << setw(10) << mShort.GetBinContent(nBins+1);
    out << endl;
}

//======================================  Save a normalized histogram.
static const int npline = 8;

bool 
Histx::saveHistogram(const string& fileName) const {
    fstream out(fileName.c_str(), ios::out);
    if (!out.good()) return false;

    string title = mHisto.GetTitle();
    int nBin = mHisto.GetNBins();

    out << title << endl;
    out << mHisto.GetTime() << endl;
    out << nBin << endl;
    out << mHisto.GetBinLowEdge(0) << endl;
    out << mHisto.GetBinLowEdge(nBin+1) << endl;
    int mBin = nBin+2;
    for (int i=0 ; i < mBin ; i+=npline) {
        int bMax = i+npline;
	if (mBin < bMax) bMax = mBin;
	for (int j=i ; j<bMax ; j++) out << mHisto.GetBinContent(j) << " ";
	out << endl;
    }
    return true;
}

//======================================  Set BaseLine subtraction mode
void
Histx::setBaseLineSubtraction(bool tf) {
    mBaseLine = tf;
}

//======================================  Set histogram condition 
void
Histx::setCondition(const string& cond) {
    mCondition = cond;
}

//======================================  Set the filter
void
Histx::setFilter(const auto_pipe& p) {
    mPipe = p;
}

//======================================  Set the filter name
void
Histx::setFilterName(const string& name) {
    mFilterName = name;
}

//======================================  Update the short period histo
void
Histx::updateShort(void) {
    mShort = mHisto;
    mHisto.Clear();
}

//======================================  Update the short period histo
void
Histx::updateLong(void) {
    mLong = mLongAcc;
    mLongAcc.Clear();
}

//======================================  Histo comparison monitor Constructor
HistCompr::HistCompr(int argc, const char* argv[]) 
  : DatEnv(argc, argv), mAvgTime(60.0), mLongTime(600.0), mTimeOut(0),
    mLongTimeOut(0), mOSC(getDacc()), mConfig("HistCompr.conf"), mTrig(false)
#ifdef  SHOWHISTO
  , mCanvas(0), mPlot(false)
#endif
{
    //----------------------------------  Parse the command line.
    for (int i=1 ; i<argc ; i++) {
        string argi(argv[i]);
	if (argi == "-conf") {
	    mConfig = argv[++i];
	} else if (argi == "-refdir") {
	    mRefDir = argv[++i];
	} else if (argi == "-savdir") {
	    mSaveDir = argv[++i];
	} else if (argi == "-time" || argi == "-short") {
	    mAvgTime = strtod(argv[++i], 0);
	} else if (argi == "-long") {
	    mLongTime = strtod(argv[++i], 0);
	} else if (argi == "-osc") {
	    mOSC.readConfig(argv[++i]);
	} else if (argi == "+trig") {
	    mTrig = true;
#ifdef  SHOWHISTO
	} else if (argi == "-plot") {
	    mPlot = true;
#endif
	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    cerr << "Syntax: " << endl;
	    cerr << "  HistCompr [-refdir <ref-dir>] [-savdir <save-dir>]" 
		 << " [-time <avg-time>] [-osc <osc-config>] [-conf <file>]";
#ifdef SHOWHISTO
	    cerr << " [-plot]";
#endif
	    cerr << endl;
	    finish();
	    return;
	}
    }


    //----------------------------------  Read the configuration file.
    getDacc().setIgnoreMissingChannel(true);
    readConfig();
    printStats();

#ifdef SHOWHISTO
    if (mPlot) mCanvas = new TCanvas("cHistCompr", "Histogram Comparator");
#endif
}

//======================================  Save reference hiastos and go away
HistCompr::~HistCompr(void) {
#ifdef SHOWHISTO
    if (mCanvas) delete mCanvas;
#endif
    if (!mSaveDir.empty()) save(mSaveDir);
    mList.clear();
}

//======================================  Fill the histogram and compare
void 
HistCompr::readConfig(void) {
    mServer = "HistCompr";

    //----------------------------------  Get histo definitions from conf.
    ParseLine pl(mConfig.c_str());

    string logfile;
    if (getenv("DMTHTMLOUT")) {
        logfile = getenv("DMTHTMLOUT");
	logfile += "/";
    }
    logfile += "HistCompr-conf.txt";
    ofstream logf(logfile.c_str(), ios::out);
    pl.setLog(logf);

    //----------------------------------  Loop over definitions.
    while (pl.getLine() >= 0) {
	int nArg = pl.getCount();
	if (!nArg) continue;
        string cmd = pl[0];
        if (cmd == "ServerName") {
	    mServer = pl[1];
	} else if (cmd == "Filter") {
	    if (nArg < 3) {
	        cerr << "HistCompr - Invalid Filter specifier." << endl;
	        logf << "*** Invalid Filter specifier. Syntax: " << endl;
		logf << "    Filter <name> <type> <parameters>" << endl;
		continue;
	    }
	    string fName(pl[1]);
	    string fType(pl[2]);
	    auto_pipe fPipe;
	    if (fType == "BaseLine") {
	        double tConst(1.0);
		if (nArg > 3) tConst = pl.getDouble(3);
	        fPipe.set(new BaseLine(tConst));
	    } else if (fType == "Blackman") {
	        fPipe.set(new Blackman);
	    } else if (fType == "Design") {
	        if (nArg < 5) {
		    cerr << "HistCompr - Invalid FilterDesign parameters" 
			 << endl;
		    logf << "*** Filter design parameters are: "
			 << "<sample-rate> <string>" << endl;
		    continue;
		}
		FilterDesign fd(pl[4], pl.getDouble(3));
	        fPipe.set(fd.release());
	    } else if (fType == "Difference") {
	        if (nArg < 4) {
		    cerr << "HistCompr - Difference filter requires a frequency!" << endl;
		    logf << "*** Difference filter requires a frequency!" 
			 << endl;
		    continue;
		}
	        fPipe.set(new Difference(pl.getDouble(3)));
	    } else if (fType == "FIRFilter") {
	        cerr << "HistCompr - FIR Filters not implemented" << endl;
	        logf << "FIR Filters not implemented" << endl;
	    } else if (fType == "Hamming") {
	        fPipe.set(new Hamming);
	    } else if (fType == "Hanning") {
	        fPipe.set(new Hanning);
	    } else if (fType == "IIRFilter") {
	        if (nArg < 6) {
		    cerr << "HistCompr - Specify rate, poles and zeros" << endl;
		    logf << "Specify rate, poles and zeros" << endl;
		    continue;
		}
		double rate = strtod(pl[3], 0);
		if (Debug()) cout << "Filter: " << fName << " rate " 
				  << rate << endl;
	        vector<dComplex> poles, zeros;
		const char* ptr = pl[4];
		double twopi=2*3.1415926535;
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    poles.push_back(dComplex(re,im)/twopi);
		    if (Debug()) cout << "pole " << poles.size() << ": "
				      << re << " + " << im << "i" << endl;
		}
		ptr = pl[5];
		while (*ptr) {
		    double re = strtod(ptr, const_cast<char**>(&ptr));
		    double im = strtod(ptr, const_cast<char**>(&ptr));
		    zeros.push_back(dComplex(re,im)/twopi);
                    if (Debug()) cout << "zero " << zeros.size() << ": "
                                      << re << " + " << im << "i" << endl;
		}
		fPipe.set(new IIRFilter(poles.size(), &poles[0], 
					zeros.size(), &zeros[0], rate));
	    } else if (fType == "MultiPipe") {
	        fPipe.set(new MultiPipe);
	        for (int i=3 ; i<nArg ; i++) {
		    if (mFiltr.find(pl[i]) != mFiltr.end()) {
		        cerr << "HistCompr - Can't build filter " << fName 
			     << endl;
		        logf << "Filter " << pl[i] 
			     << " could not be added to " << fName << endl;
		        continue;
		    }
		    dynamic_cast<MultiPipe&>(*fPipe).addPipe(*(mFiltr[pl[i]]));
		}
	    } else if (fType == "LineFilter") {
	        cerr << "HistCompr - Line Filters not implemented" << endl;
	        logf << "Line Filters not implemented" << endl;
	    } else {
	        cerr << "HistCompr - Unknown filter type: " << fType << endl;
	        logf << "*** Unrecognized filter type: " << fType << endl;
	    }
	    if (!fPipe.null()) mFiltr[fName] = fPipe;

	} else if (cmd == "Histogram") {
	    if (Debug()) cout << "Adding channel: " << pl[1] << endl;
	    getDacc().addChannel(pl[2]);
	    int nb  = pl.getInt(3);
	    if (nb <= 0) {
	        cerr << "HistCompr - Number of histogram bins <= 0." << endl;
	        logf << "Number of bins must be greater than 0." << endl;
	        continue;
	    }
	    double xlo = pl.getDouble(4);
	    double xhi = pl.getDouble(5);
	    if (xhi <= xlo) {
	        cerr << "HistCompr - xHi <= xLo." << endl;
	        logf << "xHi must be greater than xlo." << endl;
	        continue;
	    }
	    double tThresh = pl.getDouble(6);
	    Histx hist(pl[1], pl[2], nb, xlo, xhi, tThresh);
	    for (int i=7 ; i < nArg ; i++) {
	        string argi(pl[i]);
		if (argi == "-base") {
		    hist.setBaseLineSubtraction(true);
		} else if (argi == "-filter") {
		    FiltrIterator f = mFiltr.find(pl[++i]);
		    if (f == mFiltr.end()) {
		        cerr << "HistCompr - Undefined filter: " << pl[i]
			     << endl;
		        logf << "Filter: " << pl[i] << " has not been defined"
			     << endl;
		    } else {
		        hist.setFilter(f->second);
			hist.setFilterName(pl[i]);
		    }
		} else if (argi == "-while") {
		    hist.setCondition(pl[++i]);
		} else {
		    cerr << "Unrecognized configuration argument: " << argi
			 << endl;
		    logf << "Unrecognized configuration argument: " << argi
			 << endl;
		}
	    }
	    mList.push_back(hist);
	}
    }

    //----------------------------------  Fetch reference histograms
    bool ref = !mRefDir.empty();
    if (ref) {
        cout << "Reading reference histograms from directory " 
	     << mRefDir << endl;
        fetch(mRefDir);
    } else {
        cout << "No reference histograms specified" << endl;
    }

    //----------------------------------  Serve data
    setServerName(mServer.c_str());
    for (hist_iter it=mList.begin() ; it != mList.end() ; it++) { 
        string name = it->getName();
	name += "_Histo";
	serveData(name.c_str(), &it->getHistogram());
	name  = string(it->getName()) + "_Short";
	serveData(name.c_str(), &it->getShort());
	name  = string(it->getName()) + "_Stride";
	serveData(name.c_str(), &it->getStride());
	if (mLongTime != Interval(0.0)) {
	    name  = string(it->getName()) + "_Long";
	    serveData(name.c_str(), &it->getLong());
	}
	if (ref) {
	    name = it->getName();
	    name += "_Ref";
	    serveData(name.c_str(), &it->getReference());
	}
    }
}

//======================================  Fill the histogram and compare
void 
HistCompr::ProcessData(void) {

    //----------------------------------  Fill in the histograms
    for (hist_iter it=mList.begin() ; it != mList.end() ; it++) {
        it->clearStride();
        const char* cond = it->getCondition();
	if (cond && *cond && !mOSC.satisfied(cond)) continue;
        const TSeries* ts = getDacc().refData(it->getChannel());
	if (!ts) continue;
	const_cast<TSeries*>(ts)->Convert(DVector::t_double);
	it->fillData(*ts);
	if (!mTimeOut)     mTimeOut     = ts->getStartTime() + mAvgTime;
	if (!mLongTimeOut) mLongTimeOut = ts->getStartTime() + mLongTime;
	it->increment();
    }

    //----------------------------------  Check against the references
    Time now = getDacc().getCurrentTime();
    if (Debug() > 2) cout << "Time remaining: " << mTimeOut-now << endl;
    if (now >= mTimeOut) {
        for (hist_iter it=mList.begin() ; it!=mList.end() ; it++) {
	    it->updateShort();
	}
        if (!mRefDir.empty()) {
	    for (const_hist_iter it=mList.begin() ; it!=mList.end() ; it++) {
	        double ChiSq = it->getChiSquare();
		if (Debug()) {
		    cout << "Histogram: " << it->getChannel() 
			 << " nEntries: " << it->getNEntries() 
			 << " Chisquare: " << ChiSq << endl;
		}
		if (ChiSq > it->getLimit()) {

		    //-----------------  Plot if enabled
#ifdef SHOWHISTO
		    if (mPlot) {
		        mCanvas->cd();
			TH1* h = makeRootHist(it->getShort());
			h->DrawCopy("");
			TH1* r = makeRootHist(it->getReference());
			r->Scale(h->GetEntries()/r->GetEntries());
			r->DrawCopy("SAME,C");
			delete h;
			delete r;
			mCanvas->Update();
		    }
#endif
		    //-----------------  Generate a trigger
		    if (mTrig) {
		        trig::TrigRslt t("HistCompr", it->getName());
			t.setTime(getDacc().getCurrentTime()-mAvgTime);
			t.setDuration(mAvgTime);
			t.setSignificance(ChiSq);
			sendTrigger(t);
		    }
		}
	    }
	}
	printStats();
	mTimeOut = now + mAvgTime;
    }

    //-----------------------------------  Update the Long histos
    if (mLongTime != Interval(0.0) && now >= mLongTimeOut) {
        for (hist_iter it=mList.begin() ; it!=mList.end() ; it++) {
	    it->updateLong();
	}
	mLongTimeOut = now + mLongTime;
    }
}

//======================================  Save the histo, clear it, and reset 
void 
HistCompr::Attention(void) {
    MonServer::Attention();
    // if (!mSaveDir.empty()) save(mSaveDir);
    // clear();
    // if (!mRefDir.empty()) fetch(mRefDir);
}

//======================================  Clear the Histo
void 
HistCompr::clear(void) {
    for (hist_iter it=mList.begin() ; it != mList.end() ; it++) it->clear();
}

//======================================  Get all reference histograms
void 
HistCompr::fetch(const string& dir) {
    if (dir.empty()) return;
    for (hist_iter it=mList.begin() ; it != mList.end() ; it++) { 
        string file = dir + "/" + it->getName();
	bool ok = it->fetchReference(file);
	if (!ok) cout << "Error occurred when reading reference hist from "
		      << file << endl;
    }
}


//======================================  Print statistics
void 
HistCompr::printStats(void) const {
    string file;
    const char* htmldir = getenv("DMTHTMLOUT");
    if (htmldir) file = string(htmldir) + "/";
    file += mServer;
    file += "-stats.txt";
    ofstream out(file.c_str(), ios::out);

    //----------------------------------  Print the status page
    char date[128];
    TimeStr(Now(), date, "%Y-%m-%d %H:%N:%S");
    out << "                          HistCompr status at " << date << endl;
    out << endl;
    out << "Process name:           " << mServer   << endl;
    out << "Configuration file:     " << mConfig   << endl;
    out << "Accumulation time:      " << mAvgTime  << endl;
    out << "Long accumulation time: " << mLongTime << endl;
    out << "Triggers enabled:       " << mTrig     << endl;
    out << "Reference directory:    " << mRefDir   << endl;
    out << "Save Directory:         " << mSaveDir  << endl;
    out << endl;
    out << endl;
    Histx::printHdr(out);
    for (const_hist_iter it=mList.begin() ; it != mList.end() ; it++) { 
        it->printStats(out);
    }
}

//======================================  Save all histograms.
void 
HistCompr::save(const string& dir) const {
    if (dir.empty()) return;
    for (const_hist_iter it=mList.begin() ; it != mList.end() ; it++) { 
        string file = dir + "/" + it->getName();
	it->saveHistogram(file);
    }
}
