/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "xsil/stream.hh"
#include "xsil/Xwriter.hh"
#include "xsil/base64.hh"
#include "PConfig.h"
#include "test_endian.hh"
#include <string>
#include <stdexcept>
#include <fstream>
#include <iomanip>
#include <cmath>
#include <cstring>

using namespace std;
using namespace xsil;

//======================================  Base 64 line length.
static const int  b64llen=64;
static const test_endian endian;

//======================================  Unsigned character vector
UCVec::UCVec(void) 
    : own(false), pNBytes(0), pData(0)
{}

//======================================  Unsigned character vector
UCVec::UCVec(int N) 
    : own(true)
{
    pNBytes  = new int;
    *pNBytes = N;
    pData    = new unsigned char[N];
}

UCVec::UCVec(const UCVec& uc) {
    own = true;
    pNBytes = new int;
    *pNBytes = *uc.pNBytes;
    pData = new unsigned char[*pNBytes];
    memcpy(pData, uc.pData, *pNBytes);
}

//======================================  Unsigned character vector
UCVec::~UCVec(void) {
    if (own) {
	delete pNBytes;
	pNBytes = 0;
	delete[] pData;
	pData   = 0;
	own = false;
    }
}

//======================================  Constructors
Stream::Stream(void)
  : mLineSz(0), mMaxSz(80), mDelimit(" "), mEndLine(false), mInput(0)
{
}

Stream::Stream(const char* Name, const char* Type, const char* Data)
  : xobj(Name, Type), mInput(0)
{
    mLineSz  = 0;
    mMaxSz  = 80;
    mDelimit = " ";
    setData(Data);
    mEndLine = false;
}

Stream::Stream(const Stream& x) 
  : mInput(0)
{
    *this = x;
}

//======================================  Destructor
Stream::~Stream() {
    if (mInput) delete mInput;
    mInput = 0;
}

//======================================  overloaded operator
Stream&
Stream::operator=(const Stream& x) {
    setName(x.getName());
    setType(x.getType());
    mLineSz  = x.mLineSz;
    mMaxSz   = x.mMaxSz;
    mDelimit = x.mDelimit;
    mData    = x.mData;
    mRemote  = x.mRemote;
    mEncoding = x.mEncoding;
    mEndLine = x.mEndLine;      
    if (mInput) delete mInput;
    mInput = 0;
    return *this;
}

//======================================  Write out an XSIL document
void 
Stream::Spew(Xwriter& xout) const {
    const int   nAttr = 5;
    const char* Attr[] = {"Name",     "Type",    "Delimiter", 
			  "Content", "Encoding"};
    const char* Value[] = {getName(), getType(), getDelimit(), 
			   0,        getEncode()};

    if (string(getType()) == "Local") {
        xout.Tag(getObjType(), nAttr, Attr, Value);
	xout.endLine();
	bool saveindent = xout.getIndentEnable();
	if (!mEncoding.empty()) xout.setIndentEnable(false);
	xout.text(mData);
	xout.setIndentEnable(saveindent);
	xout.endLine();
	xout.endTag(getObjType());
    } else if (string(getType()) == "Remote") {
	Value[1] = getType();
        xout.Tag(getObjType(), nAttr, Attr, Value);
	xout.endLine();
	xout.text(mRemote);
	xout.endTag(getObjType());
    } else {
        cerr << "Undefined stream type " << getType() 
	     << ". Stream not printed" << endl;
    }
}

//======================================  Clone the stream.
Stream*
Stream::Clone(void) const {
    return new Stream(*this);
}

//======================================  Get the object type.
const char*
Stream::getObjType(void) const {
    return "Stream";
}

//======================================  Specify a delimiter
void 
Stream::delimit(char Delim) {
    if (Delim) mDelimit = Delim;
    else       mDelimit = "";
}

//======================================  Add a float
void 
Stream::Add(float x) {
    if (std::isnan(x)) x = 0;
    ostringstream ostr;
    ostr << x;
    append(ostr.str());
}

//======================================  Add an integer
void 
Stream::Add(int x) {
    ostringstream ostr;
    ostr << x;
    append(ostr.str());
}

//======================================  Add a double
void 
Stream::Add(double x) {
    if (std::isnan(x)) x = 0;
    ostringstream ostr;
    ostr << x;
    append(ostr.str());
}

//======================================  Convert to a string
//
//  Local function to convert a number i to a radix r string, prefix the
//  result with zeros to force a minimum width of w and append the result 
//  to string s.
//
static void 
puti(string& s, int i, int r=10, int w=0) {
    const char* hex="0123456789abcdef";
    if (i>=r || w>1) puti(s, i/r, r, w-1);
    s += hex[i%r];
}

//======================================  Add a double
void 
Stream::Add(const UCVec& x) {
    const unsigned char* pCh = x.pData;
    string bytestr;
    if (!pCh || !x.pNBytes) {
        bytestr = "*";
    } else if (!*x.pNBytes) {
        // bytestr = "0";
        bytestr = "";
    } else {
	//for (int i=0 ; i < *x.pNBytes ; i++) {
	//    if (bytestr.empty()) bytestr  = "\\";
	//    else                 bytestr += " \\";
	//    puti(bytestr, *pCh++, 8, 3);
	//}
	int b64len = b64Nout(*x.pNBytes);
	b64data_t* b64vec = new b64data_t[b64len];
	int datalen = toBase64(pCh, *x.pNBytes, b64vec, b64len);
	bytestr = string(b64vec, datalen);
	delete[] b64vec;
    }
    append(string("\"") + bytestr + "\"" );
}

//======================================  Add a char string
static void
replaceall(string& s, const char* from, const char* to) {
    string::size_type lto   = strlen(to);
    string::size_type lfrom = strlen(from);
    for (string::size_type i=0; (i=s.find(from, i)) != s.npos; i += lto) {
	s.replace(i, lfrom, to);
    }
}

void 
Stream::Add(const string& x, bool escsp) {
    string escaped(x);
    replaceall(escaped, "\\", "\\\\");    // Backslash and ampersand first
    replaceall(escaped,  "&", "&amp;");   // to avoid escaping escapes
    replaceall(escaped,  ",", "\\,");
    if (escsp) replaceall(escaped,  " ", "\\ ");
    // replaceall(escaped, "\"", "\\\""); // Quotes no longer escaped 3/8/01
    replaceall(escaped, "\"", "\\\""); // Quotes escaped again 7/11/09
    replaceall(escaped,  "<", "&lt;");
    replaceall(escaped,  ">", "&gt;");
    replaceall(escaped,  "\n", "\\n");
    append(string("\"") + escaped + '\"');
}

//======================================  Add a string.
void 
Stream::append(const string& x) {
    setType("Local");

    if (!mData.empty()) {
        mData += mDelimit;
	mLineSz++;
    }

    int len = x.length();
    if (mLineSz != 0 && mLineSz + len >= mMaxSz) lineBreak();

    if (mEndLine) {
        mData   += "\n";
	mLineSz  = 0;
	mEndLine = false;
    }

    mData   += x;
    mLineSz += len;
}

//======================================  Clear the stream
void 
Stream::Clear(void) {
    mData.erase();
    mLineSz = 0;
}

//======================================  Copy float array into stream
void 
Stream::Fill(int N, const float* x, int perLine) {
    for (int i=0 ; i<N ; i+=perLine) {
	if (!(i%1000) && i) estDone(double(i)/double(N));
        ostringstream ostr;
	ostr << setprecision(8);
	int jmax = i+perLine < N? perLine: N-i;
	for (int j=0; j<jmax; j++) {
	    if (j) ostr << mDelimit;
	    float xx = x[i+j];
	    if (isnan(xx)) xx = 0;
	    long  lx = long(xx);
	    if (float(lx) == xx) ostr << lx;
	    else                 ostr << xx;
	}
	append(ostr.str());
	lineBreak();
    }
}

//======================================  Copy double array into stream
void 
Stream::Fill(int N, const double* x, int perLine) {
    for (int i=0 ; i<N ; i+=perLine) {
	if (!(i%1000) && i) estDone(double(i)/double(N));
        ostringstream ostr;
	ostr << setprecision(16);
	int jmax = i+perLine < N? perLine: N-i;
	for (int j=0; j<jmax; j++) {
	    if (j) ostr << mDelimit;
	    double xx = x[i+j];
	    if (std::isnan(xx)) xx = 0;
	    long   lx = long(xx);
	    if (double(lx) == xx) ostr << lx;
	    else                  ostr << xx;
	}
	append(ostr.str());
	lineBreak();
    }
}

//======================================  Copy int array into stream
void 
Stream::Fill(int N, const int* x, int perLine) {
    for (int i=0 ; i<N ; i++) {
        Add(x[i]);
	if (perLine && !((i+1)%perLine)) lineBreak();
    }
}

//======================================  Copy string array into stream
void 
Stream::Fill(int N, const string x[], int perLine) {
    for (int i=0 ; i<N ; i++) {
        Add(x[i]);
	if (perLine && (i+1)%perLine == 0) lineBreak();
    }
}

//======================================  Copy string array into stream
void 
Stream::encode64Data(const void* data, int N) {
    if (N > 0) {
        if (endian.big_end()) setEncode("BigEndian,base64");
	else                  setEncode("LittleEndian,base64");
	delimit(0);
        int Nout = b64Nout(N);
	b64data_t* b64vec = new b64data_t[Nout];
	int Nenc = toBase64((const unsigned char*)data, N, b64vec, Nout);
	for (int inx=0 ; inx<Nenc ; inx+=b64llen) {
	    int llen = Nenc-inx;
	    if (llen > b64llen) llen=b64llen;
	    append(string(b64vec+inx, llen));
	    if (inx+llen < Nenc) lineBreak();
	}
	delete[] b64vec;
    }
}

//======================================  Copy string array into stream
bool
Stream::decode64Data(void* data, int dlen, int N) {
    if (N <= 0) return false;
    char* cData(reinterpret_cast<char*>(data));
    int nByt = N*dlen;
    string b64str;
    for (int iByt=0; iByt<nByt;) {
        if (b64str.empty()) {
	    (*mInput) >> b64str;
	} else {
	    string temp;
	    (*mInput) >> temp;
	    b64str += temp;
	}
	if (mInput->fail()) return true;

	int nb64  = b64str.size();
	int nLeft = nByt - iByt;
	if (nLeft >= 3) nb64 &= 4;
	else if (nb64 < (nLeft*4 + 2)/3) continue;
	int nRead = fromBase64(reinterpret_cast<b64data_t*>(&b64str[0]), nb64, 
			       cData, nLeft);
	iByt += nRead;
	nb64 = (nRead/3)*4;
	b64str.erase(0, nb64);
    }

    //-----------------------------------  Return if no swapping needed
    if (dlen<2) return false;
    if (endian.big_end()) {
        if (mEncoding.find("BigEndian")    != string::npos) return false;
    } else {
        if (mEncoding.find("LittleEndian") != string::npos) return false;
    }

    //-----------------------------------  Swap on a word-by-word basis
    int nBSwap = dlen/2;
    for (int i=0; i<nByt; i+=dlen) {
        for (int j=0; j<nBSwap; j++) {
	    char t=cData[i+j];
	    cData[i+j] = cData[i+dlen-1-j];
	    cData[i+dlen-1-j] = t;
	}
    }
    return false;
}

//======================================  Start a new line.
void 
Stream::lineBreak(void) {
    mEndLine = true;
}

//======================================  Open the input stream;
void
Stream::open(void) {
    if (mInput) return;
    if (mRemote.empty()) mInput = new istringstream(mData);
    else                 mInput = new ifstream(mRemote.c_str());
}

//======================================  Read N double floats
bool
Stream::read(double data[], int N) {
    open();
    if (mEncoding.empty()) {
        for (int i=0; i<N; i++) *mInput >> data[i];
    } else if (mEncoding.find("base64") != string::npos) {
        return decode64Data(data, sizeof(double), N);
    } else {
        throw runtime_error("Unrecognized stream encoding");
    }
    if (mInput->fail()) return true;
    return false;
}

//======================================  Read N floats
bool
Stream::read(float data[], int N) {
    open();
    if (mEncoding.empty()) {
        for (int i=0; i<N; i++) *mInput >> data[i];
    } else if (mEncoding.find("base64") != string::npos) {
        return decode64Data(data, sizeof(float), N);
    } else {
        throw runtime_error("Unrecognized stream encoding");
    }
    if (mInput->fail()) return true;
    return false;
}

//======================================  Read N ints
bool
Stream::read(int data[], int N) {
    open();
    if (mEncoding.empty()) {
        for (int i=0; i<N; i++) *mInput >> data[i];
    } else if (mEncoding.find("base64") != string::npos) {
        return decode64Data(data, sizeof(int), N);
    } else {
        throw runtime_error("Unrecognized stream encoding");
    }
    if (mInput->fail()) return true;
    return false;
}

//======================================  Read N strings
bool
Stream::read(string data[], int N) {
    open();
    for (int i=0; i<N; i++) {
        *mInput >> data[i];
	int len = data[i].size();
	if (len > 1 && data[i][0] == '"' && data[i][len-1] == '"') {
	    data[i].erase(len-1);
	    data[i].erase(0, 1);
	}
	if (mInput->fail()) return true;
    }
    return false;
}

//======================================  Point stream at remote data
void 
Stream::setData(const char* Data) {
    if (Data) mData += Data;
}


//======================================  Specify the encoding technique
void 
Stream::setEncode(const char* Code) {
    if (Code) mEncoding = Code;
}

//======================================  Point stream at remote data
void 
Stream::setRemote(const char* File) {
    setType("Remote");
    mRemote = File;
}

//======================================  Set maximum line length
void 
Stream::setSize(int sz) {
    mMaxSz = sz;
}

//======================================  Set estimated stream length
void 
Stream::estLength(std::string::size_type sz) {
    std::string::size_type cap = mData.capacity();
    if (sz > cap) mData.reserve(sz);
}

//======================================  Set estimated stream length
void 
Stream::estDone(double frac) {
    if (frac <= 0.0 || frac >=1.0) return;
    estLength(std::string::size_type(mData.size()/frac));
}
