/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_EVENTTABLE_HH
#define XSIL_EVENTTABLE_HH

#include <time.h>
#include <string>
#include <list>
#include <iosfwd>

namespace xsil {

  class XSIL;
  class table;
  class Stream;


  /** Produce an XML document describing a trigger database table.
    * Each row starts on a new output line in the XML document.
    * @ingroup IO_xsil
    * @brief Trigger database table document
    * @author J. Zweizig
    * @version 1.0; last modified February 12, 2002
    */
  class EventTable {

    /** Column data structure
      */
    struct eventcolumn {

	/**  Construct the event column data structure and initialize the 
	  *  address pointer.
	  *  \brief Default constructor
	  */
	eventcolumn(void);

	/**  Destroy the column data descriptor. The data pointed by data_Addr 
	  *  belong to someone else.
	  *  \brief Destructor
	  */
	~eventcolumn(void);

	/// Data type name string
	std::string data_Type;

	/// Data address
	const void* data_Addr;
    };

      /// Column list data type.
      typedef std::list<eventcolumn> ColumnList;

      /// Column list constant iterator type.
      typedef ColumnList::const_iterator col_iter;

  public:

    /**  Event table default constructor
      *  @brief Default constructor
      */
    EventTable(void);

    /**  Event table data constructor
      *  @brief Data constructor
      *  @param Title Title of the table
      */
    EventTable(const char* Title);

    /**  Event table destructor
      *  @brief Destructor
      */
    ~EventTable(void);

    /**  Set up the table internally. Specify the name and table type
      *  attributes.
      *  @memo make a table.
      *  \param Name Table name attribute
      *  \param Type Table type attribute
      */
    void makeTable(const char* Name, const char* Type);

    /**  Specify a table column. The column name and type are included.
      *  an address field specifies a static address from which the data 
      *  will be copied. The following types are currently supported:
      *  <table>
      *  <tr><td>Type</td><td>c-pointer</td></tr>
      *  <tr><td>STRING</td><td>const char*</td></tr>
      *  <tr><td>INT4</td><td>const int*</td></tr>
      *  <tr><td>REAL4</td><td>const float*</td></tr>
      *  <tr><td>REAL8</td><td>const double*</td></tr>
      *  </table>
      *  A '*' is added to the stream to indicate an invalid data type,
      *  and "00" is added to indicate a null pointer specified.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Type Column data type.
      *  @param Addr Pointer to the data field.
      */
    void addColumn(const char* Name, const char* Type, const void* Addr=0);

    /**  Copy all specified data fields (one for each table column) into the 
      *  data stream.
      *  @memo Copy an event into the data stream.
      */
    void putEvent(void);

    /**  Write the XML document to the specified output stream. The table
      *  data stream is cleared after the data have been written.
      *  @memo Write the XML document.
      *  @param out Output stream to which the data will be written.
      */
    void putTable(std::ostream& out);

    /**  Delete the contents of the table data stream.
      *  @memo Reset the data stream.
      */
    void resetStream(void);

  private:
    std::string mTitle;
    ColumnList mCList;
    XSIL*      mDoc;
    table*     mTable;
  };

}  // namespace xsil

inline
xsil::EventTable::eventcolumn::eventcolumn(void) {}

inline
xsil::EventTable::eventcolumn::~eventcolumn(void) {}


#endif  //  XSIL_EVENTTABLE_HH

