#include "PConfig.h"
#include "xml/XsilStd.hh"
#include "DataDesc.hh"
#include <stdio.h>
#include <cstring>
#include <cstdlib>
#include <sstream>

namespace xml {
   using namespace std;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilParamReadHandler                                                 //
//                                                                      //
// Xsil handler for reading parameter values                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   class xsilParamReadHandler : public xsilHandler {
   protected:
      xsilStd::datainfo* 	fInfo;
   public:
      xsilParamReadHandler (xsilStd::datainfo* info) 
      : xsilHandler (true), fInfo (info) {
      
      }
      /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
      /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
      /// short parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
      /// int parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const int& p, int N = 1) {
         if (strcasecmp (name.c_str(), "Subtype") == 0) {
            fInfo->fSubtype = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "MeasurementNumber") == 0) {
            fInfo->fMeasurementNumber = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "Averages") == 0) {
            fInfo->fAverages = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "N") == 0) {
            fInfo->fN = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "M") == 0) {
            fInfo->fM = p;
            return true;
         }
         return false; }
      /// long parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const long long& p, int N = 1) {
         return false; }
      /// float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const float& p, int N = 1) {
         return false; }
      /// double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const double& p, int N = 1) {
         if (strcasecmp (name.c_str(), "dt") == 0) {
            fInfo->fDt = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "f0") == 0) {
            fInfo->fF0 = p;
            return true;
         }
         else if (strcasecmp (name.c_str(), "df") == 0) {
            fInfo->fDf = p;
            return true;
         }
         if (strcasecmp (name.c_str(), "BW") == 0) {
            fInfo->fBW = p;
            return true;
         }
         return false; }
      /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
      /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
      /// string parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::string& p) {
         string n;
         int i1 = -1;
         int i2 = -1;
         xsilStd::analyzeName (name, n, i1, i2);
         if ((strcasecmp (name.c_str(), "Channel") == 0) &&
            (fInfo->fType == xsilStd::kTimeSeries)) {
            if (fInfo->fAChn.empty()) {
               fInfo->fAChn.push_back (p);
            }
            else {
               fInfo->fAChn[0] = p;
            }
            return true;
         }
         else if ((strcasecmp (name.c_str(), "ChannelA") == 0) &&
                 ((fInfo->fType == xsilStd::kSpectrum) || 
                 (fInfo->fType == xsilStd::kTransferFunction))) {
            if (fInfo->fAChn.empty()) {
               fInfo->fAChn.push_back (p);
            }
            else {
               fInfo->fAChn[0] = p;
            }
            return true;
         }
         else if ((strcasecmp (n.c_str(), "ChannelB") == 0) && 
                 (i1 >= 0) &&
                 ((fInfo->fType == xsilStd::kSpectrum) || 
                 (fInfo->fType == xsilStd::kCoefficients) || 
                 (fInfo->fType == xsilStd::kTransferFunction))) {
            if ((int)fInfo->fBChn.size() <= i1) {
               fInfo->fBChn.resize(i1 + 1);
            }
            fInfo->fBChn[i1] = p;
            return true;
         }
         else if ((strcasecmp (n.c_str(), "ChannelA") == 0) && 
                 (i1 >= 0) &&
                 (fInfo->fType == xsilStd::kCoefficients)) {
            if ((int)fInfo->fAChn.size() <= i1) {
               fInfo->fAChn.resize(i1 + 1);
            }
            fInfo->fAChn[i1] = p;
            return true;
         }
         return true; }
      /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name,
                        const attrlist& attr,
                        unsigned long sec, unsigned long nsec) {
         if (strcasecmp (name.c_str(), "t0") == 0) {
            fInfo->fSec = sec;
            fInfo->fNSec = nsec;
            return true;
         }
         return false; }
   };



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilParamReadHandlerQuery                                            //
//                                                                      //
// Query object for reading parameter values                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   class xsilParamReadHandlerQuery : public xsilHandlerQuery {
   protected:
      xsilStd::datainfo*	fInfo;
   public:
      xsilParamReadHandlerQuery (xsilStd::datainfo& info) 
      : fInfo (&info) {
      }
      virtual xsilHandler* GetHandler (const attrlist& attr);
   };

//______________________________________________________________________________
   xsilHandler* xsilParamReadHandlerQuery::GetHandler (
                     const attrlist& attr) 
   {
      attrlist::const_iterator ti = attr.find (xmlType);
      if (ti == attr.end()) {
         return 0;
      }
      fInfo->fType = xsilStd::Typeid (ti->second.c_str());
      if ((int) fInfo->fType < 0) {
         return 0;
      }
      return new (nothrow) xsilParamReadHandler (fInfo); 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilStd                                                              //
//                                                                      //
// Standard XML objects utility routines                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   void xsilStd::datainfo::init ()
   {
      fType = kTimeSeries;
      fSubtype = 0;
      fAChn.clear(); 
      fBChn.clear(); 
      fN = 0;
      fM = 0;
      fSec = 0;
      fNSec = 0;
      fDt = 0;
      fF0 = 0;
      fDf = 0;
      fBW = 0;
      fAverages = 0;
      fMeasurementNumber = -1;
   }

//______________________________________________________________________________
   bool xsilStd::GetDataType (const char* graphtype, DataType& type,
                     int& subtype)
   {
      if (strcmp (graphtype, kPTTimeSeries) == 0) {
         type = kTimeSeries;
         subtype = 0;
      }
      else if (strcmp (graphtype, kPTFrequencySeries) == 0) {
         type = kSpectrum;
         subtype = 0;
      }
      else if (strcmp (graphtype, kPTPowerSpectrum) == 0) {
         type = kSpectrum;
         subtype = 1;
      }
      else if (strcmp (graphtype, kPTCoherence) == 0) {
         type = kSpectrum;
         subtype = 3;;
      }
      else if (strcmp (graphtype, kPTCrossCorrelation) == 0) {
         type = kSpectrum;
         subtype = 2;
      }
      else if (strcmp (graphtype, kPTTransferFunction) == 0) {
         type = kTransferFunction;
         subtype = 0;
      }
      else if (strcmp (graphtype, kPTCoherenceFunction) == 0) {
         type = kTransferFunction;
         subtype = 2;
      }
      else if (strcmp (graphtype, kPTTransferCoefficients) == 0) {
         type = kCoefficients;
         subtype = 0;
      }
      else if (strcmp (graphtype, kPTHarmonicCoefficients) == 0) {
         type = kCoefficients;
         subtype = 1;
      }
      else if (strcmp (graphtype, kPTCoherenceCoefficients) == 0) {
         type = kCoefficients;
         subtype = 3;
      }
      else if (strcmp (graphtype, kPTIntermodulationCoefficients) == 0) {
         type = kCoefficients;
         subtype = 2;
      }
      else if (strcmp (graphtype, 
                      (string (kPTTransferCoefficients) + 
                      " (matrix)").c_str()) == 0) {
         type = kCoefficients;
         subtype = 8;
      }
      else if (strcmp (graphtype, kPTHistogram1) == 0) {
         type = kHistogram;
         subtype = 0;
      }
      else {
         return false;
      }
      return true;
   }

//______________________________________________________________________________
   bool xsilStd::GetGraphType (DataType type,
                     int subtype, std::string& graphtype)
   {
      switch (type) {
         case kTimeSeries:
            if ((subtype >= 0) && (subtype < 8)) {
               graphtype = kPTTimeSeries;
               return true;
            }
            else {
               return false;
            }
         case kSpectrum:
            switch (subtype) {
               case 0:
               case 4:
                  graphtype = kPTFrequencySeries;
                  break;
               case 1:
               case 5:
                  graphtype = kPTPowerSpectrum;
                  break;
               case 2:
               case 6:
                  graphtype = kPTCrossCorrelation;
                  break;
               case 3:
               case 7:
                  graphtype = kPTCoherence;
                  break;
               default:
                  return false;
            }
            return true;
         case kTransferFunction:
            switch (subtype) {
               case 0:
               case 1:
               case 3:
               case 4:
                  graphtype = kPTTransferFunction;
                  break;
               case 2:
               case 5:
                  graphtype = kPTCoherenceFunction;
                  break;
               default:
                  return false;
            }
            return true;
         case kCoefficients:
            switch (subtype) {
               case 0:
               case 4:
                  graphtype = kPTTransferCoefficients;
                  break;
               case 1:
               case 5:
                  graphtype = kPTHarmonicCoefficients;
                  break;
               case 2:
               case 6:
                  graphtype = kPTIntermodulationCoefficients;
                  break;
               case 3:
               case 7:
                  graphtype = kPTCoherenceCoefficients;
                  break;
               case 8:
                  graphtype = 
                     string (kPTTransferCoefficients) + " (matrix)";
                  break;
               default:
                  return false;
            }
            return true;
         //==== Histogram (mito)
         case kHistogram:
            switch (subtype) {
               case 0:
               case 1:
               case 6:
               case 7:
                  graphtype = kPTHistogram1;
                  break;
               default:
                  return false;
            }
            return true;
         default:
            return false;
      }
   }

//______________________________________________________________________________
   std::string xsilStd::Typename (xsilStd::DataType type)
   {
      switch (type) {
         case kTimeSeries:
            return xmlObjTypeTimeSeries;
         case kSpectrum:
            return xmlObjTypeSpectrum;
         case kTransferFunction:
            return xmlObjTypeTransferFunction;
         case kCoefficients:
            return xmlObjTypeCoefficients;
         case kHistogram:
            return xmlObjTypeHistogram;
         default:
            return "";
      }
   }

//______________________________________________________________________________
   xsilStd::DataType xsilStd::Typeid (const char* type)
   {
      if (strcasecmp (type, "TimeSeries") == 0) {
         return kTimeSeries;
      }
      else if (strcasecmp (type, "Spectrum") == 0) {
         return kSpectrum;
      }
      else if (strcasecmp (type, "TransferFunction") == 0) {
         return kTransferFunction;
      }
      else if (strcasecmp (type, "Coefficients") == 0) {
         return kCoefficients;
      }
      else if (strcasecmp (type, "Histogram") == 0) {
         return kHistogram;
      } 
      else {
         return (DataType) -1;
      }
   }

//______________________________________________________________________________
   int xsilStd::DataSubtypeXY (DataType type, int subtype)
   {
      switch (type) {
         case kTimeSeries:
            return (subtype < 4) ? subtype + 4 : subtype;
         case kSpectrum:
            return (subtype < 4) ? subtype + 4 : subtype;
         case kTransferFunction:
            return (subtype < 3) ? subtype + 3 : subtype;
         case kCoefficients:
            return (subtype < 4) ? subtype + 4 : subtype;
         case kHistogram: // hist mito
            return (subtype == 0 || subtype == 6) ? subtype + 1 : subtype;
         default:
            return subtype;
      }
   }

//______________________________________________________________________________
   bool xsilStd::analyzeName (const string& name,
                     string& n, int& index1, int& index2)
   {
      index1 = -1;
      index2 = -1;
   
      // first index 
      string::size_type	pos = name.find ('[');
      if (pos == string::npos) {
         // no index in name
         n = name;
      }
      else {
         n.assign (name, 0, pos);
         index1 = atoi (name.c_str() + pos + 1);
         if (index1 < 0) {
            return false;
         }
         string::size_type	pos2 = name.find ('[', pos + 1);
         if (pos2 != string::npos) {
            index2 = atoi (name.c_str() + pos2 + 1);
            if (index2 < 0) {
               return false;
            } 
         }
      }
      // remove blanks, etc.
      while ((pos = n.find_first_of (" \t")) != string::npos) {
         n.erase (pos, 1);
      }
      return true;
   }

//______________________________________________________________________________
   string xsilStd::makeName (const string& Name, int index1, int index2)
   {
      string::size_type	pos =  Name.find ('[');
      string		n = Name;
      char		buf[256];
   
      if (pos != string::npos) {
         n.erase (pos, string::npos);
      }
      if (index1 >= 0) {
         if (index2 >= 0) {
            sprintf (buf, "[%i][%i]", index1, index2);
         }
         else {
            sprintf (buf, "[%i]", index1);
         }
      }
      else {
         buf[0] = 0;
      }
      return n + buf;
   }

//______________________________________________________________________________
   bool xsilStd::GetDataInfo (const char* xml, datainfo& info)
   {
      info.init();
      stringstream  ss;
      ss << xsilHeader() << endl;
      ss << xml;
      ss << xsilTrailer() << endl;
      xsilParamReadHandlerQuery prmq (info);
      xsilParser parse;
      parse.AddHandler (prmq);
      bool ret = parse.Parse (ss);
      parse.Done();
      return ret;
   }

//______________________________________________________________________________
   bool xsilStd::GetDataInfo (const BasicDataDescriptor* desc,
                     int& dim1, int& dim2, bool& complex,
                     bool& XYdata)
   {
      bool cpy;
      if ((dynamic_cast<const HistDataDescriptor*>(desc))) { //  hist mito
         return GetDataInfo (desc, dim1, dim2, complex, XYdata, (const double**)0, cpy);
      }
      else {
         return GetDataInfo (desc, dim1, dim2, complex, XYdata, (const float**)0, cpy);
      }
   }

//______________________________________________________________________________
   bool xsilStd::GetDataInfo (const BasicDataDescriptor* desc,
                     int& dim1, int& dim2, bool& cmplx, bool& XYdata, 
                     const float** dataptr, bool& cpy, bool compress, 
                     bool forceXY)
   {
      if (desc == 0) {
         return false;
      }
      dim1 = 1;
      dim2 = 0;
      cmplx = desc->IsComplex();
      XYdata = false;
      cpy = false;
   
      // deal with referenced copy if compress
      const DataRef* dr;
      if (compress && (dr = dynamic_cast<const DataRef*>(desc)) &&
         dr->GetRef()) {
         dim2 = dr->GetRef()->GetN();
         if (dr->GetRef()->IsXY()) {
            dim1 = dr->GetRef()->GetM() + 1;
            XYdata = true;
            if (dataptr) {
               *dataptr = dr->GetRef()->GetX();
            }
         }
         else {
            if (dr->GetRef()->GetM() > 1) {
               dim1 = dr->GetRef()->GetM();
            }
            if (dataptr) {
               *dataptr = dr->GetRef()->GetY();
            }
         }
      }
      
      // normal data descriptors
      else {
         dim2 = desc->GetN();
         if (!desc->IsXY() && !forceXY) {
            if (dataptr) {
               *dataptr = desc->GetY();
            }
         }
         else if (cmplx) {
            dim1 = 2;
            XYdata = true;
            if (dataptr) {
               float* datacopy = new (nothrow) float [4 * dim2];
               *dataptr = datacopy;
               cpy = true;
               if (datacopy != 0) {
                  for (int i = 0; i < dim2; i++) {
                     datacopy[2*i] = desc->GetX()[i];
                     datacopy[2*i+1] = 0;
                     datacopy[2*(i + dim2)] = desc->GetY()[2*i];
                     datacopy[2*(i + dim2)+1] = desc->GetY()[2*i+1];
                  }
               }
               else {
                  return false;
               }
            }
         }
         else {
            dim1 = 2;
            XYdata = true;
            if (dataptr) {
               float* datacopy = new (nothrow) float [2 * dim2];
               *dataptr = datacopy;
               cpy = true;
               if (datacopy != 0) {
                  for (int i = 0; i < dim2; i++) {
                     datacopy[i] = desc->GetX()[i];
                     datacopy[i + dim2] = desc->GetY()[i];
                  }
               }
               else {
                  return false;
               }
            }
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool xsilStd::GetDataInfo (const BasicDataDescriptor* desc,
                     int& dim1, int& dim2, bool& cmplx, bool& XYdata, 
                     const double** dataptr, bool& cpy, bool compress,
                     bool forceXY)
   {
      if (desc == 0) {
         return false;
      }
      dim1 = 1;
      dim2 = 0;
      cmplx = desc->IsComplex();
      XYdata = false;
      cpy = false;
   
      const HistDataDescriptor* hdesc;
   
     // 1-D histogram data descriptors
      if ((hdesc = dynamic_cast<const HistDataDescriptor*>(desc))) {
      
         XYdata = hdesc->IsXY() || forceXY;
      
       // check fixed or variable bin spacing 
         if ( XYdata ) dim1++;
       // check bin error array exists
         if ( hdesc->GetBinErrors() ) dim1++;
      
         dim2 = hdesc->GetN() + 2;
      
         if (dataptr) {
            double* datacopy = new (nothrow) double[dim1 * dim2];
            *dataptr = datacopy;
            cpy = true;
            if (datacopy != 0) {
            
               int indx = 0;
            // copy bin edges for variable bin spacing histogram
               if ( XYdata ) { 
               // copy bin edges
                  memcpy (datacopy, hdesc->GetXBinEdges(), dim2 - 1);
                  indx += dim2 - 1;
               }
            
            // copy bin contents
               memcpy (datacopy +  indx, hdesc->GetBinContents(), dim2);
               indx += dim2;
            
              	// copy bin errors
               if ( hdesc->GetBinErrors() ) {
                  memcpy (datacopy +  indx, hdesc->GetBinErrors(), dim2);
               }
            }
            else {
               return false;
            }
         }
      
      }
      return true;
   
   }

}
