/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef MEMSTREAM_HH
#define MEMSTREAM_HH

#include <iostream>
#include <streambuf>
 
/**  An STL stream buffer to read/write data to/from a memory buffer
  *  @brief Memory STL stream buffer
  *  @author John Zweizig
  *  @version 1.0; Last Modified Marc 3, 2008
  *  @ingroup IO_lsmp
  */
template<typename _CharT, typename _Traits = std::char_traits<_CharT> >
class basic_membuf : public std::basic_streambuf< _CharT, _Traits >
{
public:
    ///  Character data type
    typedef _CharT                                       char_type;
    ///  Trait data type
    typedef _Traits                                      traits_type;
    ///  Integer type
    typedef typename traits_type::int_type               int_type;
    ///  Position type
    typedef typename traits_type::pos_type               pos_type;
    ///  Offset type
    typedef typename traits_type::off_type               off_type;
    ///  Stream buffer type
    typedef std::basic_streambuf<char_type, traits_type> _streambuf_type;

public:
    /**  Construct a closed membuf stream.
      *  @brief Default constructor.
      *  @param __mode Input/output mode.
      */
    basic_membuf(std::ios_base::openmode __mode = std::ios_base::out);

    /**  Construct a membuf stream and open it on a specified buffer.
      *  @brief Open constructor.
      *  @param __mode Input/output mode.
      *  @param buf    Buffer pointer.
      *  @param len    Buffer data length.
      */
    basic_membuf(char_type* buf, int_type len,
		 std::ios_base::openmode __mode = std::ios_base::out);

    /**  Destroy the stream buffer.
      *  @brief Destructor.
      */
    virtual ~basic_membuf(void) {sync();}

protected:
    /**  Handle a buffer underflow.
      *  @brief Buffer underflow handler.
      */
    virtual int_type underflow(void);

    /**  Handle a buffer overflow.
      *  @brief Buffer overflow handler.
      */
    virtual int_type overflow(void) {return traits_type::eof();}

    /**  Handle put-back failure.
      *  @brief Put-back failure handler.
      */
    virtual int_type pbackfail(int_type c) {return traits_type::eof();}

    /**  Position read/write pointers to the specified offset.
      *  @brief Seek a specified offset.
      */
    virtual pos_type seekoff(off_type off, std::ios_base::seekdir way, 
			     std::ios_base::openmode mode);

    /**  Position read/write pointers to the specified position.
      *  @brief Seek a specified position.
      */
    virtual pos_type seekpos(pos_type sp, std::ios_base::openmode mode);
    
    /**  Set the buffer pointer and length.
      *  @brief Set the buffer pointer.
      */
    virtual basic_membuf<_CharT,_Traits>* setbuf(char_type* buf, int_type len);

    /**  The synch method is supposed to synchronize the memory buffers to
      *  an external file. Since there are no external files in this case, 
      *  the sync command is a no-operation.
      *  @brief STL streambuf::sync.
      */
    virtual int sync(void) {return 0;}

private:
    std::ios_base::openmode _M_mode;
    char_type* mBuf;
    int_type   mLen;
};

template<typename _CharT, typename _Traits>
inline
basic_membuf<_CharT, _Traits>::basic_membuf(std::ios_base::openmode __mode)
  : mBuf(0), mLen(0)
{
    _M_mode = __mode;
}

template<typename _CharT, typename _Traits>
inline
basic_membuf<_CharT, _Traits>::basic_membuf(char_type* buf, int_type len,
					    std::ios_base::openmode __mode)
{
    _M_mode = __mode;
    setbuf(buf, len);
}


template<typename _CharT, typename _Traits>
inline typename basic_membuf<_CharT, _Traits>::int_type 
basic_membuf<_CharT, _Traits>::underflow(void) {
    if (this->gptr() < this->egptr()) return traits_type::to_int_type(*this->gptr());
    return traits_type::eof();
}

template<typename _CharT, typename _Traits>
inline basic_membuf<_CharT, _Traits>*
basic_membuf<_CharT, _Traits>::setbuf(char_type* buf, int_type len) 
{
    mBuf = buf;
    mLen = len;
    if ((_M_mode & std::ios_base::in )!=0) this->setg(mBuf, mBuf, mBuf+mLen);
    if ((_M_mode & std::ios_base::out)!=0) this->setp(mBuf, mBuf + mLen);
    return this;
}

template<typename _CharT, typename _Traits>
inline typename basic_membuf<_CharT, _Traits>::pos_type 
basic_membuf<_CharT, _Traits>::seekoff(off_type off, 
				       std::ios_base::seekdir way,
				       std::ios_base::openmode mode) {
    off_type ret(-1);
    bool in_mode  = (std::ios_base::in  & _M_mode & mode) != 0;
    bool out_mode = (std::ios_base::out & _M_mode & mode) != 0;
    if (way == std::ios_base::beg) {
        ret = 0;
    } else if (way == std::ios_base::cur) {
        if (out_mode) ret = this->pptr() - mBuf;
        if ( in_mode) ret = this->gptr() - mBuf;
    } else if (way == std::ios_base::end) {
	ret = mLen;
    }
    if (ret != off_type(-1)) {
        ret += off;
	if (!(ret >= 0 && ret <= mLen)) ret = off_type(-1);
    }
    if (ret != off_type(-1)) {
        if ( in_mode) this->setg(mBuf, mBuf+ret, mBuf+mLen);
	if (out_mode) {
	   this->setp(mBuf, mBuf + mLen);
	   this->pbump(ret);
	}
    }
    return pos_type(ret);
}

template<typename _CharT, typename _Traits>
inline typename basic_membuf<_CharT, _Traits>::pos_type 
basic_membuf<_CharT, _Traits>::seekpos(pos_type sp, 
				       std::ios_base::openmode mode) {
    return seekoff(off_type(sp), std::ios_base::beg, mode);
}

/**  Input memory stream template using the stream buffer defined above.
  *  @brief Input memory stream template.
  *  @author John Zweizig
  *  @version 1.1; Last modified March 4, 2008
  *  @ingroup IO_lsmp
  */
template<typename _CharT, typename _Traits = std::char_traits<_CharT> >
class basic_imemstream : public std::basic_istream<_CharT, _Traits>
{
public:
    // Types:
    /// Character data type
    typedef _CharT                                     char_type;
    /// Trait data type
    typedef _Traits                                    traits_type;
    /// Integer data type
    typedef typename traits_type::int_type             int_type;
    /// Position type
    typedef typename traits_type::pos_type             pos_type;
    /// Offset type
    typedef typename traits_type::off_type             off_type;
    /// Stream buffer type
    typedef basic_membuf<_CharT, _Traits>              __membuf_type;
    /// Input stream type
    typedef std::basic_istream<char_type, traits_type> __istream_type;
 
private:
    __membuf_type  _M_membuf;
 
public:
    // Constructors:
    /**  Construct a closed memory dtream
     */
    explicit
    basic_imemstream(std::ios_base::openmode __mode = std::ios_base::in)
      : __istream_type(NULL)
      { this->init(&_M_membuf); }
 
    /**  Construct a memory input stream and assign a buffer to the stream.
      */
    basic_imemstream(const char_type* __buf, int_type __len,
			std::ios_base::openmode __mode = std::ios_base::in)
      : __istream_type(NULL), 
	_M_membuf(const_cast<char_type*>(__buf), __len, __mode)
      { this->init(&_M_membuf); }
 
    /**  Destroy a memory input stream.
      *  @brief Destructor.
      */
      ~basic_imemstream()
      { }
 
      // Members:
    /**  Get the memory stream buffer associated with this stream.
      *  @brief Get a streambuf pointer.
      *  @return Pointer to the stream buffer.
      */
      __membuf_type* rdbuf() const
      { return const_cast<__membuf_type*>(&_M_membuf); }
};


/**  Output memory stream template using the stream buffer defined above.
  *  @brief Output memory stream template.
  *  @author John Zweizig
  *  @version 1.1; Last modified March 4, 2008
  *  @ingroup IO_lsmp
  */
template <typename _CharT, typename _Traits = std::char_traits<_CharT> >
class basic_omemstream : public std::basic_ostream<_CharT, _Traits>
{
public:
    // Types:
    ///  Character type
    typedef _CharT                                      char_type;
    ///  Traits type
    typedef _Traits                                     traits_type;

    ///  Integer type
    typedef typename traits_type::int_type              int_type;

    /// Position type
    typedef typename traits_type::pos_type              pos_type;

    ///  Offset type
    typedef typename traits_type::off_type              off_type;

    ///  Memory stream buffer data type
    typedef basic_membuf<_CharT, _Traits>               __membuf_type;

    ///  Output memory stream data type.
    typedef std::basic_ostream<char_type, traits_type>  __ostream_type;
 
    private:
      __membuf_type  _M_membuf;
 
    public:
     // Constructors/destructor:
    /**  Construct an output memory stream with no associated buffer.
     */
      explicit
      basic_omemstream(std::ios_base::openmode __mode = std::ios_base::out)
      : __ostream_type(NULL), _M_membuf(__mode | std::ios_base::out)
      { this->init(&_M_membuf); }
 
    /**  Construct an output memory stream and associate a buffer with the 
      *  specified length.
      */
      basic_omemstream(char_type* __buf, int_type __len,
		       std::ios_base::openmode __mode = std::ios_base::out)
	: __ostream_type(NULL), _M_membuf(__buf, __len, __mode | std::ios_base::out)
      { this->init(&_M_membuf); }
 

    /**  Destroy an output memory stream.
      */
      ~basic_omemstream()
      { }
 
      // Members:
    /**  Get the memory stream buffer associated with this stream.
      *  @brief Get a streambuf pointer.
      *  @return Pointer to the stream buffer.
      */
    __membuf_type*
    rdbuf() const
      { return const_cast<__membuf_type*>(&_M_membuf); }
};

typedef basic_imemstream<char> imemstream;
typedef basic_omemstream<char> omemstream;

#endif
