/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRANSLATE_HH
#define TRANSLATE_HH

/**  Translate is a template class that maps the character set to a set
  *  of enumerated values.
  *  @memo Translate character code to another data value.
  *  @author   J. Zweizig
  *  @version  1.1; Last Modified August 5, 2004
  *  @ingroup IO_lxr
  */
template <class T> 
class Translate {
public:
  /**  Construct an empty translate table.
    */
  Translate(void);

  /**  Construct a translate table and initialize all characters to the
    *  specified value.
    */
  Translate(T init);

  /**  Destry the translate table.
   */
  ~Translate(void);

  /** Set the table to translate the specified characters to listed value.
    */
  void char_type(const char*, T val);

  /**  Get the traslation of the specified character.
    *  @brief Translate a character.
    *  @param c Character to be translated.
    *  @return Translated value.
    */
  T operator[](char c) const;

  /**  Get get a reference to the traslation table entry for the specified 
    *  character.
    *  @brief Translate a character.
    *  @param c Character to be translated.
    *  @return Reference to translation table entry.
    */
  T& operator[](char c);

  /**  Reset all entries of the table to the initial value specified in the 
    *  constructor.
    *  @brief Clear the translation table.
    */
  void reset(void);

  /**  Set the translation of all alphabetic characters (a-z and A-Z) to the 
    *  specified value.
    *  @brief Set alphabet translation.
    *  @param alfval New alphabet translation value.
    */
  void set_alpha(T alfval);

  /**  Set the translation of all numeric characters (0-9) to the 
    *  specified value.
    *  @brief Set number translation.
    *  @param numval New number translation value.
    */
  void set_numer(T numval);

  /**  Unset all translation table entries with the specified value. Then set
    *  the translation of all characters in the null-terminated string to the
    *  specified value.
    *  @brief Reset the list of characters with the specified value
    *  @param x Null-terminated string with characters to be translated 
    *  to \c val.
    *  @param val Translation value to be reset.
    */
  void set_type(const char* x, T val);

  /**  Reset all the table entries that translate to \c val to the
    *  initialization value.
    *  @brief Remove all entries that translate to \c val
    *  @param val Value to be unset.
    */
  void unset(T val);

private:
#ifndef __CINT__
  static const int table_count=256;
  static const int table_mask=table_count-1;
#else
  #define table_count 256
  #define table_mask  255
#endif
  T _initval;
  T _table[table_count];
};

template <class T> 
inline 
Translate<T>::Translate(void) 
  : _initval(T(0))
{
}

template <class T> 
inline 
Translate<T>::Translate(T init) 
  : _initval(init)
{
    reset();
}

template <class T> 
inline 
Translate<T>::~Translate(void) {
}

template <class T> 
inline T&
Translate<T>::operator[](char c) {
    return _table[int(c) & table_mask];
}

template <class T> 
inline T
Translate<T>::operator[](char c) const {
    return _table[int(c) & table_mask];
}

template <class T> 
inline void
Translate<T>::reset(void) {
    for (int i=0; i<table_count; ++i) _table[i] = _initval;
}

template <class T> 
inline void
Translate<T>::unset(T val) {
    for (int i=0; i<table_count; ++i) if (_table[i] == val) _table[i] = _initval;
}

template <class T> 
inline void
Translate<T>::char_type(const char* ccc, T val) {
    const char* p=ccc;
    while (*p) (*this)[*p++] = val;
}

template <class T> 
inline void
Translate<T>::set_type(const char* ccc, T val) {
    unset(val);
    char_type(ccc, val);
}

template <class T> 
inline void
Translate<T>::set_alpha(T val) {
    char_type("abcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ", val);
}

template <class T> 
inline void
Translate<T>::set_numer(T val) {
    char_type("0123456789", val);
}

#endif // !defined(TRANSLATE_HH)
