/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   LSMP is the Ligo Shared Memory Partition object class
//
#ifndef LSMP_PROD_HH
#define LSMP_PROD_HH

#include "lsmp.hh"

/**  LSMP_PROD provides an interface to the shared memory partition for 
  *  data producer processes. The shared memory partition is optionally
  *  created and attached to by the LSMP_PROD constructors. Buffers are 
  *  allocated for write access with get_buffer() and released for reading 
  *  by the consumers with release().
  *  @memo Shared Memory Producer Class.
  *  @author John G. Zweizig
  *  @version 1.3; Modified October 7, 1999
  *  @ingroup IO_lsmp
  */
class LSMP_PROD : public LSMP {
  //------------------------------------  Public methods
  public:
    /**  Constructs a null producer, not connected to any 
      *  partition.
      *  @memo Default constructor.
      */
    LSMP_PROD(void);

    /**  Initialize a producer structure and attach it to an existing 
      *  partition. If the named partition doesn't exist, it is created
      *  with default parameters.
      *  @memo Attach to a partition.
      *  @param part Name of partition to be attached.
      */
    LSMP_PROD(const char* part);

    /**  A partition is created and is accessed by the producer. If a 
      *  partition with the desired buffer count and length already exists 
      *  it is accessed. If an incompatible partition exists, it is not
      *  accessed.
      *  @memo Create a partition and attach a producer.
      *  @param part Name of partition to be created/attached.
      *  @param nbuf Minimum number of buffer in partition.
      *  @param lbuf Minimum buffer length in bytes of partition.
      */
    LSMP_PROD(const char* part, int nbuf, int lbuf);

    /**  If a buffer has been allocated, it is released. If a partition is 
      *  attached, it is released (see LSMP::release).
      *  @memo Producer destructor.
      */
    ~LSMP_PROD(void);

    /**  This operator returns true if the producer is not ready to supply
      *  data to the specified partition.
      *  @memo Test for error condition.
      *  @return true if an error condition exists in the producer.
      */
    bool operator! (void) const;

    /**  A buffer is allocate according to the current buffer mode and its
      *  address is returned. The buffer will normally be allocated from 
      *  the free list. If there are no free buffers available and if the
      *  buffer allocation mode is set to a scavenging mode (see LSMP::bufmode)
      *  the oldest unused full buffer is allocated. If no (free or full) 
      *  buffers can be found, get_buffer will block until a buffer is 
      *  available unless (flags & NOWAIT) != 0.
      *  @memo Allocate an empty buffer.
      *  @return Pointer to allocated buffer or NULL if no buffers available.
      *  @param flags function modifier flag.
      *
      */
    char* get_buffer(int flags=0);

    /**  The current buffer is released and marked as reserved by any 
      *  consumers waiting for a buffer. The trigger bit mask indicates 
      *  the type of data contained in the buffer. The data buffer will 
      *  be sent only to those consumers with a trigger mask containing
      *  one or more common bits with the buffer mask. If flags is set
      *  to SHOWALL, the specified buffer will be reserved by all current
      *  consumers, independent of the number of buffers currently reserved 
      *  by the consumer and the reservation limit. The SHOWALL mechanism 
      *  may be used to transmit control information such as end-of-file 
      *  status.
      *  @memo Release a full buffer.
      *  @param length The length of the buffer data in bytes.
      *  @param mask   A trigger identifier mask.
      *  @param flags  Special handling flags as described above.
      */
    void release(int length, int mask=-1, int flags=0);

    /**  The current buffer is returned to the free list without being
      *  distributed to any consumers.
      *  @memo Return an unused buffer.
      */
    void return_buffer(void);

    /**  SetID sets a record ID for the buffer currently owned by the 
      *  consumer. The record ID may be used by the consumer to access
      *  a specific record.
      *  @memo Set the record ID for the current buffer.
      *  @param ID Record identifier for the current buffer.
      */
    void SetID(int ID);

  private:
    /**  ibuf is the number of the buffer currently being written by the 
      *  producer. Ibuf is set to -1 after a release.
      *  @memo Current buffer identifier. 
      */
    int    ibuf;
};

//======================================  Inline functions
inline bool
LSMP_PROD::operator!(void) const {
    return !valid();
}

#endif  // LSMP_PROD_HH

