/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "gds_shmem.hh"
#include "gds_atomic.h"
#include <cstdio>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#if defined(P__LINUX) || defined(sun)  // JCB 
#include <error.h>
#endif  // JCB 
#include <errno.h>
#include "PConfig.h"
#if defined(P__LINUX)
#include <sys/resource.h>
#endif
#include <new>

// #define GDS_SHMEM_TEST 1
#ifdef  GDS_SHMEM_TEST
#include <iostream>
using namespace std;
#endif

#define SHMPROT 0666
#if defined(sun)
#define SHMAT_MMU_FLG SHM_SHARE_MMU
#else
#define SHMAT_MMU_FLG 0
#endif

//======================================  Inline private flag manipulation 
inline void 
gds_shmem::clr_flag(enum flagbits flg) {
    _flagwd &= ~(1 << int(flg));    
}

//======================================  Construct shm descriptor
inline void 
gds_shmem::set_flag(enum flagbits flg) {
    _flagwd |= (1 << int(flg));
}

//======================================  Construct shm descriptor
gds_shmem::gds_shmem(void) 
    : _flagwd(0), _ID(0), _addr(0), _size(0), _protfl(SHMPROT)
{
}
 
//======================================  Destroy shm descriptor
gds_shmem::gds_shmem::~gds_shmem(void) {
    release();
}

//======================================  Attach a shm
bool 
gds_shmem::attach(int id) {
    //----------------------------------  Find an existing partition.
    if (!find(id, 0, 0)) return false;

    //----------------------------------  Attach it.
    return map();
}

//======================================  Create a shm
bool 
gds_shmem::create(int id, size_t minsiz) {
    //----------------------------------  Find the ID
    if (!find(id, minsiz, IPC_CREAT + _protfl)) return false;

    //----------------------------------  Map it into the memory space
    return map();
}

//======================================  Deaccess the partition.
bool
gds_shmem::deaccess(bool rmv) {
    struct shmid_ds shm_status;
    if (!is_accessed()) return false;
    if (rmv) shmctl(_ID, IPC_RMID, &shm_status);
    clr_flag(kAccessed);
    return true;
}

//======================================  Find an shm
bool 
gds_shmem::exists(int id, size_t minsize) {
    bool rv = find(id, minsize, 0);
    if (rv) release(false);
    return rv;
}

//======================================  Find an shm
bool 
gds_shmem::find(int id, size_t minsiz, int flag) {
    _error = 0;

    //----------------------------------  Make sure descriptor is available
    if (is_accessed()) return false;

    //----------------------------------  Get an ID
    _ID = shmget(id, minsiz, flag);
    if (_ID < 0) {
	_error = errno;
	// perror("gds_shmem::find - shmget");
	return false;
    }
    set_flag(kAccessed);
    return true;
}

//======================================  Lock the partition into memory
bool
gds_shmem::lock(bool flag) {
    bool rv = false;
#ifdef GDS_SHMEM_TEST
    cout << "Enter gds_shmem::lock" << endl;
#endif
    if (is_attached()) {
	//------------------------------  Sun systems need to be root
#if defined(sun)
	int saveuid = geteuid();
	if (saveuid) seteuid(0);
	if (geteuid()) {
	    perror("gds_shmem::lock Unable to set effective uid to root");
	    return false;
	} 
#endif

	//------------------------------  Set the lock limit on linux machines
#if defined(P__LINUX)
	if (flag) {
	    struct rlimit rlockmem;
	    getrlimit(RLIMIT_MEMLOCK, &rlockmem);
#ifdef GDS_SHMEM_TEST
	    cout << "Lock memory limits, current: " << rlockmem.rlim_cur
		 << " max: " << rlockmem.rlim_max << endl;
#endif
	    //const unsigned long pagemask = 0x01FFF;
	    //rlockmem.rlim_cur = (_size + pagemask) & ~pagemask;
	    rlockmem.rlim_cur = rlockmem.rlim_max;
	    setrlimit(RLIMIT_MEMLOCK, &rlockmem);
	}
#endif

	//-------------------------------  Darwin and windows are clueless
#if defined(P__DARWIN) || defined(P__WIN32)
	_error = EPERM;

	//-------------------------------  Posix lock/unlock code.
#else
	struct shmid_ds shm_status;
	if (flag) {
	    rv = !(shmctl(_ID, SHM_LOCK, &shm_status) < 0);
	    if (rv) set_flag(kLocked);
	    else    _error = errno;
	}
	else {
	    rv = !(shmctl(_ID, SHM_UNLOCK, &shm_status) < 0);
	    if (rv) clr_flag(kLocked);
	    else    _error = errno;
	}
	if (!rv) perror("gds_shmem::lock Error in shmctl");
#endif

	//--------------------------------  Restore saved uid on solaris
#if defined(sun)
	if (saveuid) seteuid(saveuid);
#endif
    }
    return rv;
}

//======================================  Map the partition into the process
//                                        address space.
bool
gds_shmem::map(void) {
    struct shmid_ds ds;

    //----------------------------------  Check state
    if (is_attached() || !is_accessed()) return false;

    //----------------------------------  Map it
    _addr = shmat(_ID, 0, SHMAT_MMU_FLG);
    if (long(_addr) == -1) {
	_error = errno;
	perror("gds_shmem::map - shmgat");
	return false;
    }
    set_flag(kAttached);

    //----------------------------------  Get the size
    int rc = shmctl(_ID, IPC_STAT, &ds);
    if (rc < 0) return false;
    _size = ds.shm_segsz;
#ifdef P__LINUX
    _mypid = ds.shm_lpid;
#endif
    return true;
}

//======================================  Test ownership
bool
gds_shmem::owner(void) const {
    int own = owner_uid();
    if (own < 0) return false;
    int uid = getuid();
    if (!uid || own == uid) return true;
    uid = geteuid();
    if (!uid || own == uid) return true;
    return false;
}

//======================================  Get partition owner uid.
int
gds_shmem::owner_uid(void) const {
    struct shmid_ds shm_status;
    if (!is_accessed()) return 0;
    if (shmctl(_ID, IPC_STAT, &shm_status)) {
	perror("gds_shmem: Unable to get status");
	return -1;
    }
    return shm_status.shm_perm.uid;
}

//======================================  Release the partition.
void 
gds_shmem::release(bool remove) {
    //----------------------------------  Unlock locked segments
    if (is_locked()) {
	lock(false);
    }

    //----------------------------------  Unmap mapped segments
    if (is_attached()) {
	unmap();
    }

    //----------------------------------  Deaccess accessed segments
    if (is_accessed()) {
	deaccess(remove);
    }
}

//======================================  Release the partition.
void 
gds_shmem::set_mmd(void) {
    size_t* ptr = reinterpret_cast<size_t*>(_addr);
    *ptr++ = _size;
    *ptr++ = 2 * sizeof(size_t);
}

//======================================  Release the partition.
bool
gds_shmem::unmap(void) {
    if (!is_attached()) return false;
    if (shmdt(_addr) < 0) {
	perror("gds_shm::unmap - shmdt");
	return false;
    }
    clr_flag(kAttached);
    _addr = 0;
    return true;
}

void* 
operator new (size_t nb, gds_shmem& part) {
    size_t* av = reinterpret_cast<size_t*>(part.ref());
#ifdef USE_ATOMIC_BUILTINS
    size_t stoff = __sync_fetch_and_add(av+1, nb);
#else
    size_t stoff = (av[1] += nb) - nb;
#endif
    if (stoff + nb > av[0]) {
#ifdef USE_ATOMIC_BUILTINS
	__sync_fetch_and_sub(av+1, nb);
#else
	av[1] -= nb;
#endif
	throw std::bad_alloc();
    }
    return reinterpret_cast<void*>(size_t(av) + stoff);
}

void* 
operator new[] (size_t nb, gds_shmem& part) {
    size_t* av = reinterpret_cast<size_t*>(part.ref());
#ifdef USE_ATOMIC_BUILTINS
    size_t stoff = __sync_fetch_and_add(av+1, nb);
#else
    size_t stoff = (av[1] += nb) - nb;
#endif
    if (stoff + nb > av[0]) {
#ifdef USE_ATOMIC_BUILTINS
        __sync_fetch_and_sub(av+1, nb);
#else
        av[1] -= nb;
#endif
	throw std::bad_alloc();
    }
    return reinterpret_cast<void*>(size_t(av) + stoff);
}
