/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_TRANSOUTPUT_HH
#define LMSG_TRANSOUTPUT_HH

//=====================================  c++17 compatibility
#ifndef NOEXCEPT
#if __cplusplus > 201100
#define NOEXCEPT noexcept
#else
#define NOEXCEPT throw()
#endif
#endif

#include "lmsg/MsgTypes.hh"
#include <time.h>
#include <string>
#include <stdexcept>
#include "Grinder.hh"

class Time;

namespace lmsg {

    class Buffer;
    class MsgAddr;

    /**  The TransOutput class is used
      *  Ligo DMT output message translation class translates output messages
      *  into the export format in a specified buffer.
      *  @author John Zweizig
      *  @version 1.1; Last modified March 4, 2008
      *  @ingroup IO_lmsg
      */
    class TransOutput {
    public:
	/**  Construct an output translator for a specified output buffer.
	  *  @brief Constructor
	  *  @param b Output buffer reference.
	  */
	TransOutput(Buffer& b);

	/**  Construct a null output translator. A buffer must be specified
	  *  before it is used. Any use withh a null buffer pointer or a
	  *  full buffer will result in a \c std::runtime_error exception.
	  *  @brief Default constructor.
	  */
	TransOutput(void);

	/**  Increment the output pointer until it is aligned on a N-byte
	  *  boundary.
	  *  @brief Align output pointer.
	  *  @param N Alignment multiple.
	  */
	void align(size_type N);

	/**  Test whether the specified number of bytes can be fit into the 
	  *  buffer.
	  *  @brief Test available buffer space.
	  *  @param N Number of bytes required.
	  *  @return True if at least N bytes are available in buffer.
	  */
	bool available(size_type N) const;

	/**  Test whether buffer is full.
	  *  @brief Test full buffer.
	  *  @return True if no bytes are available in buffer.
	  */
	bool full(void) const;

	/**  Set the output pointer and data length for the specified buffer.
	  *  @brief Set buffer pointer.
	  *  @param b Reference to output buffer.
	  */
	void setBuffer(Buffer& b);

	/**  Template method to write a data array of arbitrary type to
	  *  the output buffer. Note that although this will generate code 
	  *  for an arbitrary data class, the word ordering will be wrong
	  *  for many data classes. Classes for which the write method is
	  *  explicitly defined correctly include \c MsgAddr, \c string and 
	  *  \c Time.
	  *  @brief Write data array to buffer.
	  *  @param x Data array to be written to buffer.
	  *  @param N Number of data items in array.
	  *  @return Number of data items copied to buffer.
	  */
	template<class T> 
	size_type write(const T x[], size_type N=1);

	/**  Template class to translate and copy a single data item to 
	  *  the output buffer. This operator has the same limitations
	  *  as the \c write() method.
	  *  @brief Copy a data item to the buffer.
	  *  @param x Reference to data item to be copied.
	  *  @return Refernce to this output translator.
	  *  @exception runtime_error if the buffer is full or undefined.
	  */
	template<class T> 
	TransOutput& operator<<(const T& x);

	/**  Class to copy a null-terminated character array (string) to 
	  *  the output buffer. The string is stored as a 2-byte count field
	  *  followed by the data characters.
	  *  @brief Write a character array.
	  *  @param x Constant pointer to null-terminated string.
	  *  @return Refernce to this output translator.
	  *  @exception runtime_error if the buffer is full or undefined.
	  */
	TransOutput& operator<<(const char* x);

	/**  Get the number of bytes written to the buffer. This includes 
	  *  any alignment padding.
	  *  @return Number of data bytes currently in the buffer.
	  */
	size_type getNBytes(void) const NOEXCEPT;

    private:
	char*  mData;
	size_type mIndex;
	size_type mLength;
    };  //  class lmsg::TransOutput


    inline bool
    TransOutput::available(size_type N) const {
	return mIndex + N <= mLength;
    }

    inline bool
    TransOutput::full(void) const {
	return mIndex >= mLength;
    }

    inline size_type
    TransOutput::getNBytes(void) const NOEXCEPT {
	return mIndex;
    }

    //--------------------------------------  Implement write template
    /**  Specialize write template for \c char array.
      */
    template<> size_type TransOutput::write(const char x[], size_type N);

    /**  Specialize write template for \c MsgAddr array.
      */
    template<> size_type TransOutput::write(const MsgAddr x[], size_type N);

    /**  Specialize write template for \c string array.
      */
    template<> size_type TransOutput::write(const std::string x[],size_type N);

    /**  Specialize write template for \c Time array.
      */
    template<> size_type TransOutput::write(const Time x[], size_type N);

    template<class T> 
    inline size_type 
    TransOutput::write(const T x[], size_type N) {
	align(sizeof(T));
	if (full()) return 0;
	if (!available(N*sizeof(T))) N = (mLength - mIndex)/sizeof(T);
	export_format_grinder.SwapOut(x, mData+mIndex, N);
	mIndex += N*sizeof(T);
	return N;
    }

    //--------------------------------------  Implement << by template;
    template<class T>
    inline TransOutput&
    TransOutput::operator<<(const T& x) {
	if (!write(&x)) throw std::runtime_error("Buffer overflow");
	return *this;
    }

    inline TransOutput& 
    TransOutput::operator<<(const char* x) {
	return operator<<(std::string(x));
    }

}    // namespace lmsg

#endif  //  LMSG_TRANSOUTPUT_HH
