/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_SOCKETPOOL_HH
#define LMSG_SOCKETPOOL_HH
#include "gmutex.hh"
#include <list>

namespace lmsg {
    class Socket;

    /**  The SocketPool class holds client sockets that hav already been 
      *  opened and may be reused for client connections.
      *  @brief Reusable socket pool
      *  @author John Zweizig
      *  @version 1.2; Last modified March 4, 2008
      *  @ingroup IO_lmsg
      */
    class SocketPool {
    public:
	/**  Socket request type enumeration.
	  *  <table>
	  *    <tr><td>Value</td><td>Meaning</td></tr>
	  *    <tr><td>s_new</td><td>Allocate a new socket and bind it to
	  *                          the specified address.</td></tr>
	  *    <tr><td>s_reusable</td>Get a socket from the pool if available
	  *                           <td></td></tr>
	  *  </table>
	  *  @brief Socket request type.
	  */
	enum requestType {s_new, s_reusable};
    public:
	/**  Destroy a socket pool and close all remaining sockets.
	  *  @brief Destructor.
	  */
	~SocketPool(void);

	/**  Create an empty socket pool.
	  *  @brief default constructor.
	  */
	SocketPool(void);

	/**  Get a socket from the pool. If a new socket is requested 
	  *  (i.e. the request type is \c s_new ) or there are no available
	  *  sockets in the pool, a new socket is allocated and bound to 
	  *  the specified address. If the IP address or the port number 
	  *  is zero, the socket will be bound to an available ip address 
	  *  and/or port. The socket return address is set to the current 
	  *  pool. If a reusable socket is requested and an idle socket 
	  *  is available, a socket is pulled from the pool.
	  *  @brief Get a socket from the pool.
	  *  @param type Socket type, either \c s_new or \c s_reusable.
	  *  @param addr IP address and port to which a new socket is bound.
	  *  @return Pointer to the allocated socket descriptor.
	  */
	Socket* getSocket(requestType type, const MsgAddr* addr=0);

	/**  Return the specified socket to the pool.
	  *  @brief Return a socket.
	  *  @param s Pointer to the socket descriptor to be returned.
	  */
	void returnSocket(Socket* s);
    private:
	typedef std::list<Socket*> SocketList;
	typedef SocketList::iterator Pool_Iter;
	SocketList mClient;
	thread::mutex mMux;
    }; // SocketPool class

} // namespace lmsg

extern lmsg::SocketPool defaultLMsgSocketPool;

#endif // LMSG_SOCKETPOOL_HH
