/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_TEXT_HH
#define HTML_TEXT_HH

#include "html/textAttr.hh"
#include "html/object.hh"
#include <string>

class Time;

namespace html {
  class color;
  class font;
  class size;

  /**  The text class defines an atomic html object containing a text string
    *  and a list of text-specific attributes specific to the string. Any
    *  occurrence of an open bracket \c "<" is translated to \c "&lt;" before 
    *  it is added to the text string.
    *  @memo Html text object
    *  @author J. Zweizig
    *  @version 1.1; Modified May 4, 2001
    *  @ingroup IO_html
    */
  class text : public object {
  public:
    /**  Construct an empty text object.
      *  @memo Default constructor.
      */
    text(void);

    /**  Construct a text object and set it to contain the specified 
      *  string data.
      *  @memo Data constructor
      *  @param x String containing the initial text.
      */
    text(const std::string& x);

    /**  Construct a text object from a double precision float.
      *  @memo Construct numeric text.
      */
    text(double x);

    /**  Construct a text object from a Time.
      *  @memo Construct time string text.
      */
    text(const Time& x, const char* fmt="%M %d, %Y %02H:%02N");

    /**  Destroy a text object.
      *  @memo Text destructor.
      */
    virtual ~text(void);

    /**  Create a text object identical to the existing instance and return
      *  a pointer to the result. This method overrides the default action 
      *  of the html object clone virtual method.
      *  @memo Clone a text object.
      *  @return pointer to an exact copy of the current text instance.
      */
    text* clone(void) const;

    /**  Write the object text to the specified html writer stream.
      *  @memo Write the text.
      *  @param out Html write to which the text is written.
      */
    void write(writer& out) const;

    /**  Replace any existing text with the specified string.
      *  @memo Set the text.
      *  @param txt String containg new text.
      */ 
    void insertText(const std::string& txt);

    /**  Append a specified string to any existing text on the object.
      *  @memo Append a text string.
      *  @param txt String to be added to the text.
      */
    void addText(const std::string& txt);

    /**  Append a specified string to any existing text in the object.
      *  @memo Append a string to the text.
      *  @param txt String to be added to the text.
      *  @return Reference to the updated text object.
      */
    text& operator<<(const std::string& txt);

    /**  Append a numeric value to any existing text in the object.
      *  @memo Append a number to the text.
      *  @param x Number to be added to the text.
      *  @return Reference to the updated text object.
      */
    text& operator<<(double x);

    /**  Set the color attribute of the text.
      *  @memo Set the color.
      *  @return Reference to the updated text.
      *  @param Color A color specifier.
      */
    text& setColor(const color& Color);

    /**  Set the text font.
      *  @memo Set the font.
      *  @return Reference to the updated text.
      *  @param Font A text font specifier.
      */
    text& setFont(const font& Font);

    /**  Set the size attribute of the text.
      *  @memo Set the size.
      *  @return Reference to the updated text.
      *  @param Size A text size specifier.
      */
    text& setSize(const size& Size);

  private:
    textAttr    mAttr;
    std::string mText;
  };

}

#endif  // HTML_TEXT_HH
