/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_COMPOUND_HH
#define HTML_COMPOUND_HH

#include "html/Attrib.hh"
#include "html/object.hh"
#include <list>

namespace html {

  /**  Compound objects are HTML object containing one of more simple 
    *  or compound sub-objects. In addition to a list of constituent
    *  objects, the compound object contains an attribute list that is
    *  valid through the scope of the compound.
    *  @memo Compound object container.
    *  @author John Zweizig
    *  @version 1.1; Modified January 17, 2007
    *  @ingroup IO_html
    */
  class compound : public object {
  public:
    /**  Construct an empty  compound.
      *  @memo Null constructor.
      */
    compound(void);

    /**  Construct a compound that is identical to the argument. All
      *  constituents of the argument are cloned and added into the 
      *  new compound.
      *  @memo Copy constructor.
      *  @param x compound to be copied.
      */
    compound(const compound& x);

    /**  Destroy the compound object and all its constituent objects.
      *  @memo Compound destructor.
      */
    virtual ~compound(void);

    /**  Set the compound scope attributes and Write out all the consituent 
      *  objects.
      *  @memo Write the compound.
      *  @param out HTML writer to receive the compound and its constituents.
      */
    virtual void write(writer& out) const;

    /**  A new compound object is created from clones of all attributes and 
      *  constituent objects found in the original.
      *  @memo clone the compound object
      *  @return pointer to the clone.
      */ 
    virtual compound* clone(void) const;

    /**  Get the tag name to be used to describe this compound object.
      *  @memo Get the compound tag name.
      *  @return Pointer to a character string containing the tag name.
      */
    virtual const char* getTag(void) const;

    /**  The argument object is cloned and the clone is added to the 
      *  end of the compound object list. A reference to the cloned object
      *  is returned.
      *  @memo Add the specified object to the compound.
      *  @param thing Object to be added to the compound.
      *  @return Reference to the cloned object in the compound.
      */
    object& addObject(const object& thing);

    /**  The argument object is added to the end of the compound object
      *  list. Ownership of the object is taken by the compound. A pointer
      *  to the object is returned.
      *  @memo Add the specified object to the compound.
      *  @param thing Object to be added to the compound.
      *  @return Pointer to the object added to the compound.
      */
    object* addObject(object* thing);

    /**  Get a reference to the last object in the compound object list.
      *  @memo Get back object reference.
      */
    object& back(void);

    /**  Get a constant reference to the last object in the compound 
      *  object list.
      *  @memo Get back object reference.
      */
    const object& back(void) const;

    /**  Get a reference to the first object in the compound object list.
      *  @memo Get front object reference.
      */
    object& front(void);

    /**  Get a constant reference to the first object in the compound 
      *  object list.
      *  @memo Get front object reference.
      */
    const object& front(void) const;

    /**  Add a line-break marker.
     */
    object& lineBreak(void);

    /**  Remove all objects from the constituent list.
      *  @memo clear the compound.
      */
    void clear(void);

    /**  The specified object us removed from the constitent list and deleted.
      *  @memo remove a constituent object from a compound.
      *  @param thing Pointer to the object to be removed.
      */
    void remObject(const object* thing);

    /**  A clone of the specified attribute is added to the compound scope 
      *  attribute list with the given name.
      *  @memo Add an attribute.
      *  @param name Name of attribute to be added.
      *  @param a    Attribute to be added to the compound scope.
      */
    void addAttr(const std::string& name, const attrib& a);

    /**  Remove all attributess from the compound scope.
      *  @memo clear the compound.
      */
    void clearAttr(void);

    /**  Replace the named attribute in the compound scope with a clone of 
      *  the specified attribute.
      *  @memo Replace an attribute.
      *  @param name Name of attribute to be replaced.
      *  @param a    Replacement attribute.
      */
    void repAttr(const std::string& name, const attrib& a);

    /**  Remove the named attribute.
      *  @memo remove the named attribute
      *  @param name Attribute to be removed from the compound scope.
      */
    void remAttr(const std::string& name);

  private:
    typedef std::list<object*> HtmlList;
    typedef HtmlList::iterator html_iter;
    typedef HtmlList::const_iterator const_html_iter;

  private:
    attList  mAttr;
    HtmlList mList;
  };

  /**  The block class implements named compounds. The block name is 
    *  used as an HTML tag when the block is written to the document
    *  file.
    *  @memo Named compound HTML objects.
    *  @author J.Zweizig
    *  @version 1.1; Modified May 4, 2001
    *  @ingroup IO_html
    */
  class block : public compound {
  public:
    /**  Construct a block. 
      *  @memo Constructor
      *  @param tag Name of the block.
      */
    block(const std::string& tag) : mTag(tag) {}

    /**  Destroy a block.
      *  @memo Destructor.
      */
    virtual ~block(void) {}

    /**  Create an exact duplicate of the current block. This method
      *  pre-empts the HTML object clone virtual method.
      *  @memo Clone a block.
      *  @return Pointer to the newly created copy of the block.
      */
    virtual block* clone(void) const {return new block(*this);}

    /**  Get the tag name of this block.
      *  @memo get the tag.
      *  @return Constant pointer to a static string containing the block name.
      */
    const char* getTag(void) const {return mTag.c_str();}

  private:
    std::string mTag;
  };

}

//======================================  Inline methods.
inline void 
html::compound::addAttr(const std::string& name, const attrib& a) {
    mAttr.addAttr(name, a);
}

inline html::object*
html::compound::addObject(object* obj) {
    mList.push_back(obj);
    return obj;
}

inline html::object&
html::compound::addObject(const object& obj) {
    return *addObject(obj.clone());
}

inline html::object&
html::compound::back(void) {
    return *mList.back();
}

inline const html::object&
html::compound::back(void) const {
    return *mList.back();
}

inline html::object&
html::compound::front(void) {
    return *mList.front();
}

inline const html::object&
html::compound::front(void) const {
    return *mList.front();
}

inline void 
html::compound::repAttr(const std::string& name, const attrib& a) {
    mAttr.repAttr(name, a);
}

inline void 
html::compound::remAttr(const std::string& name) {
    mAttr.remAttr(name);
}

#endif  // HTML_COMPOUND_HH
