/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "framexmittypes.hh"
#include <sys/time.h>
#include <iostream>
#include <fstream>
#include <cstdlib>

// PBTEASEPACKET0 defined in framexmittypes
 
using namespace std;

namespace framexmit {

   fxparameters::fxparameters(void)
      : frameXmitPort(7096),
	mcast_TTL(1),
	daemonPriority(1),         // priority of daemon
	sndDefaultBuffers(7),      // sender default number of buffers
	packetSize(64000),         // packet size (in bytes)
	packetBurst(4),            // continues packet burst (in # of packets)
	packetBurstInterval(1000), // minimum time between bursts (in usec)
	sndDelayTick(1000),        // sender delay if no packets (usec)
	rcvDelayTick(1000),        // receiver idle if no packets received(usec)
	lostPacketWait(15000),     // Max. wait time for a lost packet in 
	                           //      original stream (in usec)
	maximumRetransmit(200),    // maximum number of packets for retransmit
	sndInBuffersize(65536),    // sender socket input buffer length (bytes)
	sndOutBuffersize(4*65536), // sender socket output buffer length (bytes)
	rcvInBuffersize(4*65536),  // receive socket input buffer length (bytes)
	rcvInBufferMax(16*65536),  // max. socket input buffer length (bytes)
	rcvOutBuffersize(65536),   // receive socket output buffer length(bytes)
	rcvpacketbuffersize(1024), // receiver packet buffer length (packets)
	maxRetry(5),               // maximum number of retries to rebroadcast
	retryTimeout(250000),      // timeout for each retry (usec)
	maxSequenceOutOfSync(5),   // maximum a sequence can be out of sync
	retransmissionAverage(10.1), // number of data segments used to determine
                                   // the average retransmission rate
	rcvErrorRate(0),           // artifically introduced receiver error
                                   // rate, only valid if compiled with DEBUG
	sndMaxPacketRebroadcast(3),  // maximum times a packet is rebroadcast
	sndRebroadcastEpoch(250000), // Length of rebroadcast epoch (usec)
#if defined(PBTEASEPACKET0)
        teasePacket0(0),           // Default to teasing out unseen packets
	lastFrameDurationDefault(4), // The duration of the last frame (in seconds)
	frameDrawRebroadcastInterval(450), // Issue rebroadcasts to draw out an unseen frame in 
	drawOutFrameCountMax(1),   // The maximum number of unseen frames to draw out
	maxFrameDuration(10000),   // The maximum allowable length of frame (in seconds)
	maxRetryDrawOut(5),        // The maximum number of times we try to draw out a frame
#endif // PBTEASEPACKET0
      minInterBufferTime(0),       // Minimum time (usec) between buffers
      nRepeatPktMax(0),            // Maximum buffer size (pkts) to be repeated
      nRepeatPktNum(0)             // Number of packets to be repeated
   {
   }

   //=====================================  Set parameters from the default file
   void 
   fxparameters::init(void) {
      const char* home = getenv("HOME");
      std::string file;
      if (home) file = home;
      if (file.empty()) file = ".";
      file += "/.framexmitrc";
      read_parameters(file);
   }

   //===================================  Set parameters from a resource file.
   void 
   fxparameters::read_parameters(const std::string& filename) {
      ifstream in(filename.c_str());
      if (!in.is_open()) return;
      while (in.good()) {
	 //-----------------------------  Get a line, delete everything after '#'
	 string line;
	 getline(in, line);
	 string::size_type inx = line.find('#');
	 if (inx != string::npos) line.erase(inx);

	 //-----------------------------  Remove leading blanks. 
	 while (!line.empty() && (line[0] == ' ' || line[0] == '\t')) line.erase(0,1);
	 if (line.empty()) continue;

	 //-----------------------------  Pick up the parameter name
	 inx = line.find_first_of(" \t:=");
	 if (inx == string::npos) {
	    cerr << "Syntax error in framexmit parameter file: " << filename << endl;
	    return;
	 }
	 string param = line.substr(0, inx);
	 line.erase(0, inx+1);

	 //----------------------------- Remove leading blanks in value, set parameter.
	 while (!line.empty() && (line[0] == ' ' || line[0] == '\t')) line.erase(0,1); 
	 set_parameter(param, strtod(line.c_str(), 0));
      }
   }

   //=====================================  Set a parameter
   bool
   fxparameters::set_parameter(const string& name, double data) {
      if (name == "mcast_TTL") {
	 mcast_TTL = data;  // time-to-live counter preset
      }
      else if (name == "sndMaxPacketRebroadcast") {
	 sndMaxPacketRebroadcast = data;
      }
      else if (name == "sndRebroadcastEpoch") {
	 sndRebroadcastEpoch = data;  // Length of rebroadcast epoch (usec)
      }
      else if (name == "packetBurst") {
	 packetBurst = data;  // Number of packets in a burst
      }
      else if (name == "packetBurstInterval") {
	 packetBurstInterval = data;  // delay between successive bursts
      }
      else if (name == "rcvDelayTick") {
	 rcvDelayTick = data;  // delay between successive bursts
      }
      else if (name == "lostPacketWait") {
	 lostPacketWait = data;  // Time to wait for missing packet (usec)
      }
      else if (name == "minInterBufferTime") {
	 minInterBufferTime = data;  // minimum inter-buffer delay
      }
      else if (name == "nRepeatPktMax") {
	 nRepeatPktMax = data;  // Maximum buffer size for repeat
      }
      else if (name == "nRepeatPktNum") {
	 nRepeatPktNum = data;  // Number of buffers to be repeated.
      }
#if defined(PBTEASEPACKET0)
      else if (name == "teasePacket0") {
	 teasePacket0 = data;  // Enable packet 0 teasing.
      }
      else if (name == "lastFrameDurationDefault") {
	 lastFrameDurationDefault = data;   // last frame duration (in seconds)
      }
      else if (name == "frameDrawRebroadcastInterval") {
	 frameDrawRebroadcastInterval = data;  // frame teasing recast interval
      }
      else if (name == "drawOutFrameCountMax") {
	 drawOutFrameCountMax = data; // maximum tries to draw out a frame
      }
      else if (name == "maxFrameDuration") {
	 maxFrameDuration = data; // maximum allowable frame length (10000 s)
      }
      else if (name == "maxRetryDrawOut") {
	 maxRetryDrawOut = data; // maximum retries to draw out a frame
      }
#endif // defined(PBTEASEPACKET0)
      else {
	 cerr << "fxParameters: Undefined parameter: " << name << endl;
	 return false;
      }
      return true;
   }

   //=====================================  Display all parameter values
   void
   fxparameters::display(ostream& out) const {
      out << endl << "  Frame send/recv parameters" << endl << endl;
      out << "Compilation date:         " << __DATE__ << " " << __TIME__
	  << endl << endl;
      out << "frameXmitPort             " << frameXmitPort 
	  << "  IP port number" << endl;
      out << "mcast_TTL                    " << int(mcast_TTL) 
	  << "  Multicast time-to-live (number of hops)" << endl;
      out << "daemonPriority               " << daemonPriority 
	  << "  priority of daemon" << endl;
      out << "sndDefaultBuffers            " << sndDefaultBuffers
	  << "  sender default number of buffers" << endl;
      out << "packetSize               " << packetSize
	  << " packet size (in bytes)" << endl;
      out << "packetBurst                  " << packetBurst
	  << "  continues packet burst (in # of packets)" << endl;
      out << "packetBurstInterval       " << packetBurstInterval
	  << "  minimum time between bursts (in usec)" << endl;
      out << "sndDelayTick              " << sndDelayTick
	  << "  sender delay if no packets (usec)" << endl;
      out << "rcvDelayTick              " << rcvDelayTick
	  << "  receiver idle if no packets received(usec)" << endl;
      out << "lostPacketWait           " << lostPacketWait
	  << "  Max. wait time for a lost packet in " << endl
	  << "                               original stream (in usec)" 
	  << endl;
      out << "maximumRetransmit          " << maximumRetransmit
	  << "  maximum number of packets for retransmit" << endl;
      out << "sndInBuffersize          " << sndInBuffersize
	  << "  sender socket input buffer length (bytes)" << endl;
      out << "sndOutBuffersize        " << sndOutBuffersize
	  << "  sender socket output buffer length (bytes)" << endl;
      out << "rcvInBuffersize         " << rcvInBuffersize
	  << "  receive socket input buffer length (bytes)" << endl;
      out << "rcvInBufferMax         " << rcvInBufferMax
	  << "  max. socket input buffer length (bytes)" << endl;
      out << "rcvOutBuffersize         " << rcvOutBuffersize
	  << "  receive socket output buffer length(bytes)" << endl;
      out << "rcvpacketbuffersize       " << rcvpacketbuffersize
	  << "  receiver packet buffer length (packets)" << endl;
      out << "maxRetry                     " << maxRetry
	  << "  maximum number of retries to rebroadcast" << endl;
      out << "retryTimeout            " << retryTimeout
	  << "  timeout for each retry (usec)" << endl;
      out << "maxSequenceOutOfSync         " << maxSequenceOutOfSync
	  << "  maximum a sequence can be out of sync" << endl;
      out << "retransmissionAverage     " << retransmissionAverage
	  << "  number of data segments used to determine" << endl
	  << "                                the average retransmission rate" 
	  << endl;
      out << "rcvErrorRate                 " << rcvErrorRate
	  << "  artifically introduced receiver error rate," << endl
	  << "                                only valid if compiled with DEBUG"
	  << endl;
      out << "sndMaxPacketRebroadcast      " << sndMaxPacketRebroadcast
	  << "  maximum times a packet is rebroadcast" << endl;
      out << "sndRebroadcastEpoch     " << sndRebroadcastEpoch
	  << "  Length of rebroadcast epoch (usec)" << endl;
#if defined(PBTEASEPACKET0)
       out << "teasePacket0                 " << teasePacket0
	   << " Will tease out packet 0 of unseen frame if non-zero" << endl;
       out << "lastFrameDurationDefault     " << lastFrameDurationDefault
	   << " Default duration of the last frame (in seconds)" << endl;
       out << "frameDrawRebroadcastInterval " << frameDrawRebroadcastInterval
	   << " Issue rebroadcasts to draw out an unseen frame in these intervals (msec)" << endl;
       out << "maxRetryDrawOut              " << maxRetryDrawOut
	   << " The maximum number of times we try to draw out a frame" << endl;
       out << "drawOutFrameCountMax         " << drawOutFrameCountMax
	   << " The maximum number of unseen frames to draw out" << endl;
       out << "maxFrameDuration             " << maxFrameDuration
	   << " The maximum duration of a frame (in seconds)" << endl;
#endif // PBTEASEPACKET0
      out << "  minInterBufferTime    " << minInterBufferTime
	  << "  Minimum time between buffer broadcasts (usec)" << endl;
      out << "  nRepeatPktMax         " << nRepeatPktMax
	  << "  Maximum buffer size (packets) for automatic rebroadcast" 
	  << endl;
      out << "  nRepeatPktNum         " << nRepeatPktNum
	  << "  Maximum number of packets to be automatically rebroadcast"
	  << endl;
   }

   //=====================================  Get current time in microseconds
   timestamp_type
   get_timestamp(void) {
      timeval tp;
      timestamp_type timestamp = 0;
      if (!gettimeofday(&tp, 0) ) {
	 timestamp = int64_t(tp.tv_sec) * 1000000 + int64_t(tp.tv_usec);
      }
      return timestamp;
   }

   //=====================================  Get current time in microseconds
   int
   micro_delay(int delta) {
      timespec wait;
      wait.tv_sec  = delta / 1000000;
      wait.tv_nsec = (delta % 1000000) * 1000;
      return nanosleep (&wait, 0);
   }

   //=====================================  Define the external parameter list.
   fxparameters par;
}
