/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SCANDIR_HH
#define SCANDIR_HH

#include <string>

struct dirent;

/**  Scan the contents of a filesystem directory.
  *  \brief Filesystem directory scanner.
  *  \author John Zweizig (john.zweizig@ligo.org)
  *  \version $Id$
  */
class scandir {
public:
    /**  Default constructor
      */
    scandir(void);

    /**  Construct a directory scanner and  open it to the specified 
      *  directory.
      *  \brief Construct and open.
      *  \param dir Directory path.
      */
    scandir(const std::string& dir);

    /**  Destroy the %scandir instance and release all owned resources.
      *  \brief Destructor.
      */
    ~scandir(void);

    /**  Start scanner processing on the specified directory path.
      *  \brief Open the scanner.
      *  \param dir Sirectory path to be read.
      *  \return True if open was successful.
      */
    bool open(const std::string& dir);

    /**  Terminate processing on the current directory.
      *  \brief Close the scanner.
      */
    void close(void);

    /**  Get the frame file duration field. This function is valid only 
      *  if the current entry is a frame file name and after is_frame() 
      *  has been tested for the current event.
      *  \brief Frame file duration field.
      *  \return Duration of frame data in the file.
      */
    unsigned int duration(void) const;

    /**  Get the current entry file name string.
      *  \brief Get the entry file name. 
      *  \return Current entry file name string.
      */
    std::string file_name(void) const;

    /**  Get the current entry file path string.
      *  \brief Get the entry file npath. 
      *  \return Current entry file npath string.
      */
    std::string file_path(void) const;

    /**  Get the frame file gps field. This function is valid only 
      *  if the current entry is a frame file name and after is_frame() 
      *  has been tested for the current event.
      *  \brief Frame file gps field.
      *  \return GPS start time of frame data.
      */
    unsigned long gps(void) const;

    /**  Test whether the current entry is a directory.
      *  \brief Test for directory.
      *  \return True if the current entry is a directory.
      */
    bool is_dir(void) const;

    /**  Test whether the current entry is a file.
      *  \brief Test for file
      *  \return True if the current entry is a file.
      */
    bool is_file(void) const;

    /**  Test whether the current entry is a ligo frame-file. The 
      *  file name is tested for the form \c xxxxx-\<gps>-\<dt>.gwf.
      *  If the file name is in the correct form an internal flag is 
      *  set and the GPS time and duration are saved. The internal 
      *  frame flag is reset when another directory entry read is 
      *  attempted or when the scanner is closed.
      *  \brief Test for frame file.
      *  \return True if the current entry is a frame file.
      */
    bool is_frame(void);

    /**  Test whether the current entry is a ligo file name. The 
      *  file name is tested for the form \c xxxxx-\<gps>-\<dt>.\<ext>.
      *  If the file name is in the correct form an internal flag is 
      *  set and the GPS time and duration are saved. The internal 
      *  frame flag is reset when another directory entry read is 
      *  attempted or when the scanner is closed.
      *  \brief Test for ligo-standard file name.
      *  \param ext file name type. 
      *  \return True if the current entry is a ligo standard file.
      */
    bool is_ligo_file(const std::string& ext);

    /**  Test whether the scanner is open on a directory.
      *  \brief Test for for open.
      *  \return True if the scanner has an open directory.
      */
    bool is_open(void) const;

    /**  Test whether the current entry is a symbolic link.
      *  \brief Test for symlink.
      *  \return True if the current entry is a symbolic link.
      */
    bool is_symlink(void) const;

    /**  Read the next directory entry. Set the status variables.
      *  \brief Read a directory entry.
      *  \return true if the directory entry was read successfully.
      */
    bool next_entry(void);

    /**  Reference to th current frame name prefix (observatory and frame-type 
      *  codes). Note that this is not valid unless is_frame() has been called 
      *  and returned true.
      *  \brief Frame prefix. 
      *  \return Frame name prefix.
      */
    const std::string& prefix(void) const;

    /**  Test whether the specified path is a directory.
      *  \brief Test for directory.
      *  \param path File pat to be tested.
      *  \return True if directory path.
      */
    static bool test_directory(const std::string& path);

private:
    std::string   _dirName;
    void*         _dirHandle;
    dirent*       _dirEntry;
    dirent*       _dirPtr;
    bool          _frameEntry;
    std::string   _framePrefix;
    unsigned long _frameGPS;
    unsigned int  _frameDt;
};

//==================================  Inline methods
inline unsigned int 
scandir::duration(void) const {
    if (_frameEntry) return _frameDt;
    return 0;
}

inline unsigned long
scandir::gps(void) const {
    if (_frameEntry) return _frameGPS;
    return 0;
}

inline bool
scandir::is_open(void) const {
    return _dirHandle;
}

inline const std::string&
scandir::prefix(void) const {
    return _framePrefix;
}

#endif // !defined(SCANDIR_HH)
