/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FRAMEDIR_HH
#define FRAMEDIR_HH

//
// Frame finder class.
//
#include <map>
#include <iosfwd>
#include <iterator>
#include <string>
#include "PConfig.h"
#include "Time.hh"
#include "Interval.hh"
#include "fferrors.hh"

/**  The frame file data structure contains information about the frames 
  *  in a specified frame file. A flag in the ffData object specifies 
  *  whether the information has been gathered by reading the file or by
  *  inference.
  *  @memo Frame file data.
  *  @author J. Zweizig
  *  @version V1.0; Modified February 3, 2000
  *  @ingroup IO_futils
  */
struct ffData {
public:
    /** GPS time data type.
      */
    typedef Time::ulong_t gps_t;

    /** Count data type.
     */
    typedef unsigned long count_t;
   
    /**  Default constructor.
     *  @memo Default constructor.
     */
    ffData(void);
   
    /**  Construct a table entry for a specified file. 
     *  This constructor is for files that DO NOT follow the
     *  naming convention.
     *  @memo Data constructor.
     *  @param File    Name of frame file
     *  @param time    Start time of first frame
     *  @param Dt      Length of time spanned by each frame.
     *  @param DataOK  True if measured data are supplied.
     */
    ffData(const char* File, const Time& time, 
	   Interval Dt=Interval(1.0), bool DataOK=true);
   
    /**  Construct a table entry for a specified file. 
     *  This constructor is for files that DO follow the
     *  naming convention.
     *  @memo Data constructor.
     *  @param Prefix  Name prefix of frame file
     *  @param Suffix  Name suffix of frame file
     *  @param time    Start time of first frame
     *  @param Dt      Length of time spanned by each frame.
     */
    ffData(const char* Prefix, const char* Suffix, const Time& time, 
	   Interval Dt=Interval(1.0));
	    
    /**  Get the length of time covered by the file.
     *  @memo Length of time covered by file.
     *  @return Interval covered by file.
     */
    Interval getDt(void) const;
   
    /**  Get the name prefix of the file described by this object.
     *  @memo File prefix.
     *  @return Pointer to the file prefix.
     */
    const char* getPrefix(void) const;
   
    /**  Get the name suffix of the file described by this object.
     *  @memo File suffix.
     *  @return Pointer to the file suffix.
     */
    const char* getSuffix(void) const;
	  
    /**  Get the name of the filed described by this object.
     *  The argument N can be used to derive a frame file
     *  name later in the series if the naming convention is
     *  followed. Setting N to anything than 0 if the naming
     *  convention is not followed will return an empty string.
     *  @memo File name.
     *  @param N       frame file number (0 for current).
     *  @return File name.
     */
    std::string getFile(count_t N = 0) const;

    /**  Advance the starting GPS by N frames.
     *  @memo advance the start time
     *  @param N number of frames to increment the time.
     *  @return True if successful.
     */  
    bool advance(int N);
      
    /**  Get the absolute starting time of the data in the file.
     *  @memo Start time.
     *  @return Start time for the first frame in the file.
     */
    Time getStartTime(void) const;
   
    /**  Get the absolute end time of the data in the file.
     *  @memo End time.
     *  @return End time for the last frame in the file.
     */
    Time getEndTime(void) const;
   
    /**  Test whether the file name follows the convention.
     *  @memo Test naming convention.
     *  @return true if naming convention is followed.
     */
    bool isFollowed(void) const;
   
    /**  Test whether the information has been gathered by reading
     *  the file or inferred from default values or other files in 
     *  the directory.
     *  @memo Test data quality.
     *  @return true if data have been gathered from the file.
     */
    bool isValid(void) const;
   
    /**  Get the starting GPS second.
     *  @memo GPS start time.
     *  @return Start time in GPS seconds of the first frame.
     */
    gps_t getStartGPS(void) const;
   
    /**  Get the end GPS second.
     *  @memo GPS end time.
     *  @return End time of the first frame in GPS seconds.
     */
    gps_t getEndGPS(void) const;
   
protected:
    /** File name including directory name (prefix if mNamingOK).
     */
    std::string mFilePrefix;
   
    /** File name including directory name (suffix if mNamingOK).
     */
    std::string mFileSuffix;
      
    /** Starting GPS time of first frame in file.
     */
    Time mStartGPS;
      
    /**  Duration of frame file.
     */
    Interval mDt;
   
    /**  File name follows naming convention
     */
    bool mNamingOK;
      
    /**  Data comes from reading frame, is inferred from file name
     *   or is inferred from other files in directory
     */
    bool mDataOK;
};

inline 
ffData::ffData(const char* File, const Time& time,  
	       Interval Dt, bool DataOK)
    : mFilePrefix(File), mStartGPS(time), mDt(Dt), mNamingOK(false), 
      mDataOK(DataOK)
{}

inline 
ffData::ffData(const char* Prefix, const char* Suffix, const Time& time, 
	       Interval Dt)
    : mFilePrefix(Prefix), mFileSuffix(Suffix), mStartGPS(time), 
      mDt(Dt), mNamingOK(true), mDataOK(true) 
{}
   	    
inline  
ffData::ffData(void) 
    : mNamingOK(false), mDataOK(false)
{}

inline Interval 
ffData::getDt(void) const {
    return mDt;
}

inline const char*
ffData::getPrefix(void) const {
    return mFilePrefix.c_str(); 
}
   
inline const char* 
ffData::getSuffix(void) const {
    return mFileSuffix.c_str(); 
}

inline bool 
ffData::isFollowed(void) const {
    return mNamingOK;
}

inline bool
ffData::advance (int N)
{
    if (!isFollowed()) {
	return false;
    }
    else {
	mStartGPS += mDt * double(N);
	return true;
    }
}
   
inline bool 
ffData::isValid(void) const {
    return mDataOK;
}
   
inline Time 
ffData::getStartTime(void) const {
    return mStartGPS;
}

inline Time 
ffData::getEndTime(void) const {
    return mStartGPS + mDt;
}

inline ffData::gps_t
ffData::getStartGPS(void) const {
    return mStartGPS.getS();
}

inline ffData::gps_t
ffData::getEndGPS(void) const {
    return getEndTime().getS();
}

/**  The frame file data series structure contains information about frame 
 *  files covering a continuous time and having identical file name 
 *  prefix and suffix.
 *  @memo Series of frame file data.
 *  @author D. Sigg
 *  @ingroup IO_futils
 */
struct ffDataSeries : public ffData {
public:
   
    /**  Default constructor.
     *  @memo Default constructor.
     */
    ffDataSeries(void) : mNFiles (0) {}
   
    /**  Construct a table entry for a specified file 
     *  that does not follow the naming convention.
     *  @memo Data constructor.
     *  @param File    Name of frame file
     *  @param time    Start time of first frame file
     *  @param Dt      Length of time spanned by each file.
     *  @param DataOK  True if measured data are supplied.
     */
    ffDataSeries(const char* File, const Time& time, 
		 Interval Dt=Interval(1.0), bool DataOK=true)
	: ffData (File, time, Dt), mNFiles (1) 
    {}
      
    /**  Construct a table entry for a specified file series
     *  that follows the naming convention.
     *  @memo Data constructor.
     *  @param Prefix  Prefix of name of frame file(s)
     *  @param Suffix  Suffix of name of frame file(s)
     *  @param time    Start time of first frame file
     *  @param N       Number of frames file in series.
     *  @param Dt      Length of time spanned by each file.
     */
    ffDataSeries(const char* Prefix, const char* Suffix,
		 const Time& time, Interval Dt=Interval(1.0), count_t N = 1)
	: ffData (Prefix, Suffix, time, Dt), mNFiles (N) 
    {}
         
    /**  Construct a series from a data object
     *  @memo Data constructor.
     */
    ffDataSeries(const ffData& data) : ffData (data), mNFiles (1) {}
   
    /**  Returns true if the two files series are joinable.
     *  @memo Joinable.
     *  @param series   File series.
     *  @return True if joinable series
     */
    bool joinable (const ffDataSeries& series) const;

    /**  Joins with another frame file series.
     *  @memo Join.
     *  @param series   File series.
     *  @return True if series could be joined
     */
    bool join (const ffDataSeries& series);
      
    /**  Tries to join a frame file to the series' end.
     *  @memo Join.
     *  @param Prefix  Prefix of name of frame file(s)
     *  @param Suffix  Suffix of name of frame file(s)
     *  @param time    Start time of first frame file
     *  @param Dt      Length of time spanned by each file.
     *  @return True if file could be joined
     */
    bool join (const char* Prefix, const char* Suffix, 
	       const Time& time, Interval Dt);
      
    /**  Get the absolute end time of the data series.
     *  @memo End time.
     *  @return End time for the last frame series.
     */
    Time getEndTime(void) const {
	return mStartGPS + mDt * double(mNFiles); }

    /**  Get the end GPS second.
     *  @memo GPS end time.
     *  @return End time of the first frame in GPS seconds.
     */
    gps_t getEndGPS(void) const {
	return getEndTime().getS(); }
     
    /**  Get the starting time of the nth data object.
     *  @memo Time.
     *  @return Start time for the nth data object.
     */
    Time getTime (count_t n) const {
	return mStartGPS + mDt * double(n); }
      
    /**  Get the GPS time of the nth data object.
     *  @memo Time.
     *  @return Start time for the nth data object.
     */
    gps_t getGPS (count_t n) const {
	return getTime(n).getS(); }
      
    /**  Get the length of time covered by the series.
     *  @memo Length of time covered by series.
     *  @return Interval covered by series.
     */
    Interval getLength(void) const {
	return mDt * double(mNFiles); }
	 
    /**  Get the number of frames in the file.
     *  @memo Number of frames.
     *  @return Number of frames in the file.
     */
    count_t getNFiles(void) const {
	return mNFiles; }
   
    /**  Set the number of frames in the file.
     *  @memo Number of frames.
     *  @return Number of frames in the file.
     */
    void setNFiles(count_t N) {
	mNFiles = N; }
   
private:     
    /**  Number of files in series.
     */
    count_t mNFiles;   
};


#ifdef __GNU_STDC_OLD
namespace std {
    template <class _Category, class _Tp, class _Distance = ptrdiff_t,
	      class _Pointer = _Tp*, class _Reference = _Tp&>
    struct iterator {
	typedef _Category  iterator_category;
	typedef _Tp        value_type;
	typedef _Distance  difference_type;
	typedef _Pointer   pointer;
	typedef _Reference reference;
    };
}
#endif
   
/**  Iterator for going thorugh a ffDataSeries map file by file.
 *  @memo Iterator for frame file data.
 *  @author D. Sigg
 *  @ingroup IO_futils
 */
struct ffDataConstIter : public std::iterator 
<std::bidirectional_iterator_tag, ffData, int, const ffData*, 
 const ffData&> {
public:
    /** GPS time data type.
     */
    typedef ffData::gps_t gps_t;

    /** Count data type.
     */
    typedef ffData::count_t count_t;

    /**  Directory map data type.
     */
    typedef std::map<gps_t, ffDataSeries> dmap_t;  

    /**  Directory map iterator type.
     */
    typedef dmap_t::const_iterator dmap_iterator;

    /** Default constructor
	@memo Default constructor
    ******************************************************************/
    ffDataConstIter () : mList(0), mNFiles (0) {
    }
    /** Constructor
	@memo Constructor
    ******************************************************************/
    ffDataConstIter (const dmap_t& List, const dmap_iterator& Iter,
		     count_t Index) : mList(&List), mIndex (Iter),
				      mNFiles (Index) {
	set();
    }
    /** Equality operator
	@memo Equality operator
    ******************************************************************/
    bool operator== (const ffDataConstIter& i) const {
	return ((mList == i.mList) && (mNFiles == i.mNFiles) &&
		(mIndex == i.mIndex)); }
    /** Inequality operator
	@memo Inequality operator
    ******************************************************************/
    bool operator!= (const ffDataConstIter& i) const {
	return !(*this == i); }
   
    /** Increment operator (prefix)
	@memo Increment operator (prefix)
    ******************************************************************/
    ffDataConstIter& operator++ () {
	add (1); 
	return *this; }
    /** Increment operator (postfix)
	@memo Increment operator (postfix)
    ******************************************************************/
    ffDataConstIter operator++ (int) {
	ffDataConstIter i = *this; add (1); 
	return i; }
    /** Decrement operator (prefix)
	@memo Decrement operator (prefix)
    ******************************************************************/
    ffDataConstIter& operator-- () {
	add (-1); 
	return *this; }
    /** Decrement operator (postfix)
	@memo Decrement operator (postfix)
    ******************************************************************/
    ffDataConstIter operator-- (int) {
	ffDataConstIter i = *this; add (-1); 
	return i; }
    /** Plus operator. Should be made more efficient!
	@memo Plus operator
    ******************************************************************/
    ffDataConstIter operator+ (int delta) const {
	ffDataConstIter i = *this; i.add (delta);
	return i; }
    /** Minus operator
	@memo Minus operator
    ******************************************************************/
    ffDataConstIter operator- (int delta) const {
	ffDataConstIter i = *this; i.add (-delta);
	return i; }
    /** Plus equal operator
	@memo Plus equal operator
    ******************************************************************/
    ffDataConstIter& operator+= (int delta) {
	add (delta);
	return *this; }
    /** Minus equal operator
	@memo Minus equal operator
    ******************************************************************/
    ffDataConstIter& operator-= (int delta) {
	add (-delta); 
	return *this; }
    /** Dereference operator
	@memo Dereference operator
    ******************************************************************/
    reference operator*() const {
	return mData; }
    /** Pointer operator
	@memo Pointer operator
    ******************************************************************/
    pointer operator->() const {
	return &mData; }
protected:
    /** Set data in mData
	@memo Set data
    ******************************************************************/
    void set (void);
    /** Add operator
	@memo Add operator
    ******************************************************************/
    void add (int delta);
private:
    /**  Current frame file.
     */
    ffData mData;
    /** Frame file list.
     */
    const dmap_t* mList;
    /**  Current index into list.
     */
    dmap_iterator mIndex;
    /**  Current frame file index.
     */
    count_t mNFiles;
};


/**  The frame directory class is used to keep a directory of frame files
 *  in memory. Each frame is represented by an \c ffData object indexed 
 *  by the starting GPS time. The data in the \c ffData structure are set 
 *  as follows:
 *  <ul>
 *  <li> The start time is inferred from the last numeric string in the 
 *       file name.</li>
 *  <li> The frame length and number of frames are assumed to be the 
 *       same as the preceding file if this is consistent with the time
 *       to the following file.</li>
 *  <li> If the inferred time indicates a gap between a file and the
 *       following file, the starting time and frame length are read from 
 *       the first frame header, and the number of frames in the file is
 *       read from the \c FrEndOfFile structure.</li>
 *  </ul>
 *  The frame file descriptors may be accessed sequentially using a 
 *  file_iterator type variable.
 *  @memo Frame Finder Class
 *  @author J. Zweizig
 *  @version V1.0; Modified February 3, 2000
 *  @ingroup IO_futils
 */
class FrameDir {
   
public:
    /**  GPS time data type.
     */
    typedef Time::ulong_t gps_t;

    /**  Directory map data type.
     */
    typedef std::map<gps_t, ffDataSeries> dmap_t;
   
    /**  Iterator provides sequential access to the data on the files in the
     *  directory.
     *  @memo File data iterator.
     */
    typedef ffDataConstIter file_iterator;
   
    /**  Iterator provides sequential access to the series data.
     *  @memo Series data iterator.
     */
    typedef dmap_t::const_iterator series_iterator;
      
public:
    /**  Erase all FrameDir entries.
     *  @memo FrameDir destructor.
     */
    virtual ~FrameDir(void);
   
    /**  Construct an empty frame directory.
     *  @memo Empty directory constructor.
     */
    FrameDir();
   
    /**  Construct a Frame directory and fill it with data from files 
     *  matching the specified specification. The frame file specification
     *  may contain wild-cards in the file name but not in the directory
     *  name.
     *  @memo Construct and fill a directory.
     *  @param dir Frame file specification.
     *  @param delayed Check files when needed, otherwise do it after add.
     */
    explicit FrameDir(const char* dir, bool delayed = false);

    //------------------------------------  Accessors
    /**  Get the data for the file that covers time t.
      *  \brief Get file data by time.
      *  \return Iterator pointing to the segment containing the specified time.
      *  \param t GPS time of requested data.
      */
    file_iterator find(const Time& t) const;

    /**  Get the iterator of the first period on or after the specified 
      *  time.
      *  @memo find the next data segment.
      *  @return Start time of the first data segment after the specified time.
      *  @param time Begin time for search.
      */
    file_iterator getStart(gps_t time) const;

    /**  Get the iterator of the first period after the specified time.
      *  @memo Find the end of the next data segment.
      *  @return End time of the first data segment after the specified time.
      *  @param time Begin time for search.
      */
    file_iterator getLast(gps_t time) const;
   
    /**  Get the specified debug level.
      *  \brief Debug level
      *  \return Debug print level.
      */
    int getDebug(void) const;
   
    /**  Get a file_iterator pointing the first (earliest) frame file entry 
      *  in the directory.
      *  @memo Get a iterator pointing to the sequential beginning.
      *  @return A file_iterator pointing to the beginning of the directory.
      */
    file_iterator begin(void) const;
   
    /**  Get a file_iterator pointing the end of the frame file directory.
      *  @memo Get a iterator pointing to the directory end.
      *  @return file_iterator pointing to the end of the directory.
      */
    file_iterator end(void) const;
   
    /**  Get a series_iterator pointing the first series entry 
      *  in the directory.
      *  @memo Get a iterator pointing to the beginning of the series.
      *  @return A series_iterator pointing to the beginning of the series.
      */
    series_iterator beginSeries(void) const;
   
    /**  Get a series_iterator pointing the end of the series.
      *  @memo Get a iterator pointing to the end.
      *  @return A series_iterator pointing to the end of the series.
      */
    series_iterator endSeries(void) const;

    /**  Returns the number of entries in the series list.
      *  \brief Number of series entries.
      *  \return The number of contiguous series in the directory.
      */
    int sizeSeries() const;

    /**  Returns the number of files in the frame directiry list.
      *  \brief  Number of file entries.
      *  \return Number of entries in the frame directory.
      */
    int size(void) const;

    /**  Returns true if there are no entries in the frame directory list.
      *  \brief Empty list?.
      *  \return True if the frame directory list has no entries.
      */
    bool empty(void) const {return mList.empty();}
   
    //------------------------------------  Mutators
   
    /**  Add all the frame files that match the path pattern according to the 
      *  fnmatch (glob) matching rules. 
      *  
      *  @memo Add one or more file paths to the dire.
      *  @param dir File name(s) to be added to the FrameDir.
      *  @param delayed Check files when needed, otherwise do it after add.
      */
    void add(const char* dir, bool delayed = false);
   
    /**  Add a single file to the frame directory. The file name must not
      *  contain wild-cards and must conform to the LIGO frame file naming
      *  standard i.e. the name must have the following form:
      *  "<directory>/<ifo-name>-<start-gps>[-<nsec>][.<extension>]". To
      *  allow backward compatibility, the data length (\c\<nsec>) and 
      *  extension (\c \<ext> ) may be omitted.
      *  The continuation number can be used to specify a series of
      *  frame files that cover a continueous duration, have the same
      *  length, path prefix and extension. This only works for file
      *  that provide the data length in their name. The continuation
      *  number specficies the number of additional files beyond the
      *  first one.
      *  @memo Add a file to the FrameDir.
      *  @param File Name of file to be added to the list.
      *  @param Cont Additional files following in series.
      */
    void addFile(const char* File, unsigned int Cont = 0);

    /**  Delete all file descriptors from the FrameDir list.
      *  @memo Delete all entries.
      */
    void clear(void);
   
    /**  Delete the frame series descriptor for the specified time.
     *  @memo Delete a frame file entry.
     *  @param time GPS time of start of 
     */
    void erase(gps_t time);
      
    /**  Set the debug printout threshold to a specified level.
     *  @memo Set debug level.
     *  @param Level New debug printout level.
     */
    void setDebug(int Level);

    /**  Read a list of frame files from a text file.
     *  @memo Read list file.
     *  @param File Name of file to be read.
     *  @return True if successful
     */
    bool read (const char* File);

    /**  Write the list of frame files to a text file.
     *  @memo Write list file.
     *  @param File Name of file to be written.
     *  @param Longlist If true, prints one file per line.
     *  @param Start Start time (GPS).
     *  @param Stop  Stop  time (GPS).
     *  @param SupHead If true, suppresses file:// head.
     *  @return True if successful
     */
    bool write (const char* File, bool Longlist = false,
		gps_t Start = 0, gps_t Stop = 0, bool SupHead = false);

    /**  Write the list of frame files to a text file.
     *  @memo Write list file.
     *  @param Os std::ostream to be written to.
     *  @param Longlist If true, prints one file per line.
     *  @param Start Start time (GPS).
     *  @param Stop  Stop  time (GPS).
     *  @param SupHead If true, suppresses file:// head.
     *  @return True if successful
     */
    bool write (std::ostream& Os, bool Longlist = false,
		gps_t Start = 0, gps_t Stop = 0, bool SupHead = false);

    /**  Write the list of frame files to an output stream.
     *  @memo Write list to stream.
     *  @param Os Output stream
     *  @return Output stream
     */
    std::ostream& operator<< (std::ostream& Os);
      
    /**  Parse a frame file name.
     *  @memo Parse frame file name
     *  @param File File name.
     *  @param time GPS time of frame.
     *  @param tlen Duration of frame.
     *  @param pre File name prefix (includes last dash).
     *  @param post File name postfix.
     *  @return True if a valid frame file name
     */
    static bool parseName (const char* File, 
			   gps_t& time, gps_t& tlen,
			   char* pre = 0, char* post = 0);
   
private:
    /**  Checkup mode specifier.
     */
    enum CheckLevel {none, gapsOnly, allData};
   
    /**  Check all entries in the frame directory list for contiguity.
     *  Depending on the level of checking demanded, files are inspected 
     *  in all cases or only to resolve data gaps.
     *  @memo Check frame coverage.
     *  @param Level of checking to be performed.
     */
    void checkData(CheckLevel lvl=gapsOnly);
    void checkData(CheckLevel lvl=gapsOnly) const;
   
    /**  Joins all series that can be joined.
     *  @memo Join.
     */
    void join();
      
    /**  Get a copy of the table entry for a specified file.
     *  @memo Get specified file data.
     */
    ffData getFileData(const char* File);
   
private:
    /**  Debug printout level.
     */
    int    mDebug;
    /**  Frame added since last check.
     */
    mutable bool   mDirty;
   
    /**  Frame file directory, keyed by GPS start times.
     */
    dmap_t mList;

    /**  Last element added.
     */
    dmap_t::iterator last_insert;
};

inline int 
FrameDir::getDebug(void) const {
    return mDebug;
}

inline void
FrameDir::setDebug(int Level) {
    mDebug = Level;
}

#endif   // FRAMEDIR_HH
