/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef NDS1SOCKET_HH
#define NDS1SOCKET_HH
//
//    This is a C++ interface for frame data.
//
#include "DAQC_api.hh"

namespace sends {

/** @page Network Data Access API (C++)
    This API provides a client interface for the "classic" NDS server. It
    allows the user to get channel data over the network. The syntax
    implemented by this interface is:
    <table>
    <tr>
    <td>Function</td>
    <td>DAQSocket method</td>
    <td>NDS command</td>
    </tr>
    <tr>
    <td>Get data from specified time</td>
    <td>RequestData</td>
    <td>start net-writer &lt;start&gt; &lt;duration&gt; [{"channel" ...} | all];</td>
    </tr>
    <tr>
    <td>Get file names</td>
    <td>RequestNames</td>
    <td>start name-writer all;</td>
    </tr>
    <tr>
    <td>Get available channel list</td>
    <td>Available</td>
    <td>status channels [2];</td>
    </tr>
    <tr>
    <td>Get fast online data</td>
    <td>RequestOnlineData</td>
    <td>start fast-writer [{"channel" ...} | all];</td>
    </tr>
    <tr>
    <td>Get slow online data</td>
    <td>RequestOnlineData</td>
    <td>start net-writer [{"channel" ...} | all];</td>
    </tr>
    <tr>
    <td>Get trend data</td>
    <td>RequestTrendData</td>
    <td>start trend [60] net-writer &lt;start&gt; &lt;duration&gt; 
    [{"channel" ...} | all];</td>
    </tr>
    <tr>
    <td>Stop a net-writer</td>
    <td>StopWriter</td>
    <td>kill net-writer nnnnnnnn;</td>
    </tr>
    </table>

    A typical online NDS client would do the following:
    \verbatim
    //---------  Construct a DAQD socket 
    DAQSocket nds;

    //---------  Open a socket to the specified server port.
    const char* servername = "nds-server:port";
    nds.open(servername);
    if (!nds.TestOpen()) fail("Open failed!");

    //---------  Specify the channel to be read.
    const char* chan = "channel name";
    if (!nds.AddChannel(chan)) fail("Add channel failed!");
    if (nds.RequestOnlineData()) fail("Data Request failed");

    //---------  Specify the channel to be read.
    float* Samples = new float[data_len];
    while (1) {
    int nData = nds.GetData((char*) Samples, data_len);
    if (nData <= 0) break;
    ...  Process data ...
    }
    \endverbatim
    @memo Access channel data through the network data server
    @author John Zweizig
    @version 0.1; Last modified March 5, 2008
    @ingroup IO_daqs
************************************************************************/

/*@{*/

/**  NDS1Socket provides a client interface to the Network Data Server.
 *  The server provides data in the CDS proprietary format or as standard
 *  frames. The interface may also be used to list channel names, or to
 *  specify the channels to be read in.
 *  @brief The DAQD socket class.
 *  @author John Zweizig and Daniel Sigg
 *  @version 1.2; last modified March 5, 2008
 *  @ingroup IO_daqs 
 */
class NDS1Socket : public DAQC_api {
public:
    using DAQC_api::Available;
    using DAQC_api::AddChannel;
    using DAQC_api::GetData;

    /**  Construct an unopened socket.
     */
    NDS1Socket(void);
   
    /**  Create a socket and connect it to the specified NDS1 server.
      *  The arguments have the same syntax as that passed to open().
      *  \brief Create a socket and connect it to a server.
      *  \param ipaddr  Server IP address
      *  \param ipport  Server port ID.
      *  \param RcvBufferLen System receive buffer size.
     */
    explicit NDS1Socket(const std::string& ipaddr, int ipport = DAQD_PORT,
                        long RcvBufferLen = 1048576);
   
    /**  Disconnect and close a socket.
      *  \brief Destructor. 
      */
    ~NDS1Socket(void);
   
    /**  Open an existing socket and connect it to an NDS1 server.
      *  The argument, \a ipaddr, specifies the IP address of the node on which 
      *  the network data server is running. It may be specified either as 
      *  a symbolic name or as four numeric fields separated by dots. open()
      *  returns zero if successful, a positive non-zero error code if one was 
      *  returned by DAQD or -1.
      *  \brief Open a connection to the specified server.
      *  \param ipaddr  Server IP address
      *  \param ipport  Server port ID.
      *  \param RecvBufferLen System receive buffer size.
      *  \return Zero on success, a server error response or -1.
      */
    int  open (const std::string& ipaddr, int ipport = DAQD_PORT, 
	       long RecvBufferLen = 1048576);

    /**  Disconnect and close a socket.
      *  \brief Close socket.
     */
    void close();
   
    /**  Flushes any pending input data from the socket.
      *  \brief Flush socket input.
      */
    void flush();

    /**  Add the specified channel to the request list. All channels may be 
      *  added by specifying "all" instead of a channel name.
      *  \brief Add a channel to the request list.
      *  \param chan   %Channel name.
      *  \param ty     %Channel type code.
      *  \param rate   Requested sample rate.
      *  \return 1 if successful.
      */
    int AddChannel(const std::string& chan, chantype ty, double rate);
 
    /**  The names, sample rates, etc. of all channels known by the server are 
      *  copied into the channel vector. Available() returns the number of 
      *  entries found or -1 if an error occurred.
      *  \brief List known channels.
      *  \param list    Vector to receive channel information.
      *  \param timeout Maximum wait time.
      *  @return Number of channels or -1 if an error occurred.
      */
    int addAvailable(std::vector<DAQDChannel>& list, wait_time timeout = -1);
 
    /**  The names, sample rates, etc. of all channels known by the server are 
      *  copied into the channel vector. Available() returns the number of 
      *  entries found or -1 if an error occurred.
      *  \brief List known channels.
      *  \param list    Vector to receive channel information.
      *  \param timeout Maximum wait time.
      *  @return Number of channels or -1 if an error occurred.
      */
    int Available(std::vector<DAQDChannel>& list, wait_time timeout = -1);
  
    /**  The names, sample rates, etc. of all channels known by the server are 
     *  copied into the channel vector. Available() returns the number of 
     *  entries found or -1 if an error occurred.
     *  \brief List known channels.
     *  \param chant   %Channel type selection
     *  \param gps     Valid time for channels.
     *  \param list    Vector to receive channel information.
     *  \param timeout Maximum wait time.
     *  @return Number of channels or -1 if an error occurred.
     */
    int addAvailable(chantype chant, long gps, std::vector<DAQDChannel>& list, 
		  wait_time timeout = -1);
   
    /**  A single block of data (including the header) is received and copied
      *  into the specified buffer. The data length (excluding the header) is 
      *  returned. GetData() returns -1 if an error is detected, or 0 if an
      *  end-of file record is found.
      *  \brief Receive block of data in the CDS proprietary format.
      *  \param timeout Maximum wait time for data.
      *  \return data length, negative on error
      */
    int GetData(wait_time timeout = -1);
   
    /**  A single data frame is received and copied to the specified buffer.
      *  The length of the Frame data is returned. GetFrame() returns -1 
      *  in case of an error or 0 if a trailer (end of file) block is received.
      *  \brief Receive a data frame.
      *  \param buf Buffer to receive frame data
      *  \param len Length of frame buffer.
      *  \return Frame data length or -1.
      */
    long GetFrame(char *buf, long len);
   
    /**  The next frame file name written by the NDS is copied to buf and the 
      *  data length is returned. The GetName returns -1 if a name-writer 
      *  hasn't been started, if the data buffer is too short, or if an error
      *  occurs in reading the data. GetData waits for a new message if one is 
      *  not available from the socket.
      *  \brief Receive a file name.
      *  \param buf Buffer to receive file name
      *  \param len Length of file name buffer.
      *  \return Name length or -1.
      */
    long GetName(char *buf, long len);
   
    /**  The network data server is requested to start a net-writer task
      *  for past data. Start time and duration are given in GPS seconds. 
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading CDS data.
      *  \param start Start time
      *  \param duration Number of seconds of data to fetch.
      *  \param timeout Maximum response wait time
      *  \return Request status code.
      */
    int  RequestData (unsigned long start, unsigned long duration, 
		      wait_time timeout = -1);
   
    /**  The network data server is requested to start a frame writer task.
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading frame data.
      *  \return Server response code.
      */
    int  RequestFrames();
   
    /**  The network data server is requested to start a name writer task.
      *  \brief Start reading file names.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Server response code.
      */
    int  RequestNames(wait_time timeout = -1);
   
    /**  The network data server is requested to start a net-writer task.
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading CDS data.
      *  \param stride Stride length to be received.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Server response code.
      */
    int  RequestOnlineData (double stride=1.0, wait_time timeout = -1);
   
    /**  The network data server is requested to start a trend net-writer 
      *  task. Start time and duration are given in GPS seconds. 
      *  Only channels explicitly specified by AddChannel() will be sent.
      *  \brief Start reading CDS trend data.
      *  \param start    GPS time of start of data to be retrieved.
      *  \param duration Number of seconds of data to be retrieved.
      *  \param mintrend True if minute trends.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Server response code.
      */
    int  RequestTrend (unsigned long start, unsigned long duration,
		       bool mintrend = false, wait_time timeout = -1);

    /**  Remove the specified channel from the channel list, or all channels 
      *  if "all" specified.
      *  \brief Remove a channel
      *  \param chan %Channel name string.
      */
    void RmChannel(const std::string& chan);

    /**  This function effectively countermands the RequestXXX() functions.
      *  StopWriter returns either the server response code or -1 if no 
      *  writer has been requested.
      *  \brief Stop a data writer.
      *  \return Server response code or -1.
      */
    int  StopWriter(void);
      
    /**  Known time intervals.
      *  The network data server is requested to return start time and 
      *  duration of the data stored on disk.
      *  \brief Time intervals of available data.
      *  \param chtyp    %Channel type for request.
      *  \param start    Variable to receive start time of available data.
      *  \param duration Variable to receive duration of available data.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Zero on success or server response code.
      */
    int Times(chantype chtyp, unsigned long& start, unsigned long& duration, 
	      wait_time timeout = -1);

    /**  The network data server is requested to return start time and 
      *  duration of the trend data stored on disk.
      *  \brief Time intervals of available trend data.
      *  \param start    Variable to receive start time of available data.
      *  \param duration Variable to receive duration of available data.
      *  \param mintrend True if request for minute trend data.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Zero on success or server response code.
      */
    int  TimesTrend (unsigned long& start, unsigned long& duration, 
		     bool mintrend = false, wait_time timeout = -1);

    /**  Execution is blocked until data are available to the socket. This
      *  can be used to wait for data after a request has been made. The
      *  calling function can then e.g. allocate a buffer before calling 
      *  GetData(), GetName() or GetFrame(). If poll is true the function
      *  returns immediately with 1 if data is ready, or 0 if not.
      *  \brief Wait for data to arrive.
      *  \param poll Poll flag (don't wait)
      *  \return 1: if data available, 0: no data or -1 on error.
      */
    int WaitforData (bool poll);

private:
    /**  Send a request.
     *  SendRequest sends a request specified by 'text' to the server and
     *  reads the reply status and optionally the reply text. 'reply' is 
     *  a preallocated buffer into which the reply is copied and 'length' 
     *  is the size of the reply buffer. If 'reply' is omitted or coded 
     *  as 0, the status will not be read. The reply status is returned by 
     *  SendRequest and the length of the reply message is returned in 
     *  *Size if Size is non-zero. Only the first message is read into 
     *  reply, so in general *Size != length.
     */
    int SendRequest(const std::string& text, char *reply=0, long length=0, 
		    long *Size=0, wait_time maxwait=-1);
   
    /**  Receive data from the socket.
      *  Up to 'len' bytes are read from the socket into '*buf'. If readall
      *  is true, RecvRec will perform as many reads as is necessary to 
      *  fill 'buf'.
      */
    int RecvRec(char *buf, long len, bool readall=false, 
		wait_time maxwait=-1);

    int RecvReconfig(count_type block_len, wait_time maxwait);

    /**  Receive a data header and data block.
      *  A memory buffer of correct length is allocated automatically. The 
      *  header abd data are stored in the recvBuf structure in the class data.
      *  \return number of data bytes, or <0 on error.
      */
    int RecvData(wait_time timeout=-1);
   
    /**  Send a record data to the socket.
     *  Up to 'len' bytes are written to the socket from '*buf'.
     */
    int SendRec(const char *buffer, long length, wait_time maxwait=-1);
  
private:
    /**  Socket number.
     */
    int   mSocket;
   
    /**  Set if 'all' channels was specified.
     *  The channel list is ignored if this flag is set.
     */
    bool  mGetAll;
   
    /**  ID number of the data writer currently in use.
     */
    char  mWriter[8];
   
    /**  Offline flag sent when the writer is started.
     */
    int   mOffline;
};

/*@}*/

} // namespace sends

#endif  //  DAQSOCKET_HH

