VCSID("@(#)$Id: DAQSocket.cc 7582 2016-03-01 22:25:48Z john.zweizig@LIGO.ORG $");
/* -*- mode: c++; c-basic-offset: 3; -*- */
//
//    Implementation of the C++ DAQ client interface (DAQSocket) class
//
//    Revision History
//
//    23-Feb-1999  J.Zweizig
//        First really working version.
//
//    26-Feb-1999  J.Zweizig
//        Better comments, lots of little fixes, byte-swapping support.
//
//    Notes:
//      1) The channel list data format is incorrectly documented. There 
//         is an extra 4-byte field at the start of the reply string, just 
//         after the channel count field. Also, starting with version 9,
//         there are two additional (undocumentd) 4-byte hex fields. For
//         now these are copied to fields more1 and more2 in the channel 
//         structure.
//      2) I have introduced code to swap bytes on little-endian machines.
//         This was tested on sadan (Linux alpha) on 26/02/1999.
//
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif
#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif
#include "PConfig.h"
#include "DAQSocket.hh"
#include "Time.hh"
#include "Interval.hh"
#include <cstdio>
#include <cerrno>
#include <unistd.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/file.h>
#include <iostream>
#include <cstring>
#include <sstream>

#define _TIMEOUT 10000000	/* 10 sec */
#ifdef P__WIN32
#define MSG_DONTWAIT 0
#endif

//======================================  Use union to avoid pointer casts
union int2float {
   int   i;
   float f;
};

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Forwards							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   using namespace std;
   using namespace thread;


//--------------------------------------  Hex conversion
   static int 
   CVHex(const char *text, int N) {
      int v = 0;
      for (int i=0 ; i<N ; i++) {
         v<<=4;
         if      ((text[i] >= '0') && (text[i] <= '9')) v += text[i] - '0';
         else if ((text[i] >= 'a') && (text[i] <= 'f')) v += text[i] - 'a' + 10;
         else if ((text[i] >= 'A') && (text[i] <= 'F')) v += text[i] - 'A' + 10;
         else 
            return -1;
      }
      return v;
   }

//--------------------------------------  Swap ints and shorts in place
   static void
   SwapI(int *array, int N) {
      char temp;
      for (int i=0 ; i<N ; i++) {
         char* ptr = (char*) (array + i);
         temp   = ptr[0];
         ptr[0] = ptr[3];
         ptr[3] = temp;
         temp   = ptr[1];
         ptr[1] = ptr[2];
         ptr[2] = temp;
      }
      return;
   }
/*
   static void
   SwapS(short *array, int N) {
      char temp;
      for (int i=0 ; i<N ; i++) {
         char* ptr = (char*) (array + i);
         temp   = ptr[0];
         ptr[0] = ptr[1];
         ptr[1] = temp;
      }
      return;
   }
*/
//--------------------------------------  Constructors
   DAQSocket::DAQSocket() 
   : mOpened(false), mDebug(false), mRcvBuffer (1048576), mGetAll(false), 
   mWriterType(NoWriter), mVersion (0), mRevision (0), mAbort (0)
   {
      int test = 0;
      *(char*) &test = 1;
      mReorder = (test == 1);
   }

   DAQSocket::DAQSocket(const char* ipaddr, int ipport, long RcvBufferLen) 
   : mOpened(false), mDebug(false), mRcvBuffer (RcvBufferLen), mGetAll(false), 
   mWriterType(NoWriter), mAbort (0)
   {
      int   test=0;
      *(char*) &test = 1;
      mReorder = (test == 1);
   
    //----------------------------------  Open the socket
      open(ipaddr, ipport, mRcvBuffer);
   }

//--------------------------------------  Destructor
   DAQSocket::~DAQSocket() {
      if (mOpened) close();
   }

//--------------------------------------  Open a socket
   int 
   DAQSocket::open(const char* ipaddr, int ipport, long RcvBufferLen) 
   {
   
      semlock lockit (mux);
      struct sockaddr_in socknam;
      char version[4];
      long size;
      mRcvBuffer = RcvBufferLen;
   
    //----------------------------------  Make sure the socket isn't open
      if (mOpened) 
         return -1;
   
    //----------------------------------  Get a socket
      mSocket = socket(PF_INET,SOCK_STREAM,0);
      if (mSocket < 0) 
         return -1;
   
    //----------------------------------- Set the buffer size
      if (setsockopt (mSocket, SOL_SOCKET, SO_RCVBUF, 
                     (char*) &mRcvBuffer, sizeof (int)) != 0) {
         if (mDebug) {
            cerr << "set socket buffer failed for length " << mRcvBuffer << endl;
         }
      }
   
    //----------------------------------  Bind the socket to a port.
      socknam.sin_family      = AF_INET;
      socknam.sin_port        = 0;
      socknam.sin_addr.s_addr = 0;
      int len                 = sizeof(socknam);
      if (::bind(mSocket, (struct sockaddr *)&socknam, len) < 0) 
         return -1;
   
    //----------------------------------  Connect to the server.
      socknam.sin_family      = AF_INET;
      socknam.sin_port        = htons (ipport);
      if (nslookup (ipaddr, &socknam.sin_addr) < 0) {
         return -1;
      }
      wait_time timeout = _TIMEOUT/1000000.0;	/* timeout */
      if (connectWithTimeout (mSocket, (struct sockaddr *)&socknam, 
                           sizeof (socknam), timeout) < 0)
         return -1;
      mOpened = true;
   
    //----------------------------------  Get the server version number
      mVersion = 0;
      mRevision = 0;
      int rc = SendRequest("version;", version, 4, &size, timeout);
      if (rc || (size != 4)) {
         ::close(mSocket);
         mOpened = false;
         return rc ? rc : -1;
      }
      mVersion = CVHex(version, 4);
      rc = SendRequest("revision;", version, 4, &size, timeout);
      if (rc || (size != 4)) {
         ::close(mSocket); 
         mOpened = false;
         return rc ? rc : -1;
      }
      mRevision = CVHex(version, 4);
      if (mDebug) cerr << "Connected to server version " << Version() << endl;
   
    //----------------------------------  Done, Return.
      return rc;
   }

//--------------------------------------  Close a socket
   void DAQSocket::close() 
   {
      semlock lockit (mux);
      if (mOpened) {
         StopWriter();
         SendRequest("quit;");
         ::close(mSocket);
         mOpened = false;
      }
      mChannel.clear();
      mWriterType = NoWriter;
   }

//--------------------------------------  flush data from socket
   void DAQSocket::flush () {
      semlock lockit (mux);
      const int buflen = 16 * 1024;
      char text[buflen];
   
      int i = 0;
      int rc;
      do {
         rc = recv (mSocket, text, buflen, MSG_DONTWAIT);
      } while ((rc >= buflen) && (++i < 100));
   }

//--------------------------------------  Stop data transfer
   int DAQSocket::StopWriter() {
      semlock lockit (mux);
   
    //----------------------------------  Make sure a writer is running.
      if (mWriterType == NoWriter) 
         return -1;
   
    //----------------------------------  Build the request
      ostringstream request;
      request << "kill net-writer " << CVHex (mWriter, 8) << ";";
      request.put(0);
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.str().c_str(), mWriter, 0);
      mWriterType = NoWriter;
   
      return rc;
   }

//--------------------------------------  Request a stream of frame data
   int DAQSocket::RequestFrames() {
      semlock lockit (mux);
   
    //----------------------------------  Build the request
      ostringstream request;
      request << "start frame-writer ";
      if (mGetAll) {
         request << "all;";
      } 
      else {
         request << "{";
         for (Channel_iter i = mChannel.begin() ; i != mChannel.end() ; i++) {
            request << "\"" << i->first << "\"";
         }
         request << "};";
      }
      request.put(0);
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.str().c_str(), mWriter, sizeof(mWriter));
      if (rc) 
         return rc;
      mWriterType = FrameWriter;
   
    //----------------------------------  Read in the offline flag
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false);
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Done, return
      return rc;
   }

//--------------------------------------  Request file names
   int DAQSocket::RequestNames (long timeout) {
      semlock lockit (mux);
      int rc = SendRequest("start name-writer all;", mWriter, sizeof(mWriter),
                          0, timeout);
      if (rc) 
         return rc;
      mWriterType = NameWriter;
   
    //----------------------------------  Read in the offlne flag
      rc = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (rc != sizeof(mOffline)) 
         return -1;
      return 0;
   }

//--------------------------------------  Request a stream of channel data
   int DAQSocket::RequestOnlineData (bool fast, long timeout) 
   {
      semlock lockit (mux);

    //----------------------------------  Build the request
      ostringstream request;
      if (fast) {
         request << "start fast-writer ";
      }
      else {
         request << "start net-writer ";
      }
      if (mGetAll) {
         request << "all;";
      } 
      else {
         request << "{";
         for (Channel_iter i = mChannel.begin() ; i != mChannel.end() ; i++) {
            request << "\"" << i->first << "\"";
         }
         request << "};";
      }
      request.put(0);
      if (mDebug)
         cerr << "NDS request = " << request.str() << endl;
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.str().c_str(), mWriter, sizeof(mWriter),
                          0, timeout);
      if (mDebug) cerr << string(mWriter, 8) << " = " 
		       << CVHex (mWriter, 8) << endl;
      if (rc) {
         return rc;
      }
      mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Done, return
      return rc;
   }

//--------------------------------------  Request a stream of channel data
   int DAQSocket::RequestData (unsigned long start, unsigned long duration,
                     long timeout)
   {
      semlock lockit (mux);

    //----------------------------------  Build the request
      ostringstream request;
      request << "start net-writer " << start << " " << duration << " ";
      if (mGetAll) {
         request << "all;";
      } 
      else {
         request << "{";
         for (Channel_iter i = mChannel.begin() ; i != mChannel.end() ; i++) {
            request << "\"" << i->first << "\"";
         }
         request << "};";
      }
      request.put(0);
      if (mDebug)
         cerr << "NDS past data request = " << request.str() << endl;
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.str().c_str(), mWriter, sizeof(mWriter),
                          0, timeout);
      if (mDebug) cerr << string(mWriter, 8) << " = " 
		       << CVHex (mWriter, 8) << endl;
      if (rc) {
         return rc;
      }
      mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (mDebug) cerr << mOffline << endl;
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Done, return
      return rc;
   }

//--------------------------------------  Request trend data
   int DAQSocket::RequestTrend (unsigned long start, unsigned long duration,
                     bool mintrend, long timeout)
   {
      semlock lockit (mux);

    //----------------------------------  Build the request
      ostringstream request;
      request << "start trend " << (mintrend ? "60 " : "") << 
         "net-writer " << start << " " << duration << " ";
      if (mGetAll) {
         request << "all;";
      } 
      else {
         request << "{";
         for (Channel_iter i = mChannel.begin() ; i != mChannel.end() ; i++) {
            request << "\"" << i->first << "\"";
         }
         request << "};";
      }
      request.put(0);
      if (mDebug) 
         cerr << "NDS trend data request = " << request.str() << endl;
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.str().c_str(), mWriter, sizeof(mWriter),
                          0, timeout);
      if (mDebug) cerr << string(mWriter, 8) << " = " 
		       << CVHex (mWriter, 8) << endl;
      if (rc) {
         return rc;
      }
      mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Done, return
      return rc;
   }

//--------------------------------------  Wait for a message to arrive
//                                        WaitforData() <= 0 means no data.
   int DAQSocket::WaitforData (bool poll) 
   {
      wait_time t = poll? 0 : -1;
      int nset = socketWait(mSocket, t, wm_read);
      if (nset < 0) perror("DAQSocket: Error in select()");
      return nset;
   }

//--------------------------------------  Add a channel to the channel list
   int 
   DAQSocket::AddChannel(const DAQDChannel& chn) {
      semlock lockit (mux);
      mGetAll = false;
   
    //----------------------------------  Add the channel.
      mChannel.insert (channellist::value_type (chn.mName, chn));
      return 1;
   }

//--------------------------------------  Add a channel to the channel list
   int 
   DAQSocket::AddChannel(const char* chan, rate_bps_pair rb) {
      semlock lockit (mux);
    //----------------------------------  Set the all flag if all specified.
      if (string(chan) == "all") {
         mGetAll = true;
         mChannel.clear();
      }
      if (mGetAll) 
         return 1;
      DAQDChannel chn;
      strncpy (chn.mName, chan, sizeof (chn.mName));
      chn.mName[sizeof (chn.mName)-1] = 0;
      chn.mRate = rb.first;
      chn.mBPS = rb.second;
    //----------------------------------  Add the channel.
      return AddChannel (chn);
   }

//--------------------------------------  Remove a channel from the list
   void 
   DAQSocket::RmChannel(const char* chan) 
   {
      semlock lockit (mux);
      string Channel(chan);
      if (Channel == "all") {
         mGetAll = false;
         mChannel.clear();
      }
      else {
         Channel_iter iter = mChannel.find (Channel);
         if (iter != mChannel.end()) {
            mChannel.erase (iter);
         }
      }
   }

//--------------------------------------  Receive frame data into a buffer
   long
   DAQSocket::GetFrame(char *buffer, long length) {
      semlock lockit (mux);
      if (mWriterType != FrameWriter) 
         return -1;
      return RecvData(buffer, length);
   }

//--------------------------------------  Receive frame data into a buffer
   long 
   DAQSocket::GetName(char *buffer, long length) {
      semlock lockit (mux);
      if (mWriterType != NameWriter) 
         return -1;
      return RecvData(buffer, length);
   }

//--------------------------------------  Receive proprietary data into buffer
   int 
   DAQSocket::GetData(char *buffer, long length, long timeout) {
      semlock lockit (mux);
      if (mWriterType != DataWriter) 
         return -1;
      char* bufpt = buffer + sizeof(DAQDRecHdr);
      int nbyt = RecvData(bufpt, length-sizeof(DAQDRecHdr), 
                         (DAQDRecHdr*)buffer, timeout);
      if (nbyt <= 0) 
         return nbyt;
      return nbyt;
   }

   int DAQSocket::GetData (char** buffer, long timeout) {
      semlock lockit (mux);
      *buffer = 0;
      if (mWriterType != DataWriter) 
         return -10;
      int nbyt = RecvData (buffer, timeout);
      if (nbyt <= 0) {
         return nbyt;
      }
      return nbyt;
   }

//--------------------------------------  Receive proprietary data into buffer
   int 
   DAQSocket::RecvData(char* buffer, long length, DAQDRecHdr* hdr,
                     long timeout) {
      DAQDRecHdr header, *h;
   
    //----------------------------------  Set header pointer.
      if (hdr) {
         h = hdr;
      } 
      else {
         h = &header;
      }
   
    //----------------------------------  Read in the data header. Calculate
    //                                    the record length.
      int rc = RecvRec((char*) h, sizeof(DAQDRecHdr), true, timeout);
      if (rc != sizeof(DAQDRecHdr)) 
         return -1;
      if (mReorder) SwapI((int*) h, sizeof(DAQDRecHdr)/sizeof(int));
      if (mDebug) {
         cerr << "Record Header: BLen=" << h->Blen << " Secs=" << h->Secs 
            << " GPS=" << h->GPS << " NSec=" << h->NSec << " SeqNum=" 
            << h->SeqNum << endl;
      }
      int ndata = h->Blen - (sizeof(DAQDRecHdr) - sizeof(int));
      if (ndata == 0) 
         return 0;
      if (ndata  < 0) 
         return -1;
      if (ndata > length) {
         cerr << "DAQSocket::RecvData - Buffer length (" << length 
            << " bytes) is too small for record (" << ndata 
            << " bytes)." << endl;
         return -1;
      }
   
    //----------------------------------  Read in the data body
      rc = RecvRec(buffer, ndata, true, timeout);
      return rc;
   }

   int 
   DAQSocket::RecvData (char** buffer, long timeout) 
   {
      DAQDRecHdr h;
      memset (&h, 0, sizeof(DAQDRecHdr));
      *buffer = 0;
   
    //----------------------------------  Read in the data header. Calculate
    //                                    the record length.
      int rc = RecvRec((char*) &h, sizeof(int), true, timeout);
      if (rc != sizeof(int)) { 
         return -2;
      }
      if (mReorder) SwapI((int*) &h, sizeof(int));
      int min = h.Blen < (int) (sizeof(DAQDRecHdr) - sizeof(int)) ? 
         h.Blen : sizeof(DAQDRecHdr) - sizeof(int);
      if (min > 0) {
         rc = RecvRec((char*) &h.Secs, min, true, timeout);
         if (rc != min) { 
            return -3;
         }
      }
      if (mReorder) SwapI((int*) &h.Secs, sizeof(DAQDRecHdr)/sizeof(int)-1);
      if (mDebug) {
         cerr << "Record Header: BLen=" << h.Blen << " Secs=" << h.Secs 
            << " GPS=" << h.GPS << " NSec=" << h.NSec << " SeqNum=" 
            << h.SeqNum << endl;
      }
      int ndata = h.Blen - (int)(sizeof(DAQDRecHdr) - sizeof(int));
      if (ndata <= 0) {
         ndata = 0;
      }
      *buffer = new (nothrow) char [ndata + sizeof(DAQDRecHdr)];
      if (*buffer == 0) {
         return -4;
      }
      memcpy (*buffer, &h, sizeof(DAQDRecHdr));
      if (ndata == 0) 
         return 0;
   
    //----------------------------------  Read in the data body
      rc = RecvRec (*buffer + sizeof(DAQDRecHdr), ndata, true, timeout);
      return rc;
   }

//--------------------------------------  Get a list of available channels
   int 
   DAQSocket::Available (std::vector<DAQDChannel>& list) 
   {
      semlock lockit (mux);
      char buf[1024];
      bool extendedList = ((mVersion > 11) || 
                          ((mVersion == 11) && (mRevision >= 3)));
      bool longNames = (mVersion >= 12) ;	// longer names introduced in version 12.
      int  nameLength = 40;			// Either 40 or MAX_CHNNAME_SIZE
   
    //----------------------------------  Request a list of channels.
      int rc = 0;
      if (extendedList) {
         rc = SendRequest("status channels 2;", buf, 8);
      }
      else {
         rc = SendRequest("status channels;", buf, 4);
      }
      if (rc) 
         return -1;
  
    //----------------------------------  Get the number of channels.
      int nw= extendedList ? CVHex(buf, 8) : CVHex(buf, 4);
      int recsz = 52;
      if ((mVersion == 9) || (mVersion == 10)) recsz = 60;
      if (mVersion >= 11) recsz = 124;
      if (extendedList) recsz = 128;
      if (longNames) 
      {
	 recsz = 128 + (MAX_CHNNAME_SIZE - 40) ;
	 nameLength = MAX_CHNNAME_SIZE ;
      }

    //----------------------------------  Skip DAQ data rate.
      if (!extendedList) rc = RecvRec(buf, 4, true);
   
    //----------------------------------  Fill in the list (test)
      list.clear();
      DAQDChannel chn;
      for (int i = 0 ; i < nw ; i++) {
         rc = RecvRec(buf, recsz, true);
         if (rc <recsz) 
            return -1;
         memcpy(chn.mName, buf, nameLength);
	 //----------------------------- Note: the current mName length > 40
	 chn.mName[nameLength] = 0;
	 // Change all trailing spaces in the name field to 0.
         for (int j=(nameLength-1) ; j>=0 && isspace(chn.mName[j]) ; j--) {
            chn.mName[j] = 0;
         }
         if (extendedList) {
	    int2float i2f;

            chn.mRate  = CVHex (buf + nameLength, 8);
            chn.mNum = CVHex (buf + nameLength + 8, 8);
            chn.mGroup = CVHex (buf + nameLength + 16, 4);
            /*chn.mBPS = CVHex (buf + 56, 4);*/
            chn.mDatatype = CVHex (buf + nameLength + 20, 4);
	    i2f.i       = CVHex(buf + nameLength + 24, 8);
	    chn.mGain   = i2f.f;
	    i2f.i       = CVHex(buf + nameLength + 32, 8);
	    chn.mSlope  = i2f.f;
	    i2f.i       = CVHex(buf + nameLength + 40, 8);
	    chn.mOffset = i2f.f;
            memcpy(chn.mUnit, buf + nameLength + 48, 40);     
	    //---------------------------- don't allow >39 digit units
	    chn.mUnit[39] = 0;
	    for (int j=38; j>=0 && isspace(chn.mUnit[j]) ; j--) {
               chn.mUnit[j] = 0;
            }
	 }
	 else {
            chn.mRate  = CVHex(buf+40, 4);
            chn.mNum = CVHex(buf+44, 4);
            chn.mGroup = CVHex(buf+48, 4);
            if (recsz > 52) {
               chn.mBPS = CVHex(buf+52, 4);
               chn.mDatatype = CVHex(buf+56, 4);
            } 
            else {
               chn.mBPS = 0;
               chn.mDatatype = 0;
            }
            if (recsz > 60) {
	       int2float i2f;
	       i2f.i       = CVHex(buf+60, 8);
	       chn.mGain   = i2f.f;
	       i2f.i       = CVHex(buf+68, 8);
	       chn.mSlope  = i2f.f;
	       i2f.i       = CVHex(buf+76, 8);
	       chn.mOffset = i2f.f;
               memcpy(chn.mUnit, buf+84, 40);
               for (int j=39; j>=0 && isspace(chn.mUnit[j]) ; j--) {
        	  chn.mUnit[j] = 0;
               }
            }
            else {
               chn.mGain = 0;
               chn.mSlope = 0;
               chn.mOffset = 0;
               strcpy (chn.mUnit, "");
            }
	 }
         list.push_back (chn);
      }
      return nw;
   }

//--------------------------------------  Get a list of available channels
   int 
   DAQSocket::Available(DAQDChannel list[], long N) {
      std::vector<DAQDChannel> l;
      int rc = Available (l);
      if (rc < 0) {
         return rc;
      }
      for (int i = 0; (i < N) && (i < rc); ++i) {
         list[i] = l[i];
      }
      return rc;
   }

//--------------------------------------  Get a list of available channels
   int 
   DAQSocket::Times (unsigned long& start, unsigned long& duration, 
                    long timeout) 
   {
      semlock lockit (mux);
    //----------------------------------  Build the request
      string request = "status main filesys;";
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.c_str(), mWriter, sizeof(mWriter),
                          0, timeout);
      if (mDebug) cerr << string(mWriter, 8) << " = " 
		       << CVHex (mWriter, 8) << endl;
      if (rc) {
         return rc;
      }
      mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (mDebug) cerr << mOffline << endl;
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Get an empty data block
      DAQDRecHdr hdr;
      rc = RecvData(0, 0, &hdr, timeout);
      if (rc) {
         if (mDebug) cerr << "times failed" << rc << endl;
         return rc;
      }
      start = hdr.GPS;
      duration = hdr.Secs;
    //----------------------------------  Done, return
      return rc;
   
   }

//--------------------------------------  Get a list of available channels
   int 
   DAQSocket::TimesTrend (unsigned long& start, unsigned long& duration, 
                     bool mintrend, long timeout) 
   {
      semlock lockit (mux);

    //----------------------------------  Build the request
      string request = mintrend ? 
         "status minute-trend filesys;" : "status trend filesys;";
   
    //----------------------------------  Send the request
      int rc = SendRequest(request.c_str(), mWriter, sizeof(mWriter),
                          0, timeout);
      if (mDebug) cerr << string(mWriter, 8) << " = " 
		       << CVHex (mWriter, 8) << endl;
      if (rc) {
         return rc;
      }
      mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
      int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
      if (mDebug) cerr << mOffline << endl;
      if (ldata != sizeof(mOffline)) 
         return ldata;
   
    //----------------------------------  Get an empty data block
      DAQDRecHdr hdr;
      rc = RecvData(0, 0, &hdr, timeout);
      if (rc) {
         return rc;
      }
      start = hdr.GPS;
      duration = hdr.Secs;
    //----------------------------------  Done, return
      return rc;
   
   }

//--------------------------------------  Send a request, wait for response.
   int
   DAQSocket::SendRequest(const char* text, char *reply, long length, 
                     long *Size, wait_time maxwait) {

    //----------------------------------  Send the request
      if (mDebug) cerr << "Request: " << text << endl;
      int rc = SendRec (text, strlen(text), maxwait);
      if (rc <= 0) {
         if (mDebug) cerr << "send ret1 = " << rc << endl; 
         return rc;
      }
   
    //----------------------------------  Return if no reply expected.
      if (reply == 0) 
         return 0;
   
    //----------------------------------  Read the reply status.
      const int stat_len=4;
      char status[stat_len];
      rc = RecvRec (status, stat_len, true, maxwait);
      if (rc != 4) {
         if (mDebug) cerr << "send ret2 = " << rc << endl; 
         return -1;
      }
      if (mDebug) cerr << "Status: " << string(status, stat_len) << endl;
      rc = CVHex(status, stat_len);
      if (rc) 
         return rc;
   
    //----------------------------------  Read the reply text.
      if (length != 0) {
         rc = RecvRec(reply, length, true, maxwait);
         if (rc < 0) {
            if (mDebug) cerr << "send ret3 = " << rc << endl; 
            return rc;
         }
         if (rc < length) reply[rc] = 0;
         if (mDebug) cerr << "reply: " << string(reply, rc) << endl;
         if (Size)       *Size = rc;
      }
      return 0;
   }

//--------------------------------------  Receive data into a buffer
   int 
   DAQSocket::RecvRec(char *buffer, long length, bool readall, wait_time maxwait) {
      Time stop;
      int  fileflags = fcntl (mSocket, F_GETFL, 0);
      if (fileflags == -1) {
	 return -1;
      }
      char* point = buffer;
      int   nRead = 0;
      int poll = (maxwait > 0);
      if (poll) {
         stop = Now() + Interval (maxwait);
      }
      bool timedout = false;
      do {
         // use select to test for timeout
         if (poll || mAbort) {
	    wait_time timeout = mAbort? 0.1 : maxwait;
	    if (mDebug) cerr << "DAQSocket::RecvRec wait time=" 
			     << timeout << endl;
            int nset = socketWait(mSocket, timeout, wm_read);
            //cerr << "select returns with " << nset << endl;
            if (nset < 0) {
               perror("DAQSocket: Error in select()");
               return -12;
            }
            if (nset == 0) {
               // timeout?
               if (errno == 0) {
                  if (!mAbort || *mAbort) {
                     return -13;
                  }
               }
               // signal
               else {
                  cerr << "Signal received in select, errno = " << errno << " ++++++++++++++++++++++++++++++++++++++++++++" << endl;
                  continue;
               }
            }
            // compute how much time is left in timeout
            if (poll) {
               Interval diff = stop - Now();
               if (diff <= Interval (0.)) {
                  maxwait = 0;
                  timedout = true;
               }
               else {
                  maxwait = diff;
               }
            }
            // continue if no data and abort not set
            if ((nset == 0) && mAbort && !timedout) {
               continue;
            }
            /* set socket to non blocking */
            if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
               return -1;
            }
         }
         int nB = recv(mSocket, point, length - nRead, 0);
         //cerr << "recv returns with " << nB << endl;
         if (poll || mAbort) {
            if (mDebug && (nB == 0)) { 
               cerr << "RecvRec with zero length" << endl;
            }
            fcntl (mSocket, F_SETFL, fileflags & ~O_NONBLOCK);
         }
         if (nB == -1) {
            if (mDebug)
               cerr << "RecvRec failed with errno " << errno << endl;
            return -10;
         }
         point += nB;
         nRead += nB;
         if (timedout || (mAbort && *mAbort)) {
            return -13;
         }
      } while (readall && (nRead < length));
      if (mDebug) cerr << "RecvRec read " << nRead << "/" << length << endl;
      return nRead;
   }


//--------------------------------------  Send data from a buffer
   int 
   DAQSocket::SendRec(const char *buffer, long length, wait_time maxwait) {
      int 	fileflags;
      const char* point = buffer;
      int   nWrite = 0;
      int poll = (maxwait >= 0);
   
      bool timedout = false;
      if (mDebug) cerr << "SendRec send " << buffer << endl ;
      do {
         // use select to test for timeout
         if (poll || mAbort) {
	    wait_time timeout = mAbort? 0.1: maxwait;
            Time now = Now();
            // wait
            int nset  = socketWait(mSocket, timeout, wm_write);
            if (nset < 0) {
               perror("DAQSocket: Error in select()");
               return -12;
            }
            else if (nset == 0) {
               // timeout error
               if (!mAbort || *mAbort) {
                  return -13;
               }
            }
            // compute how much time is left in timeout
            if (poll) {
               Interval left = now + Interval(maxwait) - Now();
               if (left < Interval (0.)) {
                  maxwait  = 0;
                  timedout = true;
               }
               else {
                  maxwait = left;
               }
            }
            // continue if no data sent and abort not set
            if ((nset == 0) && mAbort && !timedout) {
               continue;
            }
            /* set socket to non blocking */
            if ((fileflags = fcntl (mSocket, F_GETFL, 0)) == -1) {
               return -1;
            }
            if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
               return -1;
            }
         }
         int nB = send (mSocket, point, length - nWrite, 0);
         if (poll || mAbort) {
            fcntl (mSocket, F_SETFL, fileflags & !O_NONBLOCK);
         }
         if (nB == -1) {
            if (mDebug)
               cerr << "SendRec failed with errno " << errno << endl;
            return -10;
         }
         point += nB;
         nWrite += nB;
         if (timedout || (mAbort && *mAbort)) {
            return -13;
         }
      } while (nWrite < length);
      if (mDebug) cerr << "SendRec write " << nWrite << "/" << length << endl;
      return nWrite;
   }
