/* Version $Id: awgexc_run.c 1242 2011-06-07 18:45:15Z james.batch@LIGO.ORG $ */
/*=============================================================================
awgexc_run - modification of awgstream for use as a back-end for Matlab
  The command line usage is
    awgexc_run <start time> <is fork> <config file>

  <start time> is the GPS start time of the excitation.  It must be at least
     a 3 seconds from now.

  <is fork> is a logical flag.  If true, awgexc_run with fork to the
     background before starting the excitation.

  <config file> is a configuration file generated by awgexc_conf

-- Original awgstream command line tool history:
Written Oct 2001 by Peter Shawhan
Modified June 2002 to call SIStrAppInfo
Modified 11 Feb 2003 to print out start time, etc. (unless '-q' is specified)
Modified 11 Feb 2003 to inject calibration lines with S2 freqs and amplitudes
Modified 24 Feb 2005 by Vuk Mandic to make calibration lines optional with '-c'

-- awgexc_run
Written Jan 8 2008 by Matt Evans
Modified 14 Sep 2017 K. Thorne return correct error from append_data

Then compile this with the Makefile in this directory.

=============================================================================*/

#define CLIENT "awgexc"
#define DEBUG 0
#define BUF_LENGTH 256 /* must match awgexc_conf */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "dtt/SIStr.h"
#include "tconv.h" /* for TAInow and TAIsec */


/*===========================================================================
  dTAInow returns the current GPS time as a double.

  This would be better located in tconv.c, but rather than try to
  change things that aren't mine, I put it here.
  =========================================================================*/
double dTAInow(void)
{
  tai_t tai;
  TAIsec(TAInow(), &tai);
  
  return tai.tai + tai.nsec / 1.0e9;
}


/*===========================================================================
  This is just an interface to SIStrAppend with error checking.
  Normally this would appear in main, but it is used several times,
  so I made it a function to avoid code duplication.
  ===========================================================================*/
int append_data(SIStream* sis, float* data, int ndata)
{
  int status;

  status = SIStrAppend(sis, data, ndata, 1.0);
# if DEBUG 
    printf( "SIStrAppend returned %d\n", status );
# endif
  if( status != SIStr_OK )
  {
    printf("Error while adding data to stream: %s\n", SIStrErrorMsg(status));

    /* try to close */
    int closeStatus;
    closeStatus = SIStrClose(sis);
# if DEBUG 
    printf("SIStrClose returned %d\n", closeStatus);
# endif
    if( closeStatus != SIStr_OK )
    {
        printf("Error while closing SIStream: %s\n", SIStrErrorMsg(status));
    }
  }

  return status;
}

/*===========================================================================
  ===========================================================================*/
int main(int argc, char** argv)
{
  /* config file */
  FILE* fid;
  int len;

  /* config data */
  char channel[BUF_LENGTH];
  int samprate;
  double starttime;
  int nrep;
  int nramp;
  int isfork;
  int ndata;

  /* fork return value */
  int childpid;

  /* SIStr interaction */
  SIStream sis;
  char info[256];
  int status;
  double time_now;

  /* data copying temps */
  int i, j;
  float* fdata;
  float* fbuf;
  double dphi;

  /* ============ Setup ============ */
  /* parse command line*/
  if( argc != 4 )
  {
    printf("Usage: awgexc_run <start time> <is fork> <config file>\n");
    printf("\nawgexc_run version %s\n$Id: awgexc_run.c 1242 2011-06-07 18:45:15Z james.batch@LIGO.ORG $\n", VERSION) ;
    return 1;
  }

  /* start time and fork flag */
  starttime = atof(argv[1]);
  isfork = atoi(argv[2]);

  /* open the config file */
  fid = fopen(argv[3], "r");
  if( fid == NULL )
  {
    printf("Unable to open configuration file \"%s\"\n", argv[1]);
    return 2;
  }

  /* read channel name, parameters and data size */
  if( fread(&len, sizeof(int), 1, fid) != 1 ||
      fread(channel, sizeof(char), len, fid) != len ||
      fread(&samprate, sizeof(int), 1, fid) != 1 ||
      fread(&nrep, sizeof(int), 1, fid) != 1 ||
      fread(&nramp, sizeof(int), 1, fid) != 1 ||
      fread(&ndata, sizeof(int), 1, fid) != 1 )
  {
    printf("Error while reading configuration parameters.\n");
    return 3;
  }

  /* null terminate channel name */
  channel[len] = '\0';

  /* read data */
  fdata = calloc(ndata, sizeof(float));
  fbuf = calloc(ndata, sizeof(float));
  if( fdata == NULL || fbuf == NULL )
  {
    printf("Unable to allocate memory for data.\n");
    return 4;
  }

  len = fread(fdata, sizeof(float), ndata, fid);
  if( len != ndata )
  {
    printf("Error while reading data (%d of %d read).\n",
            len, ndata);
    return 5;
  }

  /*-----------------------------------*/
  /* Check for obvious parameter errors */

  /* check start time */
  time_now = dTAInow();
  if( starttime < time_now + 3.0 )
  {
    printf("Start time too soon (now = %.6f, start = %.6f).\n",
            time_now, starttime);
    return 6;
  }

  /* check channel name */
  len = strlen(channel);
  if( len > SIStr_MAXCHANNAMELENGTH )
  {
    printf("Channel name too long (is = %d, max = %d).\n",
            len, SIStr_MAXCHANNAMELENGTH);
    return 6;
  }
  
  if( len < 3 )
  {
    printf("Channel name too short (is = %d, min = %d).\n", len, 3);
    return 6;
  }

  if( (nrep + nramp) < 1 )
  {
    printf("Nothing to do (nrep = %d, nramp = %d).\n", nrep, nramp);
    return 6;
  }

  if( ndata < 1 )
  {
    printf("No data (ndata = %d).\n", ndata);
    return 6;
  }

  /*-----------------------------------*/
  /* print parameter info */
  printf("%s at %d Hz\n", channel, samprate);
  printf("  now gps   %17.6f\n", time_now);
  printf("  start gps %17.6f\n", starttime);
  printf("  nrep %d, nramp %d, ndata %d\n", nrep, nramp, ndata);

  /* ============ Fork ============ */
  /* fork and let parent return */
  if( isfork )
  {
    childpid = fork();
    if( childpid < 0 )
    {
      printf("Unable to fork (error %d).\n", childpid);
      return 7;
    }

    /* if this is the parent, just return */
    if( childpid > 0 )
    {
      printf("Forked excitation to background.\n");
      return 0;
    }
  }

  /* ============ SIStr Interaction ============ */

  /*-----------------------------------*/
  /* Report some information about this application and waveform */
  sprintf(info, "%s from Matlab", CLIENT);
  SIStrAppInfo(info);

  /*-----------------------------------*/
  /* Open the Signal Injection Stream */
  status = SIStrOpen(&sis, channel, samprate, starttime);
# if DEBUG 
    printf( "SIStrOpen returned %d\n", status );
# endif

  if( status != SIStr_OK )
  {
    printf("Error while opening SIStream: %s\n", SIStrErrorMsg(status));
    return 10;
  }

  /*-----------------------------------*/
  /* Append data */

  /* allocate buffer */
  fbuf = calloc(ndata, sizeof(float));
  if( fdata == NULL || fbuf == NULL )
  {
    printf("Unable to allocate memory for data.\n");
    return 4;
  }

  /* compute ramp phase per sample */
  if( nramp > 0 )
    dphi = M_PI / (nramp * ndata - 1);
  else
    dphi = 0.0;

  int appStatus;
  /* ramp up */
  for( i = 0; i < nramp; ++i )
  {
    for( j = 0; j < ndata; ++j )
      fbuf[j] = fdata[j] * 0.5 * (1.0 - cos((j + i * ndata) * dphi));
      
    appStatus = append_data(&sis, fbuf, ndata);
    if(appStatus != SIStr_OK)
    {
      printf("append_data failed.\n");
      return 8;
    }
  }

  /* repetitions */
  for( i = 0; i < nrep; ++i )
    appStatus = append_data(&sis, fdata, ndata);
    if( appStatus != SIStr_OK)
    {
      printf("append_data failed.\n");
      return 8;
    }

  /* ramp down */
  for( i = 0; i < nramp; ++i )
  {
    for( j = 0; j < ndata; ++j )
      fbuf[j] = fdata[j] * 0.5 * (1.0 + cos((j + i * ndata) * dphi));

    appStatus = append_data(&sis, fbuf, ndata);
    if( appStatus != SIStr_OK)
    {
      printf("append_data failed.\n");
      return 8;
    }
  }

  /*-----------------------------------*/
  /* Close the stream */
  status = SIStrClose(&sis);
# if DEBUG 
    printf( "SIStrClose returned %d\n", status );
# endif

  if( status != SIStr_OK )
  {
    printf("Error while closing SIStream: %s\n", SIStrErrorMsg(status));
    return 9;
  }

  /*-----------------------------------*/
  /* done, return success */
  return 0;
}
