// Yo, Emacs! This is -*- C++ -*-
//
// $Id: Scanner.cc 6344 2010-11-10 06:19:42Z john.zweizig@LIGO.ORG $
//
// Author: David Chin <dwchin@umich.edu>
//
//         2477 Randall Lab, Univ. of Michigan
//         500 E. University Ave.
//         Ann Arbor, MI 48109
//         +1-734-764-5146
//         +1-734-730-1274
//
// Written based on idea by K. Riles <kriles@umich.edu>

#include <time.h>
#include "osc/boolean/Scanner.hh"

namespace osc
{
    namespace boolean
    {
        /// Maps a character to its code
        CharCode charCodeMap[128];
    }
}

using namespace std;

osc::boolean::TextScanner::TextScanner(const string &buffer)
{
    pmBuffer = new InfixStrBuffer(buffer);

    //
    // Initialize the character code map
    //
    for (int i = 0; i <= 127; ++i)
        charCodeMap[i] = ccError;

    for (int i = 'a'; i <= 'z'; ++i)
        charCodeMap[i] = ccLetter;

    for (int i = 'A'; i <= 'Z'; ++i)
        charCodeMap[i] = ccLetter;

    for (int i = '0'; i <= '9'; ++i)
        charCodeMap[i] = ccDigit;

    charCodeMap[int('_')] = ccUnderScore;
    charCodeMap[int(':')] = ccColon;
    charCodeMap[int('!')] = ccUnary;
    charCodeMap[int('&')] = charCodeMap[int('|')] = ccBinary;
    charCodeMap[int('(')] = charCodeMap[int(')')] = ccSpecial;
    charCodeMap[int(' ')] = charCodeMap[int('\t')] = ccWhiteSpace;
    charCodeMap[int(nullChar)] = ccEndOfString;
}


//
// skipWhiteSpace()
// Repeatedly fetch characters from the text input as long as they're
// white space.
//
void osc::boolean::TextScanner::skipWhiteSpace()
{
    int ch = pmBuffer->curChar();

    while (charCodeMap[ch] == ccWhiteSpace)
        ch = pmBuffer->nextChar();
}

//
// get()
// Extract the next token from the infix string, based on the current
// character
//
osc::boolean::Token *osc::boolean::TextScanner::get()
{
    Token *pToken;  // ptr to token to return

    skipWhiteSpace();

    // Determine the token class, based on the current character
    switch (charCodeMap[int(pmBuffer->curChar())]) {
    case ccLetter:
        pToken = new WordToken;
        break;

    case ccUnary:
        pToken = new UnaryToken;
        break;

    case ccBinary:
        pToken = new BinaryToken;
        break;

    case ccSpecial:
        pToken = new SpecialToken;
        break;

    case ccEndOfString:
        pToken = new EOSToken;
        break;

    default:
        pToken = new ErrorToken;
        break;
    }

    // Extract a token of that class, and return a pointer to it
    pToken->get(pmBuffer);
    return pToken;
}

