//
//   Replay a previously recorded strip chart
//   The strip-chart file recorded by Chart.cc is replayed on the screen
//   Some additional control should be added to:
//     a) Keep the display from vanishing after Replay terminates
//     b) Allow the user to select a specific time interval to be displayed.
//
class Strip {
public:
  Strip() {}
  ~Strip() {}
  float mFlow;
  float mFhi;
  char mName[40];
};

void
Replay(const char* config) {
    int nPens = 0;
    int nStep = 0;
    float sum[16];
    float dTime=1.0;
    char line[128], Title[128];

    //---------------------------------  Check file name
    if (!config || !*config) {
        cerr<< "Chart: configuration file not specified" << endl;
	return;
    }

    //---------------------------------  Get parameters.
    ifstream ifs(config, ios::in);
    if (!ifs) {
        cerr<< "Replay: Unable to open file " << config << endl;
	return;
    }

    //---------------------------------  Get the global parameters
    ifs.getline(line, 128);
    ifs.getline(line, 128);
    strcpy(Title, line+strlen("# Title: "));
    ifs.getline(line, 128);
    sscanf(line,"# NPens: %i Sample interval: %f Step Count: %i", 
	   &nPens, &dTime, &nStep);
    if (nPens <= 0 || nPens > 16) {
        cerr << "Replay: Invalid log file (bad pen count)" << endl;
	return;
    }
    if (nStep <= 0) nStep = 1000;

    //---------------------------------  Read in the pen descriptors.
    Strip stripv[16];
    for (int i=0 ; i<nPens ; i++) {
	int id;
	ifs.getline(line, 128);
	sscanf(line, "# Pen: %i Channel: %s Low Freq: %f High Freq: %f",
	       &id, &stripv[i].mName, &stripv[i].mFlow, &stripv[i].mFhi);
    }

    //---------------------------------  Create the strip chart.
    StripChart sc(nPens, Title);
    sc.setN(nStep);
    sc.setLogY(true);
    for (int p=0 ; p<nPens ; p++) {
        if (stripv[p]->mFlow >= 1.0) {
	    sprintf(line, "%s\n%.0f-%.0f Hz", stripv[p].mName, 
		    stripv[p].mFlow, stripv[p].mFhi);
	} else {
	    sprintf(line, "%s\n%.2f-%.2f Hz", stripv[p].mName, 
		    stripv[p].mFlow, stripv[p].mFhi);
	}
	sc.setTitle(p, line);
    }
    sc.setAxisType(StripChart::kLocalTime);

    //---------------------------------  Loop over sampling intervals
    bool first = true;
    unsigned long x;
    while (!ifs.eof()) {
        ifs >> x >> sum[0];
	for (int i=1 ; i<nPens ; i++) ifs >> sum[i];
	double tbin = getUTC(Time(x));

	if (first) {
	    for (int i=0 ; i<nPens ; i++) sc.setStripe(i, sum[i], 10.0);
	    sc.setMaxX(tbin + nStep*dTime);
	    sc.setMinX(tbin);
	    first = false;
	}
	sc.plot(tbin, sum);
    }

    //---------------------------------  Clean up at the end
    ifs.close();
    cout << "Replay finished" << endl;
}

