/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Generator.hh"
#include <iostream>
#include <stdexcept>

using namespace generator;
using namespace std;

//======================================  Generator constructor
Generator::Generator(void) {
}

//======================================  Generator destructor
Generator::~Generator(void) {
    for (src_iter i=mSource.begin() ; i!=mSource.end() ; ++i) {
        delete *i;
	*i = 0;
    }
}

//======================================  Add a channel to be generated
int 
Generator::addChannel(const char* chan, int detID, const Pipe* daq) {
    int rc = mChannel.size();
    mChannel.push_back(GenChan(chan, detID, daq));
    mChannel.back().setDebug(mDebug);
    return rc;
}

//======================================  Add a detector
int 
Generator::addDetector(const char* ifo) {
    return -1;
}

//======================================  Add a detector
int 
Generator::addDetector(const GenDet& det) {
    int rc = mDetect.size();
    mDetect.push_back(det);
    return rc;
}

//======================================  Copy a data source
int 
Generator::addSource(const std::string& name, const DataSource& data) {
    return addSource(name, data.clone());
}

//======================================  Add a data source
int 
Generator::addSource(const std::string& name, DataSource* data) {
    int rc = mSource.size();
    mSource.push_back(data);
    mSource.back()->setName(name);
    mSource.back()->setDebug(mDebug);
    return rc;
}

//======================================  Generate data.
void 
Generator::generate(const Time& t0, Interval dT) {
    int nSrc = getNSource();
    Interval tStep = mGenerStride;
    if (double(tStep) == 0.0) tStep = dT;
    for (Interval off=0.0; off < dT; off+=tStep) {
        for (int i=0 ; i<nSrc ; ++i) {
	    try {
	        mSource[i]->generate(t0+off, tStep);
	    } catch (exception& e) {
	        cout << "Exception: " << e.what() << " occurred in: ";
		mSource[i]->dumpDefinition(cout) << endl;
		throw;
	    }
	}
    }

    int nChan = getNChannel();
    Time tLate(0);
    for (int i=0 ; i<nChan ; ++i) {
        int iDet = mChannel[i].getDetID();
        mChannel[i].generate(mDetect[iDet], mSource);
    }
}

//======================================  Get the ID of the named data source
int
Generator::getSourceID(const string& name) const {
    int N = mSource.size();
    for (int i=0 ; i < N ; ++i) {
        if (name == mSource[i]->getName()) return i;
    }
    return -1;
}

//======================================  Get the ID of the named data source
Time
Generator::getLatest(void) const {
    int N = mChannel.size();
    Time r(0);
    if (N) r = mChannel[0].getLatest();
    for (int i=1 ; i < N; ++i) {
        Time ti = mChannel[i].getLatest();
        if (ti < r) r = ti;
    }
    return r;
}

//======================================  Release data from data sources.
ostream&
Generator::print_stats(ostream& out) const {
    //----------------------------------  Print source statistics
    out << "Data source statistics" << endl << endl;
    int N = mSource.size();
    for (int i=0 ; i < N ; ++i) {
        mSource[i]->print_stats(out);
    }
    return out;
}

//======================================  Release data from data sources.
void 
Generator::releaseSourceData(const Time& t0) {
    int nSrc = getNSource();
    for (int i=0; i<nSrc; ++i) mSource[i]->release(t0);
}

//======================================  Release data from data  channels.
void 
Generator::releaseChannelData(const Time& t0) {
    int nChn = getNChannel();
    for (int i=0; i<nChn; ++i) mChannel[i].release(t0);
}

//======================================  Set the debugging printout level.
void
Generator::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Set the debugging printout level.
void
Generator::setGenerStride(Interval dt) {
    mGenerStride = dt;
}
