/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_FRAMEDATA_HH
#define GENERATOR_FRAMEDATA_HH

#include "DataSource.hh"
#include "Dacc.hh"
#include <list>

namespace generator {

    /**  %FrameData allows the user to inject pre-recorded data into the 
      *  generation streams. Each %FrameData source has uses the following 
      *  parameters:
      *  <table>
      *  <tr><td>%Parameter</td>
      *      <td>Contents</td>
      *  </tr>
      *  <tr><td>%Channel</td>
      *      <td>Name of channel providing the data.</td>
      *  </tr>
      *  <tr><td>Files</td>
      *      <td>Path(s) to one or more (globbed) frame files.</td>
      *  </tr>
      *  <tr><td>Mode</td>
      *      <td>Timing mode as described below.</td>
      *  </tr>
      *  <tr><td>%Delay</td>
      *      <td>%Delay from the original frame GPS to the generated GPS</td>
      *  </tr>
      *  </table>
      *
      *  The frame data source run in one of four timing modes. These are:
      *  <ol>
      *  <li>\b synch: data are synchronized to the request times. If no data 
      *      are found for the requested time, an exception is thrown.</li>
      *  <li>\b as-available: Data are synchronized to the request times. If  
      *      the requested data are not available they are replaced with 
      *      zeros.</li>
      *  <li>\b delayed: The data are synchronized to the requested time minus
      *      the specified delay.</li>
      *  <li>\b wheel: The first available data from the specified frame list 
      *      are used. When the list of frames runs out, the data restart from 
      *      the beginning of the list.</li>
      *  </ol>
      *
      *  \note The data source sample rate must be the same as 
      *  the sample rate defined in the frame data.
      *
      *  @memo Data source reading channels from frame files.
      *  @author John Zweizig
      *  @version 1.1; last modified April 13, 2004
      */
    class FrameData : public DataSource {
    public:
	using DataSource::setParameter;

    public:
	/** Frame data constructor.
	  */
	FrameData(void);

	/**  Construct a frame data source and initialize the frame file name
	  *  and channel name parameters.
	  *  @memo Parameter constructor.
	  *  @param files   A list of files to be searched for data.
	  *  @param channel name of channel to be fetched.
	  */
	FrameData(const char* files, const char* channel);

	/**  Pre-recorded data source destructor.
	  *  @memo Destructor.
	  */
	~FrameData(void);

	/**  Make an identical copy of a frame data source.
	  *  @memo Clone a frame data source.
	  *  @return Pointer to the newly created object.
	  */
	FrameData* clone(void) const;

	/**  Return the data source type name ("FrameData").
	  *  @memo Source data type name string.
	  *  @return Constant pointer to source type name string.
	  */
	const char* getDataType(void) const;

	/**  Specify the name of the channel to be read in.
	  *  @memo Set channel name
	  *  @param name Channel name string.
	  */
	void setChannel(const std::string& name);

	/**  Set the delay for "delayed" and "wheel" data. The delay is
	  *  subtracted from the current (generated) time to give the GPS
	  *  to read from the frame.
	  *  @memo Set delay time
	  *  @param delay Delay time (in seconds)
	  */
	void setDelay(const std::string& delay);

	/**  Specify the frame file(s) to be searched for appropriate data. The
	  *  frame file(s) may be specified as a single frame path or as a 
	  *  globbed path specifier.
	  *  @memo Set the file list.
	  *  @param frames Frame name list.
	  */
	void setFileList(const std::string& frames);

	/**  Specify the timing mode for the frame data. Modes are "synch",
	  *  in which the time is requested to match the input data, or 
	  *  "wheel" in which the any data is allowed.
	  *  @memo Set the timing mode.
	  *  @param mode Timing mode name.
	  */
	void setMode(const std::string& mode);

	/**  Specify the value of a parameter
	  *  @memo Set a parameter 
	  *  @param name Parameter name
	  *  @param val  Parameter value.
	  */
	virtual void setParameter(const std::string& name, 
				  const std::string& val);

    private:  // private data types
	/**  The data input class maintains a list of files to be searched by 
	  *  a frame data source. The input data structures may be shared by 
	  *  several %FrameData data sources for efficient access of multiple 
	  *  data channels.
	  *  @memo Data input class.
	  */
	class input {
	public:
	    /**  Construct a data input object and optionally set the file list 
	      *  parameter.
	      */
	    input(const char* fl=0);

	    /**  Destroy the data input object.
	      */
	    ~input(void);

	    /**  Build a file list from the specifier string. All frames with 
	      *  data before the specified start time will be ignored.
	      *  \brief Build file list.
	      *  \param start Earliest gps for which data will be needed.
	      */
	    void buildFileList(const Time& start);

	    /**  Test whether files exist in the data accessor list of 
	      *  frame files to be read.
	      *  \brief Test for frames to be read.
	      *  \return True if no more frame files are defined.
	      */
	    bool empty(void) const;

	    /**  Get the frame file list specifier.
	      *  \brief Frame path specifier string.
	      *  \return Pointer to the path glob string.
	      */
	    const char* getFileList(void) const;

	    /**  Set the frame file specifier string.
	      *  \brief Set the file list specifier.
	      *  \param fList Frame path specifier string.
	      */
	    void setFileList(const std::string& fList);

	    /**  Read in the channel data for the specified stride into an
	      *  internal frame structure.
	      *  \brief Read a data stride.
	      *  \param t0 Data stride start time.
	      *  \param dT Data stride length.
	      */
	    void readDataStride(const Time& t0, Interval dT);

	    /**  Return a reference to the internal data accessor.
	      *  \brief Data accessor reference.
	      *  \return Reference to the data accessor (Dacc) class instance.
	      */
	    Dacc& refDacc(void);

	    /**  Return a reference to the internal data accessor.
	      *  \brief Data accessor reference.
	      *  \return Constant reference to the data accessor instance.
	      */
	    const Dacc& refDacc(void) const;

	private:
	    std::string mFileList;
	    Dacc        mIn;
	};
	typedef std::list<input> InpList;
	typedef InpList::iterator  inp_iter;

    private:
	/**  Get the default data type. For frame data, we read in the channel
	  *  For the specified epoch, and return the data type from the
	  *  frame structure.
	  *  \brief Get the channel data type.
	  *  \param t0 Start of epoch
	  *  \param dt Epoch duration.
	  *  \return data type code.
	  */
	long defaultType(const Time& t0, Interval dt);

	/**  Get the default sample rate. For frame data, we read in the channel
	  *  For the specified epoch, and return the sample rate from the
	  *  frame structure.
	  *  \brief Get the channel sample rate.
	  *  \param t0 Start of epoch
	  *  \param dt Epoch duration.
	  *  \return Sample rate.
	  */
	double defaultRate(const Time& t0, Interval dt);

	/**  Get a data series for the specified time interval.
	  *  \brief Get the data time series.
	  *  \param t0   Stride start time.
	  *  \param dT   Sample step length.
	  *  \param N    Number of samples to be read.
	  *  \param data Reference to the TSeries into which the data are 
	  *              to be placed.
	  */
	void getSourceTSeries(const Time& t0, Interval dT, int N, 
			      TSeries& data);

	/**  Search the global input accessor list for one covering the 
	  *  correct data files. If the appropriate accessor is not found,
	  *  the return value is \c mInputList.end()
	  *  \brief Find input accessor for the specified frame files.
	  *  \param filelist Frame file path string.
	  *  \return Iterator pointing to the requested input accessor or
	  *          mInputList.end() if no such accessor is defined.
	  */
	inp_iter findInput(const std::string& filelist);

	/**  Add a new input accessor to the global list for the specified
	  *  Frame file paths.
	  *  \brief Define an input accessor.
	  *  \param filelist Frame file path string.
	  */
	inp_iter addInput(const std::string& filelist);

    private:  // private data
	static InpList mInputList;
	enum timing_mode {
	    kSynch,
	    kAsAvailable,
	    kWheel,
	    kDelay
	} mTMode;
	long        mDelay;
	std::string mChannel;
	inp_iter    mInput;
	Time        mStartSeg;
    };


    //====================================  In-line FrameData methods.
    inline const char*
    FrameData::getDataType(void) const {
	return "FrameData";
    }

    //====================================  In-line Framedata::input methods.
    inline bool 
    FrameData::input::empty(void) const {
	return mIn.refList().empty();
    }

    inline const char* 
    FrameData::input::getFileList(void) const {
	return mFileList.c_str();
    }

    inline Dacc& 
    FrameData::input::refDacc(void) {
	return mIn;
    }

    inline const Dacc& 
    FrameData::input::refDacc(void) const {
	return mIn;
    }

} // namespace generator

#endif  // GENERATOR_FRAMEDATA_HH
