/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "FrameData.hh"
#include "framedir.hh"
#include "DVecType.hh"
#include <stdexcept>
#include <sstream>

using namespace std;

namespace generator {

    FrameData::InpList  FrameData::mInputList;

    //====================================  FrameData default constructor
    FrameData::FrameData(void) 
	: mTMode(kSynch), mDelay(0), mStartSeg(0)
    {
	setSample(Interval(0.0));
	mInput = mInputList.end();
    }

    //====================================  Frame data source constructor
    FrameData::FrameData(const char* files, const char* channel) 
      : mTMode(kSynch), mDelay(0), mStartSeg(0)
    {
	setSample(Interval(0.0));
	mInput = mInputList.end();
	if (files)   setFileList(files);
	if (channel) setChannel(channel);
    }

    //====================================  Frame data source destructor
    FrameData::~FrameData(void) {
    }

    //====================================  Clone a frame data source
    FrameData* 
    FrameData::clone(void) const {
	return new FrameData(*this);
    }

    //====================================  Get the default sample-rate;
    double
    FrameData::defaultRate(const Time& t0, Interval dT) {
	double rate = 0;
	Time tReq(0);
	switch (mTMode) {
	case kSynch:
	case kAsAvailable:
	    tReq = t0;
	    break;
	case kDelay:
	    tReq = t0 - Interval(mDelay);
	default:
	    break;
	}
	if (mInput != mInputList.end()) {
	    Time tFill = mInput->refDacc().getFillTime();
	    if (!tFill || !(!tReq || tReq == tFill)) {
		mInput->readDataStride(tReq, dT);
	    }
	    TSeries* ts = mInput->refDacc().refData(mChannel.c_str());
	    if (ts) {
		rate = double(ts->getTStep());
		if (rate != 0) rate = 1.0/rate;
	    }
	}
	return rate;
    }

    //====================================  Get the default data type
    long
    FrameData::defaultType(const Time& t0, Interval dT) {
	long dty = -1;
	Time tReq(0);
	switch (mTMode) {
	case kSynch:
	case kAsAvailable:
	    tReq = t0;
	    break;
	case kDelay:
	    tReq = t0 - Interval(mDelay);
	default:
	    break;
	}
	if (mInput != mInputList.end()) {
	    Time tFill = mInput->refDacc().getFillTime();
	    if (!tFill || !(!tReq || tReq == tFill)) {
		mInput->readDataStride(tReq, dT);
	    }
	    TSeries* ts = mInput->refDacc().refData(mChannel.c_str());
	    if (ts && !ts->empty()) {
		dty = ts->refDVect()->getType();
	    }
	}
	return dty;
    }

    //====================================  Set the channel ID
    void 
    FrameData::setChannel(const string& value) {
	mChannel = value;
	if (mInput != mInputList.end() && value != "data-available") {
	    mInput->refDacc().addChannel(mChannel.c_str());
	}
	DataSource::setParameter("Channel", mChannel);
    }

    //====================================  Set the channel ID
    void 
    FrameData::setDelay(const string& value) {
	mDelay = strtol(value.c_str(), 0, 0);
	DataSource::setParameter("Delay", value);
    }

    //====================================  Set the file list
    void 
    FrameData::setFileList(const string& value) {
	mInput = findInput(value);
	if (mInput == mInputList.end()) mInput = addInput(value);
	mInput->refDacc().setDebug(getDebug());
	if (!mChannel.empty() && mChannel != "data-available") {
	    mInput->refDacc().addChannel(mChannel.c_str());
	}
	DataSource::setParameter("Files", value);
    }

    //====================================  Set the file list
    void 
    FrameData::setMode(const string& value) {
	if (value == "synch") {
	    mTMode = kSynch;
	} else if (value == "wheel") {
	    mTMode = kWheel;
	} else if (value == "delay") {
	    mTMode = kDelay;
	} else if (value == "as-available") {
	    mTMode = kAsAvailable;
	} else {
	    throw runtime_error(string("FrameData: invalid mode option: ") 
				+ value);
	}
	DataSource::setParameter("Mode", value);
    }

    //====================================  Set an arbitrary parameter
    void 
    FrameData::setParameter(const string& name, const string& value) {
	if (getDebug()) cout << "Setting FrameData::" << name << " to " 
			     << value << endl;
	if (name == "Channel")    setChannel(value);
	else if (name == "Files") setFileList(value);
	else if (name == "Mode")  setMode(value);
	else if (name == "Delay") setDelay(value);
	else                      DataSource::setParameter(name, value);
    }

    //==================================  Frame data input constructor
    FrameData::input::input(const char* fl) {
	if (fl) setFileList(fl);
    }

    //==================================  Frame data input destructor
    FrameData::input::~input(void) {
    }

    //==================================  Build a list of frame files.
    void 
    FrameData::input::buildFileList(const Time& start) {
	if (mFileList.substr(0,8) == "/online/") {
	    mIn.addPath(mFileList);
	    return;
	}
	FrameDir fd(mFileList.c_str(), true);
	int nFrames = 0;
	FrameDir::file_iterator biter = (!start) ? fd.begin() : fd.find(start);
	for (FrameDir::file_iterator i=biter; i != fd.end(); ++i) {
	    mIn.addPath(i->getFile());
	    // cout << "Add file: " << i->getFile() << endl;
	    nFrames++;
	}
	cout << "File list for: " << mFileList << " (re)scanned, " 
	     << nFrames << " data frames found." << endl;
    }

    //==================================  Get data stride of appropriate length
    void 
    FrameData::input::readDataStride(const Time& t0, Interval dT) {
	bool any = !t0; // test for any-data-will-do (wheel) mode.

	//------------------------------  Check that the frame list is set
	if (empty()) buildFileList(t0);
	if (empty()) throw runtime_error("Data not available");

	//------------------------------  Fill a specific request.
	if (!any) {
	    //--------------------------  See if requested data are in the
	    //                            current fill.
	    if (mIn.getFillTime() != Time(0) && mIn.getFillTime() <= t0 && 
		mIn.getFillTime() + mIn.getFillStride() >= t0+dT) return;

	    //--------------------------  Update the current time stamp
	    if (mIn.synch() == -4) {
		throw runtime_error("No more frames.");
	    }

	    //--------------------------  Seek the start of the data.
	    if (mIn.getCurrentTime() < t0) {
		if (mIn.seek(t0)) throw runtime_error("Data not available");
	    }

	    //--------------------------  If seek went past the requested data
	    if (mIn.getCurrentTime() > t0) {
		throw runtime_error("Data not available");
	    }

	    //--------------------------  We're here!
	    int rc = mIn.fillData(dT);
	    if (rc) {
		cout << "fillData failed, source: " << mFileList << " at " 
		     << t0.getS() << " " << mIn.getMsgText(rc) << endl;
		throw runtime_error("Unable to read frame data.");
	    }
	}

	//------------------------------  Get unspecific request.
	else {
	    int rc = mIn.fillData(dT);
	    if (rc) {
		cout << "fillData failed, source: " << mFileList << " at " 
		     << t0.getS() << " " << mIn.getMsgText(rc) << endl;
		throw runtime_error("Unable to read frame data.");
	    }
	}
    }

    //==================================  Set list of frame files to be read
    void 
    FrameData::input::setFileList(const string& fList) {
	mFileList = fList;
    }

    //==================================  Get a source data time series.
    void 
    FrameData::getSourceTSeries(const Time& t0, Interval dT, int N, 
				TSeries& data) 
    {
	if (mInput == mInputList.end()) {
	    throw runtime_error("FrameData: Input stream not specified");
	} 
	if (mChannel.empty()) {
	     throw runtime_error("FrameData: Channel not specified");
	}
	if (getDebug()) {
	    mInput->refDacc().setDebug(getDebug());
	    cout << "FrameData: getting channel: " << mChannel 
		 << " data for gps " << t0 << endl;
	}

	Interval duration = dT * double(N);
	Time tReq(t0);
	TSeries* ts = 0;
	bool nodata = false;
	switch (mTMode) {
	case kSynch:
	    mInput->readDataStride(t0, duration);
	    ts = mInput->refDacc().refData(mChannel.c_str());
	    if (!ts || ts->empty()) {
		ostringstream msg;
		msg << "FrameData: Channel '" << mChannel << "' not found in: "
		    << mInput->refDacc().getFile();
		throw runtime_error(msg.str());
	    }
	    data = ts->extract(t0, duration);
	    break;
	case kAsAvailable:
	    try {
		mInput->readDataStride(t0, duration);
		ts = mInput->refDacc().refData(mChannel.c_str());
	    } catch (exception& e) {
		ts = 0;
		nodata = true;
		if (getDebug()) {
		    cout << "FrameData: No data for as-available source at gps "
			 << t0.getS() << endl;
		    cout << "           exception text: " << e.what() << endl;
		}
	    }
	    if (!ts || ts->empty()) {
		DVectD dvd(N);
		dvd.replace_with_zeros(0, N, N);
		if (!nodata && mChannel == "data-available") dvd += 1;
		data = TSeries(t0, dT, dvd);
		if (mStartSeg != Time(0)) {
		    cout << "End of " << mChannel << " data segment " 
			 << mStartSeg.getS() << " - " << t0.getS() << endl;
		}
		mStartSeg = Time(0);
	    } else {
		data = ts->extract(t0, duration);
		if (!mStartSeg) mStartSeg = t0;
	    }
	    break;
	case kDelay:
	    tReq -= Interval(mDelay);
	    mInput->readDataStride(tReq, duration);
	    ts = mInput->refDacc().refData(mChannel.c_str());
	    if (!ts || ts->empty()) {
		ostringstream msg;
		msg << "FrameData: Channel '" << mChannel << "' not found in: "
		    << mInput->refDacc().getFile();
		throw runtime_error(msg.str());
	    }
	    data=TSeries(t0, dT, *(ts->extract(tReq, duration).refDVect()));
	    break;
    	//  in wheel mode, get data from any time spot. The mStartSeg variable
    	//  is used to keep track of the previous request. If mStartSeg is 
    	//  zero or mStartSeg is contained in the current fill, the data will 
    	//  copied from the current buffer. If mStartSeg is after the fill, a 
    	//  new segment of data will be read.
	case kWheel:
	    if (getDebug()) {
		cout << "wheel entry. source: " << getName() << " start: " 
		     << mStartSeg.getS() << " fill: "
		     << mInput->refDacc().getFillTime().getS() << " current: "
		     << mInput->refDacc().getCurrentTime().getS() << endl;
	    }
	    if (!mInput->refDacc().getFillTime()) {
		try {
		    mInput->readDataStride(Time(0), duration);
		    ts = mInput->refDacc().refData(mChannel.c_str());
	            data = TSeries(t0, dT, *(ts->refDVect()));
		    mStartSeg = mInput->refDacc().getCurrentTime();
		} catch (runtime_error& e) {
		    if (getDebug()) {
			cout << "FrameData: Caught exception: " << e.what() 
			     << endl;
			cout << "           In first attempt of wheel request"
			     << endl;
		    }
		}
	    }
	    else if (!mStartSeg) {
		mStartSeg = mInput->refDacc().getFillTime();
		Interval dtFill = mInput->refDacc().getCurrentTime()-mStartSeg;
		if (duration < dtFill) dtFill = duration;
		TSeries t2 = mInput->refDacc().refData(mChannel.c_str())
			           ->extract(mStartSeg, dtFill);
	        data = TSeries(t0, dT, *(t2.refDVect()));
		mStartSeg += dtFill;
	    }
	    else if (mStartSeg >= mInput->refDacc().getFillTime() && 
		     mStartSeg < mInput->refDacc().getCurrentTime()) {
		Interval dtFill = mInput->refDacc().getCurrentTime()-mStartSeg;
		if (duration < dtFill) dtFill = duration;
		TSeries t2 = mInput->refDacc().refData(mChannel.c_str())
			           ->extract(mStartSeg, dtFill);
	        data = TSeries(t0, dT, *(t2.refDVect()));
		mStartSeg += dtFill;
	    }
	    else {
		try {
		    mInput->readDataStride(Time(0), duration);
		    ts = mInput->refDacc().refData(mChannel.c_str());
	            data = TSeries(t0, dT, *(ts->refDVect()));
		    mStartSeg = mInput->refDacc().getCurrentTime();
		} catch (runtime_error& e) {
		    if (getDebug()) {
			cout << "FrameData: Caught exception: " << e.what() 
			     << endl;
			cout << "           In first attempt of wheel request"
			     << endl;
		    }
		}
	    }
	    break;
	}
    }

    FrameData::inp_iter 
    FrameData::findInput(const std::string& filelist) {
	for (inp_iter i=mInputList.begin() ; i != mInputList.end() ; ++i) {
	    if (filelist == i->getFileList()) return i;
	}
	return mInputList.end();
    }

    FrameData::inp_iter
    FrameData::addInput(const std::string& fileList) {
	mInputList.push_back(fileList.c_str());
	return --mInputList.end();
    }

}
