/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "engine.hh"
#include "eval_stack.hh"
#include "Lexr.hh"
#include <stdexcept>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <cmath>
#include <cstdlib>

using namespace std;

//======================================  Constructor
Engine::Engine(eval_stack& stk)
  : mDebug(0), mEvalStack(stk)
{
}

//======================================  Destructor
Engine::~Engine(void) {
    ;
}

//======================================  Top level parser
void 
Engine::Parse(istream& in) {
    mLevel = 0;
    mOpList.clear();
    recursive_parse(in);
    if (mLevel) throw runtime_error("Unmatched parentheses");
}

//======================================  Recursive parse method
int
Engine::recursive_parse(istream& in) {
    mLevel++;
    int nStack = 0;
    string token;
    op_stack sav_op(eval_stack::kNoop);
    string  save_sym;
    bool end(false);
    Lexr lx;
    while (!end) {
        Lexr::LxTags tkType = lx.getToken(in, token);
	if (mDebug >1) cerr << "Parse token type: " << tkType 
			    << " value: " << token << endl;
	if (!save_sym.empty()) {
	    if (tkType == Lexr::tkOParen) {
	        int nArg = recursive_parse(in);
		ostringstream fcNarg;
		fcNarg << nArg;
	        mArgList.push_back(fcNarg.str());
	        mOpList.push_back(eval_stack::kPushNumer);
	        mArgList.push_back(save_sym);
	        mOpList.push_back(eval_stack::kCallFc);
		save_sym.clear();
		nStack++;
		continue;
	    } else {
	        mOpList.push_back(eval_stack::kPushSymbol);
	        mArgList.push_back(save_sym);
		save_sym.clear();
		nStack++;
	    }
	}
	switch (tkType) {
	case Lexr::tkSymbol:
	    save_sym = token;
	    break;
	case Lexr::tkLiteral:
	    mOpList.push_back(eval_stack::kPushNumer);
	    mArgList.push_back(token);
	    nStack++;
	    break;
	case Lexr::tkString:
	    mOpList.push_back(eval_stack::kPushConst);
	    mArgList.push_back(token);
	    nStack++;
	    break;
	case Lexr::tkOper:
	  {
	    eval_stack::OpsEnum myop   = mEvalStack.getOpID(token);
	    eval_stack::op_level mylvl = mEvalStack.getOpLevel(myop);
	    while (!sav_op.empty() && 
		   mEvalStack.getOpLevel(sav_op.back()) <= mylvl) {
	        mOpList.push_back(sav_op.back());
		sav_op.pop_back();
		nStack--;
	    }
	    sav_op.push_back(myop);
	    break;
	  }	  
	case Lexr::tkOParen:
	  {
	    int nArg = recursive_parse(in);
	    if (nArg != 1) throw runtime_error("Invalid list");
	    nStack += nArg;
	    break;
	  }
	case Lexr::tkOSubs:
	  {
	    int nArg = recursive_parse(in);
	    if (nArg != 1) throw runtime_error("Invalid list");
	    nStack += nArg;
	    break;
	  }

	//------------------------------  Terminal tokens.
	case Lexr::tkCSubs:
	case Lexr::tkCParen:
	case Lexr::tkEnd:
	    end = true;
	case Lexr::tkComma:
	    while (!sav_op.empty()) {
	        mOpList.push_back(sav_op.back());
		sav_op.pop_back();
		nStack--;
	    }
	    break;
	}
    }

    //----------------------------------  Return
    mLevel--;
    return nStack;
}

//======================================  Evaluate the rpm
void 
Engine::crank(void) {
    int nArg = mArgList.size();
    int pArg = 0;
    int nOp  = mOpList.size();
    for (int i=0; i<nOp; ++i) {
        switch (mOpList[i]) {
	case eval_stack::kNoop:
	    break;
	case eval_stack::kPushNumer:
	    if (pArg >= nArg) throw logic_error("Argument stack overflow");
	    push_numeric(mArgList[pArg++]);
	    break;
	case eval_stack::kPushConst:
	    if (pArg >= nArg) throw logic_error("Argument stack overflow");
	    push_literal(mArgList[pArg++]);
	    break;
	case eval_stack::kPushSymbol:
	    if (pArg >= nArg) throw logic_error("Argument stack overflow");
	    push_symbol(mArgList[pArg++]);
	    break;
	case eval_stack::kCallFc:
	  {
	    int N = int(mEvalStack.pop_numeric());
	    if (pArg >= nArg) throw logic_error("Argument stack overflow");
	    eval_funct(mArgList[pArg++], N);
	    break;
	  }
	default:
	    if (mDebug>3) cerr << "Eval: " 
			       << peek(2).getString() 
			       << " " << mEvalStack.getOpSym(mOpList[i]) 
			       << " " << peek(1).getString();
	    mEvalStack.evaluate(mOpList[i]);
	    if (mDebug>3) cerr << " ==> " << peek().getString() << endl;
	    break;
	}
	if (mDebug > 4) refStack().dump(cerr);
    }
}

//======================================  Dump the rpn code
ostream& 
Engine::dump_rpn(std::ostream& out) const {
    int nArg = mArgList.size();
    int pArg = 0;
    int nOp  = mOpList.size();
    out << endl << "Parse results - nOps: " << nOp << ", nArgs: " << nArg
	<< endl;
    for (int i=0; i<nOp; ++i) {
        switch (mOpList[i]) {
	case eval_stack::kNoop:
	    out << "No Op       " << endl;
	    break;
	case eval_stack::kPushConst:
	    out << "Push Const  ";
	    if (pArg >= nArg) out << "Invalid argument stack" << endl;
	    else              out << mArgList[pArg++] << endl;
	    break;
	case eval_stack::kPushNumer:
	    out << "Push Numeric  ";
	    if (pArg >= nArg) out << "Invalid argument stack" << endl;
	    else              out << mArgList[pArg++] << endl;
	    break;
	case eval_stack::kPushSymbol:
	    out << "Push Symbol ";
	    if (pArg >= nArg) out << "Invalid argument stack" << endl;
	    else              out << mArgList[pArg++] << endl;
	    break;
	case eval_stack::kCallFc:
	    out << "Call Func   ";
	    if (pArg >= nArg) out << "Invalid argument stack" << endl;
	    else              out << mArgList[pArg]   << "(" 
				  << mArgList[pArg-1] << ")" << endl;
	    pArg++;
	    break;
	default:
	    out << "Operator " << mEvalStack.getOpSym(mOpList[i]) << endl;
	    break;
	}
    }
    return out;
}

//======================================  Push value of a symbol onto the stack
void 
Engine::push_symbol(const std::string& sym) {
    throw runtime_error("Undefined symbol handler");
}

//======================================  Push a constant onto the eval stack
void 
Engine::push_literal(const std::string& lit) {
    mEvalStack.push(lit);
}

//======================================  Push a constant onto the eval stack
void 
Engine::push_numeric(const std::string& lit) {
    double x = strtod(lit.c_str(), 0);
    mEvalStack.push(x);
}

//======================================  Evaluate a function
void 
Engine::eval_funct(const std::string& func, int nArg) {
    throw runtime_error("Undefined function handler");
}

//======================================  Set debug print level
void
Engine::setDebug(int lvl) {
    mDebug = lvl;
}

