/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Skeleton Frame processing routine Template
//
// beginincludes
#include "DatTemplate.hh"
#include "FilterDesign.hh"
#include "DaccAPI.hh"
#include <iostream>
#include <string>
#include <cstdlib>
#include <unistd.h>
#include <pwd.h>
//endincludes

//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/DMT/Templates/DatTemplate.cc 7128 2014-07-16 22:29:21Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Data monitor template"
#include "ProcIdent.hh"

using namespace std;

//======================================  Generate the main routine.
EXECDAT(DatTemplate)

//======================================  Skeleton object constructor.
DatTemplate::DatTemplate(int argc, const char *argv[]) 
  : DatEnv(argc, argv), maxFrame(999999), mStep(2.0), mHistory(43200),
    mLowLimit(0), mHiLimit(32767.0), mSeg(false), mSegAcct(mTClient)
{
    //----------------------------------  Look for arguments
    bool syntax  = false;
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-channel") {
	    mChannel = argv[++i];
	} else if (argi == "-filter") {
	    mFilter = argv[++i];
	} else if (argi == "-frame") {
	    maxFrame = strtol(argv[++i], 0, 0);
	} else if (argi == "-hilimit") {
	    mHiLimit = strtod(argv[++i], 0);
	} else if (argi == "-lolimit") {
	    mLowLimit = strtod(argv[++i], 0);
	} else if (argi == "-stride") {
	    mStep = strtod(argv[++i], 0);
	} else if (argi == "+seg") {
	    mSeg = true;
	} else if (argi == "-settle") {
	    mSettle = strtod(argv[++i], 0);
 	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    } // end parse loop

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " [-channel <nMax>] [-frames <nMax>] [-stride <time>] \\" 
	     << endl
	     << "      [-filter <filter>] [-settle <s-time>] [-dvIndex] \\"
	     << endl
	     << "      [-lolimit <segMax>] [-hilimit <segMin>] [+seg]" 
	     << endl
	     << "      [<DatEnv-args>]" 
	     << endl;
	finish();
	return;
    }

    //----------------------------------  Make sure there's a valid channel
    if (mChannel.empty()) mChannel = "H0:PEM-LVEA_SEISX";

    //----------------------------------  set the data accessor before Process
    getDacc().setStride(mStep);       // Length of data to be read
    getDacc().addChannel(mChannel);   // Channel name(s)

    //----------------------------------  Specify DMT Viewer channel names
    //
    //   These methods refer to the monServer base class. The User name is 
    //   Used as a base service name, but some thing more descriptie is better.
    struct passwd *pws;
    pws = getpwuid(geteuid());
    setServerName(pws->pw_name);
    
    mSigmaName = mChannel + "_sigma";
    serveData(mSigmaName.c_str(), &mHistory, 0);

    //----------------------------------  Set up the trender
    mTrend.setName("DatTemplate");    //  Set trend frame name
    mTrend.setFrameCount(1);          //  Set frames per file
    mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
    mTrend.addChannel(mSigmaName.c_str());

    //----------------------------------  Set up segment accountant.
    if (mSeg) {
      mTClient.enroll();
      std::string ifo=mChannel.substr(0, 2);
      mLoSegId = segid_type("Blrms_Low",  1);
      mLoSegId.setIfo(ifo);
      mSegAcct.addSegment(mLoSegId, "Channel Blrms is low", 16);
      mHiSegId = segid_type("Blrms_High", 1);
      mHiSegId.setIfo(ifo);
      mSegAcct.addSegment(mHiSegId, "Channel Blrms is high", 16);
    }
}

//======================================  Skeleton object destructor.
DatTemplate::~DatTemplate() 
{
    cout << "DatTemplate is finished" << endl;
    if (mSeg) mSegAcct.close();
}

//======================================  Frame processing function.
void
DatTemplate::ProcessData(void) {
    //----------------------------------  Get pointers to the current data.
    const TSeries* ts = getDacc().refData(mChannel);
    if (!ts || ts->isEmpty()) {
        cout << "Channel: " << mChannel << " was not found." << endl;
        return;
    }

    //----------------------------------  Construct the filter if it isn't 
    //                                    already there
    if (mPipe.null() && !mFilter.empty()) {
        double samplerate(1./double(ts->getTStep()));
	FilterDesign fd(mFilter.c_str(), samplerate);
	mPipe.set(fd.release());
    }

    //----------------------------------  Filter the data
    TSeries tf(mPipe(*ts));

    //----------------------------------  Calculate channel sigma.
    float Sig = sqrt( tf*tf/tf.getNSample() - pow(tf.getAverage(), 2) );

    //----------------------------------  Trend the data points
    Time t0 = tf.getStartTime();
    mTrend.trendData(mSigmaName.c_str(), t0, Sig);
    mTrend.Update();

    //----------------------------------  set up data for display.
    mHistory.fixedAppend(t0, tf.getInterval(), &Sig);

    //----------------------------------  Add a segment
    if (mSeg) {                       //  Segments requested?
	mSegAcct.start_stride(t0);    //  Set the start time of this stride
	Time tEnd = tf.getEndTime();  //  Get the end time
	mSegAcct.set_segment("Seg_Low",  t0, tEnd, Sig < mLowLimit);
	mSegAcct.set_segment("Seg_High", t0, tEnd, Sig > mHiLimit);
	mSegAcct.update(tEnd);        //  This stride is over
    }
}


//======================================  Handle Messages
void
DatTemplate::Attention(void) {
    MonServer::Attention();
}
