// Wavelet Analysis Tool
// S.Klimenko, University of Florida
// library of general functions

#ifndef WATFUN_HH
#define WATFUN_HH

#include <cstdlib>
#include <cmath>

#define PI 3.141592653589793

// WAT functions

// Calculates polynomial interpolation coefficient using Lagrange formula.
inline double Lagrange(const int n, const int i, const double x)
{
    double c = 1.;
    double xn = x+n/2.-0.5;	// shift to the center of interpolation interval

    for(int j=0; j<n; j++) 
       if(j!=i) c *= (xn-j)/(i-j);

    return c;
}


// Nevill's polynomial interpolation.
// x - polynom argument (x stride is allways 1) 
// n - number of interpolation samples
// p - pointer to a sample with x=0.
// q - double array of length n
template<class DataType_t>
inline double Nevill(const double x0,
		    int n,   
		    DataType_t* p,
		    double* q)
{
   int i;
   double x = x0;
   double xm = 0.5;

   n--;
   *q = *p;

   for(i=0; i<n; i++)
      q[i] = p[i] + (x--)*(p[i+1]-p[i]);

   while(--n >= 1){
      x = x0;

      q[0] += xm*(x--)*(q[1]-q[0]);
      if(n == 1) goto M0;
      q[1] += xm*(x--)*(q[2]-q[1]);
      if(n == 2) goto M0;
      q[2] += xm*(x--)*(q[3]-q[2]);
      if(n == 3) goto M0;
      q[3] += xm*(x--)*(q[4]-q[3]);
      if(n == 4) goto M0;
      q[4] += xm*(x--)*(q[5]-q[4]);
      if(n == 5) goto M0;
      q[5] += xm*(x--)*(q[6]-q[5]);
      if(n == 6) goto M0;

      for(i=6; i<n; i++)
	 q[i] += xm*(x--)*(q[i+1]-q[i]);

M0:   xm /= (1.+xm);
   }

   return *q;
}

// calculate significance for sign cross-correlation
inline double signPDF(const size_t m, const size_t k) 
{
   size_t i;
   double pdf = 0.;
   size_t n = m+k;
   double rho = (double(m)-double(k))/n;

   if(n < 1) return 0.;
   if(n < 100) {
      for(i=1; i<k+1; i++){ pdf -= log(double(m+i)/double(i)); }
      pdf -= log(double(n))-(n+1)*log(2.);
      pdf -= log(sqrt(2.*PI/n));  // normalization from Gaussian approximation
   }
   else pdf = n*rho*rho/2.;
   return pdf;
}


// survival probability for Gamma distribution
// calculates integral I=int(x*x^n*exp(-x)) from Y to infinity
// returns confidence = -log(I/Gamma(n))
// Y - low integration limit
// n - Gamma function parameter
inline double gammaCL(double Y, int n){   
   double y,s;

//   if(Y<=n){
     y = Y; s = 1.;
     for(int k=n-1; k>0; k--){ 
       s += y; 
       y*=double(Y)/(n-k+1); 
       if(y>1.e290) break;
     }
     return double(Y-log(s));
/*
   }
   else{
     double z = log(Y/n);
     y = n/Y; s = n/Y;
     for(int k=n-1; k>0; k--){ 
       y *= double(k/Y); 
       s += y; 
       z += log(Y/k); 
       if(y<1.e-290) break;
     }
     return float(Y-z-log(s));
   }
*/
}


// Calculates polynomial interpolation coefficients using Lagrange formula.
// gap is allouded in the middle of interval, like
// n=13, m=5:  x x x x o o o o o x x x x 
inline void fLagrange(int n, int m, double* c)
{
    if(!(n&1)) n++;
    if(!(m&1)) m++;
    int i,j;

    for(i=0; i<n; i++) c[i]=0.; 

    for(i=0; i<n; i++) {
       if(abs(n/2-i)<=m/2) continue;
       c[i] = 1.;
       for(j=0; j<n; j++) {
	  if(abs(n/2-j)<=m/2) continue;
          if(j!=i) c[i] *= double(n/2-j)/(i-j);
       }
    }
    return;
}

#endif // WATFUN_HH
