#ifndef HISTOGRAM1_HH
#define HISTOGRAM1_HH

#include "Time.hh"
#include <string>
#include <iosfwd>

/** The Histogram1 class holds a 1-dimensional histogram data.
  * @brief Histogram API.
  * @author Masahiro Ito
  * @version 1.0; Last modified October 15, 2001
  */

class Histogram1
{
public:
  /// Bin Type
  enum EBinType{ kUndefinedBin,
		 kFixedBin,
		 kVariableBin };

  /**  Histogram axis data type.
    *  @brief Axis type;
    */
  typedef double xbin_t;

  /**  Histogram bin content data type.
    *  @brief Bin content type;
    */
  typedef double histdata_t;

  /**  Histogram statistics data type.
    *  @brief Statistics type;
    */
  typedef double stat_t;

private:
  /// Number of Bins
  int fNbinx;

  /// Number of data entries
  int fNEntries;

  /// Sum of weights
  stat_t fTsumw;

  /// Sum of weights^2
  stat_t fTsumw2;

  /// Sum of weight * data
  stat_t fTsumwx;

  /// Sum of weight * (data^2)
  stat_t fTsumwx2;

  /// array for bin contents + Over/Underflow
  histdata_t* fArray;

  /// array for bin error^2 + Over/Underflow if fSumw2 is allocated, fBinErrorFlag = true.
  stat_t* fSumw2;

  /// array for lower bin edges + highest bin edge 
  xbin_t* fXbins;

  /// Histogram title
  std::string fTitle;

  /// X-axis label
  std::string fXLabel;

  /// Bin content axis label
  std::string fNLabel;

  /// bin type (0:Undefined 1: Fixed, 2: Variable)
  int fBinType;

  /// flag to check bin-error array is ON/OFF. (true: ON, flase: OFF) 
  bool fBinErrorFlag;

  /// Time stamp
  Time fTime;

  /// Look for the corresponding bin number to the data
  int SearchBin(int min, int max, xbin_t x) const;

protected:
  /// Free old memory, then allocate new memory
  void Allocate(int nbin = 0);

public:

  /**  A histogram object is initialized.
    *  @brief Default constructor.
    */
  Histogram1():fArray(0),fSumw2(0),fXbins(0) { Reset(); }

  /**  The argument histogram is duplicated into the new histogram.
    *  @brief  Copy Constructor.
    *  @param h Histogram to be duplicated.
    */
  Histogram1(const Histogram1& h):fArray(0),fSumw2(0),fXbins(0) { *this = h; }

  /**  A 1-d histogram is created with a specified number of bins, lower edge and upper edge.
    *  Bin spacings are fixed.
    *  @brief  Dataless constructor.
    *  @param name Histogram name.
    *  @param nbinx number of bins.
    *  @param xmin lower edge.
    *  @param xmax upper edge.
    *  @param xlabel labe for the X-axis.
    *  @param nlabel upper for the bin content axis.
    */
  Histogram1(const char* name, int nbinx, xbin_t xmin, xbin_t xmax,
	  const char* xlabel = "", const char* nlabel = "");

  /**  A 1-d histogram is created with a specified number of bins, lower edges.
    *  Bin spacings are variable.
    *  @brief  Dataless constructor.
    *  @param name Histogram name.
    *  @param nbinx number of bins.
    *  @param xbins array of lower-edges for each bin
    *  (the last element is the upper edge).
    *  @param xlabel label for the X-axis.
    *  @param nlabel upper for the bin content axis.
    */
  Histogram1(const char* name, int nbinx, const xbin_t* xbins,
	  const char* xlabel = "", const char* nlabel = "");

  /**  Histogram1 destructor.
    *  @brief Histogram1 Destructor.
    */
  virtual ~Histogram1();

  /**  Bin contents, number of data entries and time are zeroed.
    *  @brief   Clear bin contents, number of data entries and time.
    */
  virtual void Clear();

  /**  A formatted dump of the histogram contents are written to the specified output
    *  stream.
    *  @brief   Output formatted dump of the Histogram1 data.
    *  @param  out  Output stream to receive the formatted Histogram1 dump.
    *  @return reference to the specified output stream.
    */
  virtual std::ostream& Dump(std::ostream& out) const;

  /**  The corresponding bin to the data is incremented by the given weight.
    *  The default value of the weight is 1.
    *  @brief  Increment the corresponding bin to the data by the weight.
    *  @param x Data.
    *  @param w Weight.
    */
  virtual void Fill(xbin_t x, double w = 1.0);

  /**  Corresponding bins to each element of the data array are incremented by 1.
    *  @brief  Increment corresponding bins to each element of the data array.
    *  @param ntimes Number of elements in the data array.
    *  @param x Data array.
    */
  virtual void FillN(int ntimes, const xbin_t* x);

  /**  Corresponding bin to each data array element is 
    *  incremented by given weights.
    *  @brief  Increment corresponding bins to each data array element 
    *  by given weights.
    *  @param ntimes Number of entries in the data array.
    *  @param x Data array.
    *  @param w Array of weights.
    */
  virtual void FillN(int ntimes, const xbin_t* x, const double* w);

  /**  All bin contents of the histogram is copied to an array.
    *  N+2 arrays required. 
    *  data[0] and data[N+1] contain under and overflow of the histogram, respectively.
    *  (N: Number of bins)
    *  @brief  Copy bin contens to a double array.
    *  @param data Double array into which the bin contents to be copied.
    */
  virtual void GetBinContents(histdata_t* data) const;

  /**  Bin edges are copied to an array.
    *  N+1 arrays required.
    *  bin[0] to bin[N-1] hold low-edges.
    *  bin[N] contains the upper edge.
    *  (N: Number of bins)
    *  @brief  Copy low-edges and upper edge to a double array.
    *  @param bin Double array into which the lower-edges to be copied.
    */
  virtual void GetBinLowEdges(xbin_t* bin) const;

  /**  The low-edge of n-th bin is returned.
    *  @brief  Return low-edge of the n-th bin.
    *  @param n Bin number. (N: Number of bins)
    *  <table>
    *    <tr><td>1 to N </td><td> low-edges of n-th bin </td></tr>
    *    <tr><td>N+1    </td><td> Upper edge </td></tr>
    *  </table>
    */
  virtual xbin_t GetBinLowEdge(int n) const;

  /**  The content of the n-th bin is returned.
    *  @brief  Return the n-th bin content.
    *  @param n Bin number. (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @return n-th bin content.
    */
  virtual histdata_t GetBinContent(int n) const;

  /**  The center of the n-th bin is returned.
    *  @brief  Return the center of n-th bin content.
    *  @param n Bin number. ( n = 1 to N where N is Number of bins)
    *  @return n-th bin center.
    */
  virtual xbin_t GetBinCenter(int n) const;

  /**  The errors of bins are copied to an array. If bin errors aren't defined, returns false.
    *  N+2 arrays required.
    *  err[0] and err[N+1] contain errors of under and overflow, respectively.
    *  (N: Number of bins)
    *  @brief  Copy the error of bins to an array.
    *  @param err Array into which the errors to be copied.
    *  @return return true if bin errors are defined, otherwise false.
    */ 
  virtual bool GetBinErrors(stat_t* err) const;

  /**  The error of the n-th bin is returned.
    *  @brief  Return the n-th bin error.
    *  @param n Bin number. (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @return n-th bin error, if bin errors aren't defined return -1.
    */
  virtual stat_t GetBinError(int n) const;

  /**  The bin number corresponding to the argument value is returned.
    *  @brief  Return the bin number.
    *  @param x Data.
    *  @return bin number.
    */
  virtual int GetBinNumber(xbin_t x) const;

  /**  The bin spacing for a fixed bin spacing histogram is returned.
    *  @brief  Return bin spacing for fixed bin spacing histogram.
    *  @return bin spacing. -1 if bin spacing is variable or undefined.
    */
  virtual xbin_t GetBinSpacing(void) const;

  /**  The number of data entries is returned. 
    *  @brief   Return the number of data entries.
    *  @return Number of data entries.
    */
  virtual int GetNEntries(void) const { return fNEntries; }

  /**  The number of bins in the histogram is returned. 
    *  @brief   Return the number of bins.
    *  @return Number of bins.
    */
  virtual int GetNBins(void) const { return fNbinx; }

  /**  The title of the histogram is returned. 
    *  @brief   Return title of the histogram.
    *  @return Title of the histogram.
    */
  virtual const char* GetTitle(void) const;

  /**  The label of the X-axis is returned. 
    *  @brief   Return X-axis label.
    *  @return X-axis Label.
    */
  virtual const char* GetXLabel(void) const;

  /**  The label of the bin-count axis is returned. 
    *  @brief   Return bin-count axis label.
    *  @return bin-count axis Label.
    */
  virtual const char* GetNLabel(void) const;

  /**  The minimum of bin contents is returned. 
    *  @brief   Return the minimum of bin contents.
    *  @return Minimum of bin contents.
    */
  virtual histdata_t GetMinContent(void) const;

  /**  The maximum of bin contents is returned. 
    *  @brief   Return the maximum of bin contents.
    *  @return Maximum of bin contents.
    */
  virtual histdata_t GetMaxContent(void) const;

  /**  The bin number for the minimum bin content is returned. 
    *  @brief   Return the bin number for the minimum bin content.
    *  @return Bin number for the minimum bin contents.
    */
  virtual int GetMinContentBin(void) const;

  /**  The bin number for the maximum bin content is returned. 
    *  @brief   Return the bin number for the maximum bin content.
    *  @return Bin number for the maximum bin contents.
    */
  virtual int GetMaxContentBin(void) const;

  /**  The mean value is returned. 
    *  @brief   Return the mean value.
    *  @return Mean.
    */
  virtual stat_t GetMean(void) const;

  /**  The standard deviation is returned. 
    *  @brief   Return the standard deviation.
    *  @return standard deviation.
    */
  virtual stat_t GetSdev(void) const;

  /**  The statistics of the histogram are copied to the specified array.
    *  <table>
    *    <tr><td> stats[0] </td><td> Sum of weight </td></tr>
    *    <tr><td> stats[1] </td><td> Sum of weight^2 </td></tr>
    *    <tr><td> stats[2] </td><td> Sum of weight * data </td></tr>
    *    <tr><td> stats[3] </td><td> Sum of weight * data^2 </td></tr>
    *  </table>
    *  @param stats Statistics output array
    *  @brief   Copy statistics of the histogram to an array.
    */
  virtual void GetStats(stat_t *stats) const;

  /**  The statistics of the histogram are replaced by new values.
    *  <table>
    *    <tr><td> stats[0] </td><td> Sum of weight </td></tr>
    *    <tr><td> stats[1] </td><td> Sum of weight^2 </td></tr>
    *    <tr><td> stats[2] </td><td> Sum of weight * data </td></tr>
    *    <tr><td> stats[3] </td><td> Sum of weight * data^2 </td></tr>
    *  </table>
    *  @param stats Statistics input array
    *  @brief Replace the statistics of the histogram.
    */
  virtual void PutStats(const stat_t *stats);

  /**  The type of the histogram bin is returned.
    *  @brief   Return the bin type.
    *  @return Bin type.
    *  <table>
    *    <tr><td> 0 </td><td> Undefined </td></tr>
    *    <tr><td> 1 </td><td> Fixed Bin </td></tr>
    *    <tr><td> 2 </td><td> Variable Bin </td></tr>
    *  </table>
    */
  virtual int GetBinType(void) const { return fBinType; }

  /**  GPS time is returned.
    *  @brief  Return GPS time.
    *  @return GPS time.
    */
  virtual Time GetTime(void) const { return fTime; }

  /**  If error of bins are defined, return true.
    *  @brief  Return true, if bin errors are defined.
    *  @return true if bin errors are defined, otherwise false.
    */
  virtual bool IsErrorFlagON(void) const { return fBinErrorFlag; }

  /**  New title is copied to the title string.
    *  @brief  Set the histogram title.
    *  @param name Pointer to the character array.
    */
  virtual void SetTitle(const char* name) { fTitle = name; }

  /**  New label is copied to the X-axis label string.
    *  @brief  Set the X-axis label.
    *  @param xlabel Pointer to the character array.
    */
  virtual void SetXLabel(const char* xlabel) { fXLabel = xlabel; }

  /**  New label is copied to the bin-count axis label string.
    *  @brief  Set the bin-count axis label.
    *  @param nlabel Pointer to the character array.
    */
  virtual void SetNLabel(const char* nlabel) { fNLabel = nlabel; }

  /**  The histogram bins are redefined.
    *  @brief  Set the histogram bins.
    *  @param nbinx Number of bins.
    *  @param xmin Lower edge.
    *  @param xmax Upper edge.
    */
  virtual void SetBinLowEdges(int nbinx, xbin_t xmin, xbin_t xmax);

  /**  The histogram bins are redefined.
    *  @brief  Set the histogram bins.
    *  @param nbinx Number of bins.
    *  @param xbins Array of low-edges (the last element is the upper edge).
    */
  virtual void SetBinLowEdges(int nbinx, const xbin_t* xbins);

  /**  The histogram contents are replaced by new contents.
    *  Require N+2 data array.
    *  data[0] and data[N+1] contain underflow and overflow, respectivly. 
    *  @brief  Replace the contents of bins.
    *  @param data Array of new contents.
    */
  virtual void SetBinContents(const histdata_t* data);

  /**  The n-th bin content is replaced by the new content.
    *  @brief  Replace n-th bin content.
    *  @param n Bin number. (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @param content New content.
    */
  virtual void SetBinContent(int n, histdata_t content);

  /**  The errors of bins are replaced by new values.
    *  Require N+2 data array.
    *  err[0] and err[N+1] contain errors of underflow and overflow, respectivly. 
    *  @brief  Replace errors of bins.
    *  @param err Array of new values of error.
    */
  virtual void SetBinErrors(const stat_t* err);

  /**  The n-th bin error is replaced by the new value.
    *  If bin errors aren't defined, return false and does nothing on err array.
    *  @brief  Replace n-th bin error.
    *  @param n Bin number. (0: Underflow, N+1: Overflow where N is Number of bins)
    *  @param err New content.
    *  @return true if bin errors are defined, otherwise false.
    */
  virtual bool SetBinError(int n, stat_t err);

  /**  The number of entries to the histogram is replaced by the new number.
    *  @brief  Replace the number of entries with new number.
    *  @param n Number of entries to be replaced.
    */
  virtual void SetNEntries(int n) { fNEntries = n; }

  /**  The number of bins of the histogram is changed to the specified number.
    *  @brief  Set number of bins.
    *  @param nbinx Number of bins.
    */
  virtual void SetNBins(int nbinx) { fNbinx = nbinx; }

  /**  The bin type of the histogram is replaced to the specifed type.
    *  @brief  Set the bin type.
    *  @param n Bin type. (0: Undefined, 1: Fixed, 2: Variable)
    */
  virtual void SetBinType(int n);

  /**  A time stamp in GPS seconds is set.
    *  @brief  Set a time stamp (GPS).
    *  @param t GPS time.
    */
  virtual void SetTime(const Time& t) { fTime = t; };

  /**  The errors of bins are cleared or 
    *  updated based on the current bin content.
    *  @brief  Clear or Update errors of bins based on the current bin content.
    *  @param reset (true: clear bin error with zero, 
    *  false: update bin error)
    */
  virtual void Sumw2(bool reset = true);

  /**  All the histogram data and statistics are cleared.
    *  @brief  Clear all the histogram data.
    */
  virtual void Reset(void);
  
  /**  A constant reference to the Nth bin is returned.
    *  @brief   Get the reference to the Nth bin.
    *  @param  n Bin number.
    *  @return Reference to the Nth bin.
    */
  virtual const histdata_t& operator [] (int n) const { return fArray[n]; }
  
  /**  Reference to the Nth bin is returned.
    *  @brief   Get the reference to the Nth bin.
    *  @param  n Bin number.
    *  @return Reference to the Nth bin.
    */
  virtual       histdata_t& operator [] (int n)       { return fArray[n]; }

  /**  The Histogram1 data are copied to the current Histogram1.
    *  @brief  Copy histogram.
    *  @param h Constant Histogram1 to be copied.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator = (const Histogram1& h);

  /**  The Histogram1 data are added to the current Histogram1.
    *  @brief  Add a histogram to the current histogram.
    *  @param h Constant Histogram1 to be added.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator += (const Histogram1& h);

  /**  A bias is added to each bin of the histogram.
    *  @brief  Add a bias to each bin.
    *  @param bias Number to be added.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator += (histdata_t bias);

  /**  The current Histogram1 is subtracted by another Histogram1 data.
    *  @brief  Subtract the histogram from the current histogram.
    *  @param h Constant Histogram1 to subtract.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator -= (const Histogram1& h);

  /**  Bin contents are multiplied by a scalar.
    *  @brief  Multiply bin contents by a scalar.
    *  @param scale Number to be multiplied.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator *= (double scale);

  /**  Current Histogram1 is multiplied by the argument histogram.
    *  @brief  Multiply current Histogram1 by the argument histogram.
    *  @param h Constant Histogram1 to multiply.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator *= (const Histogram1& h);

  /**  Current Histogram1 is divied by the argument histogram bin by bin.
    *  @brief  Divide bin contents by bin contents of the argument histogram.
    *  @param h Constant Histogram1 to divide.
    *  @return Reference to the updated Histogram1.
    */
  Histogram1& operator /= (const Histogram1& h);
};

/** @name Histogram1 functions
  * This section contains functions of Histogram1 that are not members of the 
  * Histogram1 class. 
  * @brief Non-member functions of Histogram1.
  * @author Masahiro Ito
  * @version 1.0; Last modified October 15, 2001
  */
//@{

/**  A bias is added to bins of the histogram.
  *  @brief  Add a bias to each bin.
  *  @param bias Number to be added.
  *  @param h Constant reference to original histogram.
  *  @return Biased Histogram1.
  */
Histogram1 operator + (Histogram1::histdata_t bias, const Histogram1& h);

/**  A bias is added to bins of the histogram.
  *  @brief  Add a bias to each bin.
  *  @param bias Number to be added.
  *  @param h Constant reference to original histogram.
  *  @return Biased Histogram1.
  */
Histogram1 operator + (const Histogram1& h, Histogram1::histdata_t bias);

/**  Two Histogram1 data are added together.
  *  @brief  Add two Histogram1 data together.
  *  @param h1 Constant Histogram1 to be added.
  *  @param h2 Constant reference to original histogram.
  *  @return Summed Histogram1.
  */
Histogram1 operator + (const Histogram1& h1, const Histogram1& h2);

/**  A histogram is subtracted by the other.
  *  @brief  Subtract a Histogram1 from the other.
  *  @param h1 Constant reference to original histogram.
  *  @param h2 Constant Histogram1 to subtract.
  *  @return Subtracted Histogram1.
  */
Histogram1 operator - (const Histogram1& h1, const Histogram1& h2);

/**  The histogram is multiplied by a scalar.
  *  @brief  Multiply the histogram by a scalar.
  *  @param h Constant reference to original histogram.
  *  @param scale Number to be multiplied.
  *  @return Scaled Histogram1.
  */
Histogram1 operator * (double scale, const Histogram1& h);

/**  The histogram is multiplied by a scalar.
  *  @brief  Multiply the histogram by a scalar.
  *  @param h Constant reference to original histogram.
  *  @param scale Number to be multiplied.
  *  @return Scaled Histogram1.
  */
Histogram1 operator * (const Histogram1& h, double scale);

/**  Two histograms are multiplied together.
  *  @brief  Multiply two histograms together.
  *  @param h1 Histogram1 to be multiplied.
  *  @param h2 Histogram1 to be multiplied.
  *  @return Multiplied Histogram1.
  */
Histogram1 operator * (const Histogram1& h1, const Histogram1& h2);

/**  The first histogram is devided by the second.
  *  @brief  Divide the first histogram by the second.
  *  @param h1 Histogram1 to be divided.
  *  @param h2 Histogram1 to divide.
  *  @return Divided Histogram1.
  */
Histogram1 operator / (const Histogram1& h1, const Histogram1& h2);

//@}

#endif  // HISTOGRAM1_HH
