/* -*- mode: c++; c-basic-offset: 4; -*- */
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: fzstreambuf						*/
/*                                                         		*/
/* Module Description: Stream buffer which performs I/O to/from 	*/
/*                     gzipped files.                                   */
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Author  	Comments				*/
/* 0.1	 21Jul08  J. Zweizig  	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: 					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FZSTREAMBUF_H
#define _LIGO_FZSTREAMBUF_H

#include <iosfwd>
#include <ios>
#include <streambuf>
#include <stdexcept>


namespace gdsbase {

    void* fzopen  (const char* file, std::ios::openmode m);
    int   fzclose (void* file);
    int   fzread  (void* file, char* buf, int length, int size);
    int   fzseek  (void* file, int off, std::ios_base::seekdir way, int size);
    int   fzwrite (void* file, const char* buf, int length, int size);
 
}

namespace std {

/** @name Streams that work with file descriptors. 
    This header defines streams that work with normal UNIX file
    descriptors. It utilizes a stream buffer which reads and writes from
    file descriptors. The current implementation does not use buffered 
    output.
   
    @memo File descriptor streams
    @author Written April 2001 by Daniel Sigg
    @version 1.0
 ************************************************************************/

//@{

/** Basic stream buffer with file descriptors.
    
    @memo Basic file descriptors stream buffer
 ************************************************************************/
    template <class charT, class traits = char_traits <charT> >
    class basic_fzbuf : public basic_streambuf <charT, traits> {
    public:
	typedef typename basic_streambuf <charT, traits>::int_type int_type;
	typedef typename basic_streambuf <charT, traits>::off_type off_type;
	typedef typename basic_streambuf <charT, traits>::pos_type pos_type;
	typedef typename basic_streambuf <charT, traits>::char_type char_type;
	typedef traits traits_type;

	/**  Constructor.
	  *  Create an idle streambuf.
	  */
	basic_fzbuf (void);
      
	/**  Constructor.
	  *  Create an fzbuf attached to a file stream.
	  */
	basic_fzbuf(const char* file, ios_base::openmode m);

	/**  Open a stream to a file.
	  */
	basic_fzbuf* open(const char* file, ios_base::openmode m);

	/**  Destructor.
	  *  Destroy a streambuf.
	  */
	virtual ~basic_fzbuf (void);

	/** close the stream
	  */
	void close(void);

	/**  Test whether stream is open.
	  */
	bool is_open(void) const;

    protected:
	/// size of input data buffer
	static const long fBufferSize = 1024;

	/// File (compressed) side datastream
	void*	        fgzFile;

	/// Stream side (uncompressed) data buffer
	char_type*	fBuffer;

	/// Stream side (uncompressed) data buffer
	char_type*	fCurrent;

	/// Stream side (uncompressed) data buffer
	char_type*	fLast;

	/// Stream side (uncompressed) data buffer
	pos_type       	fPosition;
   
	/// Read one character
	virtual int_type underflow ();

	/// Write one character
	virtual int_type overflow (int_type c = traits::eof());

	/// Write multiple characters
	virtual streamsize xsputn (const char_type* s, streamsize num);

	/// Seek with offset
	virtual pos_type seekoff(off_type off, ios_base::seekdir way,
				 ios_base::openmode which =
				 ios_base::in | ios_base::out);

	/// Seek with position
	virtual pos_type seekpos(pos_type sp,
				 ios_base::openmode which =
				 ios_base::in | ios_base::out);
   
	/// file descriptor read
	static streamsize fzread (void* fz, char_type* s, streamsize num) {
	    return gdsbase::fzread (fz, reinterpret_cast<char*>(s), num, 
				    sizeof (char_type));
	}

	/// file descriptor write
	static streamsize fzwrite(void* fz, const char_type* s, streamsize num){
	    return gdsbase::fzwrite(fz, reinterpret_cast<const char*>(s), num, 
				    sizeof(char_type));
	}

	/// file descriptor seek
	static pos_type fzseek(void* fz, off_type off, ios_base::seekdir way) {
	    return gdsbase::fzseek (fz, off, way, sizeof (char_type)); 
	}
    };
    
    /**  Stream buffer with file descriptors (char).
      *	 @memo Stream buffer with file descriptors (char)
      *******************************************************************/
    typedef basic_fzbuf<char> fz_streambuf;

    /**  Basic input stream with file descriptors.
      *  @memo Basic input stream buffer with file descriptors
      *******************************************************************/
    template <class charT, class traits = char_traits <charT> >
    class basic_fzistream : public basic_istream <charT, traits> {
    public:
	/// Create an output stream from an open file descriptors
	basic_fzistream (void) 
	    : basic_istream <charT, traits>(0)
	{
	    this->init(&fBuf); 
	    this->clear(); 
	}

	explicit basic_fzistream (const char* file, 
				  ios::openmode mode=ios_base::in) 
	    : basic_istream <charT, traits>(0), fBuf (file, mode) 
	{
	    this->init(&fBuf); 
	    this->clear();
	}

	bool is_open(void) const {return fBuf.is_open();}
    protected:
	/// Stream buffer
	basic_fzbuf<charT, traits> fBuf;
    };


    /**  Basic output stream with file descriptors.
      *  @memo Basic output stream buffer with file descriptors
      *******************************************************************/
    template <class charT, class traits = char_traits <charT> >
    class basic_fzostream : public basic_ostream <charT, traits> {
    public:
	/// Create an output stream from an open file descriptors
	basic_fzostream (void) 
	    : basic_ostream <charT, traits>(0)
	{
	    this->init (&fBuf); 
	    this->clear();
	}

	explicit basic_fzostream (const char* file, 
				  ios::openmode mode=ios_base::out) 
	    : basic_ostream <charT, traits>(0), fBuf (file, mode) 
	{
	    this->init (&fBuf); 
	    this->clear();
	}

	bool is_open(void) const {return fBuf.is_open();}
    protected:
	/// Stream buffer
	basic_fzbuf<charT, traits> fBuf;
    };


    /**  Basic IO stream with file descriptors.
      *  @memo Basic IO stream buffer with file descriptors
      *******************************************************************/
    template <class charT, class traits = char_traits <charT> >
    class basic_fziostream : public basic_iostream <charT, traits> {
    public:
	/// Create an output stream from an open file descriptors
	explicit basic_fziostream (void) 
	    : basic_iostream <charT, traits>(0)
	{
	    this->init (&fBuf); 
	    this->clear(); 
	}
	/// Create an output stream from an open file descriptors
	basic_fziostream (const char* file, ios_base::openmode mode) 
	    : fBuf (file, mode)
	{
	    this->init (&fBuf); 
	    this->clear(); 
	}

	bool is_open(void) const {return fBuf.is_open();}

    protected:
	/// Stream buffer
	basic_fzbuf<charT, traits> fBuf;
    };

    /**  Input stream with file descriptors (char).
      *  @memo Input stream buffer with file descriptors (char)
      ******************************************************************/
    typedef basic_fzistream<char> fz_istream;

    /**  Input stream with file descriptors (wchar_t).
      *  @memo Input stream buffer with file descriptors (wchar_t)
      ******************************************************************/
    typedef basic_fzistream<wchar_t> fz_wistream;

    /**  Output stream with file descriptors (char).
      *  @memo Output stream buffer with file descriptors (char)
      *******************************************************************/
    typedef basic_fzostream<char> fz_ostream;

    /** Output stream with file descriptors (wchar_t).
      *  @memo Output stream buffer with file descriptors (wchar_t)
      *******************************************************************/
    typedef basic_fzostream<wchar_t> fz_wostream;

    /**  IO stream with file descriptors (char).
      *  @memo IO stream buffer with file descriptors (char)
      *******************************************************************/
   typedef basic_fziostream<char> fz_iostream;

    /**  IO stream with file descriptors (wchar_t).
      *  @memo IO stream buffer with file descriptors (wchar_t)
      *******************************************************************/
#ifndef __CINT__
   typedef basic_fziostream<wchar_t> fz_wiostream;
#endif

//@}

    //_____________________________________________________________
    template <class charT, class traits>
    basic_fzbuf<charT,traits>::basic_fzbuf (void) 
	: fgzFile(0), fBuffer(0), fCurrent(0), fLast(0)
    {
    }

    //________________________________________________________________________
    template <class charT, class traits>
    basic_fzbuf<charT,traits>::basic_fzbuf(const char* file, ios::openmode m)
	: fgzFile(0), fBuffer(0), fCurrent(0), fLast(0)
    {
	if (! open(file, m)) throw runtime_error("unable to open file");
    }

    //________________________________________________________________________
    template <class charT, class traits>
    basic_fzbuf<charT,traits>::~basic_fzbuf(void) {
	close();
	if (fBuffer) delete[] fBuffer;
    }

    //________________________________________________________________________
    template <class charT, class traits>
    void
    basic_fzbuf<charT,traits>::close(void) {
	if (fgzFile) {
	    gdsbase::fzclose(fgzFile);
	    fgzFile = 0;
	}
    }

    //________________________________________________________________________
    template <class charT, class traits>
    bool
    basic_fzbuf<charT,traits>::is_open(void) const {
        return (fgzFile != NULL);
    }

    //________________________________________________________________________
    template <class charT, class traits>
    basic_fzbuf<charT,traits>* 
    basic_fzbuf<charT,traits>::open(const char* file, ios::openmode mode) {
	fgzFile = gdsbase::fzopen(file, mode);
	if (!fgzFile) return 0;

	fBuffer  = new char_type[fBufferSize];
	fCurrent = fBuffer;
	fLast    = fBuffer;
	this->setg (fBuffer, fCurrent, fLast);
	return this;
    }

    //________________________________________________________________________
    template <class charT, class traits >
    typename basic_fzbuf<charT,traits>::int_type 
    basic_fzbuf<charT,traits>::underflow () {
	if (!fgzFile) {
	    return traits_type::eof(); 
	}
	//cerr << "Underflow" << endl;
	
	// read before end of buffer?
	if (this->gptr() < this->egptr()) {
	    return *this->gptr();
	}
	// process putback area
	streamsize numPutback;
	numPutback = this->gptr() - this->eback();
	if (numPutback > 4) numPutback = 4;
	// copy putback area upfront
	traits_type::copy(fBuffer+(4-numPutback), this->gptr()-numPutback, 
			  numPutback);
	// get new characters
	streamsize num = fzread (fgzFile, fBuffer+4, fBufferSize-4);
	//cerr << "read char " << num << endl;
	if (num <= 0) {
	    return traits_type::eof();
	}
	// reset buffer pointers
	this->setg (fBuffer+(4-numPutback), fBuffer+4, fBuffer+4+num);
	return *this->gptr(); 
    }

    //____________________________________________________________________
    template <class charT, class traits >
    typename basic_fzbuf<charT,traits>::int_type 
    basic_fzbuf<charT,traits>::overflow (int_type c) {
	if (! fgzFile) {
	    return traits_type::eof(); 
	}
	//cerr << "Overflow = " << (int) c << endl;
	if (c != traits_type::eof()) {
	    char z = c;
	    if (fzwrite (fgzFile, &z, 1) != 1) {
		return traits_type::eof();
	    }
	}
	//cerr << "WRITE = " << (int)c << endl;
	return c;
    }

    //_______________________________________________________________________
    template <class charT, class traits >
    streamsize 
    basic_fzbuf<charT,traits>::xsputn (const char_type* s, streamsize num) {
	if (! fgzFile) {
	    return 0;
	}
	// cerr << "WRITE " << num << endl;
	// ::write (2, s, num);
	// cerr << endl;
	return fzwrite (fgzFile, s, num); 
    }

    //__________________________________________________________________________
    template <class charT, class traits>
    typename basic_fzbuf<charT,traits>::pos_type 
    basic_fzbuf<charT,traits>::seekoff (off_type off, 
					ios_base::seekdir way,
					ios_base::openmode which){
	return pos_type(off_type(-1));
    }

    //__________________________________________________________________________
    template <class charT, class traits>
    typename basic_fzbuf<charT,traits>::pos_type 
    basic_fzbuf<charT,traits>::seekpos (pos_type sp, ios_base::openmode which) {
	return seekoff (sp, ios_base::beg, which);
    }

}

#endif // _LIGO_FZSTREAMBUF_H

