/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "eval.hh"
#include "Time.hh"
#include <sstream>
#include <iostream>
#include <cmath>

using namespace table;
using namespace std;

//======================================  Eval constructor
eval::eval(const Table& t, const string& cmd, int debug) 
  : Engine(mStack), mTable(t), mRow(0)
{
    if (cmd.empty()) throw runtime_error("Empty evaluation string.");
    setDebug(debug);
    istringstream in(cmd);
    if (getDebug()>2) cerr << "Parsing '" << cmd << "'...";
    Parse(in);
    if (getDebug() > 2) {
        cerr << " Done!" << endl;
	dump_rpn(cerr);
    }
}

//======================================  add a symbol
void
eval::addSymbol(const std::string& name, const std::string& value) {
    mDictionary[name] = value;
}

//======================================  Evaluate  string value for one row.
string 
eval::evalRow(int row) {
    mRow = row;
    crank();
    if (refStack().size() != 1) 
        throw runtime_error("Void or multi-valued result");
    return refStack().pop_string();
}


//======================================  Evaluate bool value for one row.
bool
eval::evalRow_bool(int row) {
    mRow = row;
    crank();
    if (refStack().size() != 1) 
        throw runtime_error("Void or multi-valued result");
    return refStack().pop_logical();
}

//======================================  Evaluate numeric value for one row.
double
eval::evalRow_numeric(int row) {
    mRow = row;
    crank();
    if (refStack().size() != 1) 
        throw runtime_error("Void or multi-valued result");
    return refStack().pop_numeric();
}

//======================================  Evaluate symbol values
void 
eval::push_symbol(const string& sym) {
    if (sym == "_row") {
        refStack().push(double(mRow));
    } else if (sym == "_next") {
        refStack().push(double(mRow+1 < mTable.getNRows()?mRow+1:0));
    } else if (sym == "_nrow") {
        refStack().push(double(mTable.getNRows()));
    } else if (sym == "_prev") {
        refStack().push(double(mRow!=0?mRow-1:mTable.getNRows()-1));
    } else if (sym == "_nextok") {
        refStack().push(mRow+1 < mTable.getNRows());
    } else if (sym == "_prevok") {
        refStack().push(mRow>0);
    } else if (sym == "true") {
        refStack().push(true);
    } else if (sym == "false") {
        refStack().push(false);
    } else if (mDictionary.find(sym) != mDictionary.end()) {
        refStack().push(mDictionary[sym]);
    } else {
        const table::Cell& Tij(mTable[sym].refCell(mRow));
        if (Tij.getType() == table::Cell::kNumeric) {
	    refStack().push(Tij.getNumeric());
	} else {
	    refStack().push(Tij.getString());
	}
    }
}

//======================================  Evaluate functions.
void 
eval::eval_funct(const string& func, int nArg) {
    if (nArg == 1) {
	if (func == "int") {
	    refStack().push(double(long(refStack().pop_numeric())));
	} 
	else if (func == "date") {
	    string datstr = TimeString(Time(long(refStack().pop_numeric())), 
				       "%Y/%02m/%02d %02H:%02N:%02S");
	    refStack().push(datstr);
	} 
	else if (func == "abs") {
	    refStack().push(fabs(refStack().pop_numeric()));
	} 
	else if (func == "log10") {
	    refStack().push(log10(refStack().pop_numeric()));
	} 
	else if (func == "sqrt") {
	    refStack().push(sqrt(refStack().pop_numeric()));
	} 
	else if (func == "type") {
	    str_token::datype t = refStack().type(1);
	    refStack().pop_string();
	    switch (t) {
	    case str_token::kString:
		refStack().push(string("string"));
		break;
	    case str_token::kNumeric:
		refStack().push(string("numeric"));
		break;
	    case str_token::kLogical:
		refStack().push(string("bool"));
		break;
	    case str_token::kNone:
		refStack().push(string("none"));
	    }
	} 
	else if (mTable.exists(func)) {
	    unsigned int irow = int(refStack().pop_numeric());
	    if (irow >= mTable.getNRows()) 
		throw range_error("Invalid row number");
	    refStack().push(mTable[func][irow].getString());
	} 
	else {
	    throw runtime_error(string("Unrecognized function:") + func);
	}
    } 
    else if (nArg == 2) {
	if (func == "min") {
	    double t1 = mStack.pop_numeric();
	    double t2 = mStack.pop_numeric();
	    if (t1 > t2) mStack.push(t2);
	    else         mStack.push(t1);
	} 
	else if (func == "max") {
	    double t1 = mStack.pop_numeric();
	    double t2 = mStack.pop_numeric();
	    if (t1 < t2) mStack.push(t2);
	    else         mStack.push(t1);
	} 
	else if (func == "date") {
	    string fmt = refStack().pop_string();
	    if (fmt.empty()) fmt = "%Y/%02m/%02d %02H:%02N:%02S";
	    Time gps(long(refStack().pop_numeric()));
	    string datstr = TimeString(gps, fmt.c_str());
	    refStack().push(datstr);
	} 
	else {
	    throw runtime_error(string("Unrecognized function:") + func);
	}
    } else {
	throw runtime_error("Invalid argument count");
    }
}

//======================================  Set dictionary
void 
eval::setDictionary(const dict_type& dict) {
    mDictionary = dict;
}
