/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "seg_iohdf5.hh"
#include "Time.hh"
#include <H5Cpp.h>
#include <iostream>
#include <sstream>
#include <stdexcept>
#include <cstring>

static const char* definer_name = "definers";
static const char* segment_name = "segments";

//======================================  Segment definer structure
struct hdf_segdef {
public:
    hdf_segdef(void) {}
    hdf_segdef(int defID, const std::string& name, int version);
    hdf_segdef(int defID, const segID& seg_id);
    int def_id(void) const;
    std::string name(void) const;
    segID seg_id(void) const;
    int version(void) const;
public:
    int _def_id;
    //std::string _name;
    //std::string _ifo;
    char _name[64];
    char _ifo[4];
    int _version;
};

inline
hdf_segdef::hdf_segdef(int defID, const std::string& name, int version) 
// : _def_id(defID), _name(name), _version(version)
    : _def_id(defID), _version(version)
{
    std::string::size_type cln = name.find(':');
    if (cln == std::string::npos) {
	strcpy(_name, name.c_str());
	_ifo[0] = 0;
    }
    else {
	strcpy(_ifo, name.substr(0, cln).c_str());
	strcpy(_name, name.substr(cln+1).c_str());
    }
}

inline
hdf_segdef::hdf_segdef(int defID, const segID& segid) 
    : _def_id(defID), _version(segid.version())
{
    strcpy(_ifo,  segid.ifo().c_str());
    strcpy(_name, segid.name().c_str());
}

inline int
hdf_segdef::def_id(void) const {
    return _def_id;
}

inline std::string
hdf_segdef::name(void) const {
    return _name;
}

inline segID
hdf_segdef::seg_id(void) const {
    return segID(_name, _ifo, _version);
}

inline int
hdf_segdef::version(void) const {
    return _version;
}

typedef std::vector<hdf_segdef> segdef_vect;

//======================================  Segment data structure
struct hdf_segdat {
public:
    hdf_segdat(void) {}
    hdf_segdat(int defID, int start, int end);
    int def_id(void) const;
    int start(void) const;
    int end(void) const;
public:
    int _def_id;
    int _start;
    int _end;
};

inline
hdf_segdat::hdf_segdat(int defID, int stime, int etime) 
    : _def_id(defID), _start(stime), _end(etime)
{}

inline int
hdf_segdat::def_id(void) const {
    return _def_id;
}

inline int
hdf_segdat::start(void) const {
    return _start;
}

inline int
hdf_segdat::end(void) const {
    return _end;
}
typedef std::vector<hdf_segdat> segdat_vect;

using namespace std;

#ifndef H5_NO_NAMESPACE
using namespace H5;
#endif

//======================================  Constructor
seg_iohdf5::seg_iohdf5(void)
    : mDefType(0), mSegType(0)
{}

//======================================  Destructor
seg_iohdf5::~seg_iohdf5(void)
{}

//======================================  
#define Offset(s,m) (long(&s.m)-long(&s))


//======================================  Define file structures
void
seg_iohdf5::defineTypes(void) {
    //----------------------------------  Define the segment definer type
    if (!mDefType) {
	//------------------------------  Get a structure to pluck offsets
	hdf_segdef _hdf_segdef;

	//------------------------------  Non-atomic data types
	StrType str_type64(PredType::C_S1);
	str_type64.setSize(64);
	str_type64.setStrpad(H5T_STR_NULLTERM);

	StrType str_type4(PredType::C_S1);
	str_type4.setSize(4);
	str_type4.setStrpad(H5T_STR_NULLTERM);

	//------------------------------  Build the composite type
	mDefType = new CompType( sizeof(hdf_segdef) );
	mDefType->insertMember("SegID", 
			       Offset(_hdf_segdef, _def_id), 
			       PredType::NATIVE_INT);
	mDefType->insertMember("Name", 
			       Offset(_hdf_segdef, _name[0]), 
			       str_type64);
	mDefType->insertMember("Ifo", 
			       Offset(_hdf_segdef, _ifo[0]), 
			       str_type4);
	mDefType->insertMember("Version", 
			       Offset(_hdf_segdef, _version),
			       PredType::NATIVE_INT);
    }

    //----------------------------------  Define the segment data type
    if (!mSegType) {
	//------------------------------  Get a structure to pluck offsets
	hdf_segdat _hdf_segdat;

	//------------------------------  Build the composite type
	mSegType = new CompType( sizeof(hdf_segdat) );
	mSegType->insertMember("SegID", 
			       Offset(_hdf_segdat, _def_id), 
			       PredType::NATIVE_INT);
	mSegType->insertMember("Start" , 
			       Offset(_hdf_segdat, _start), 
			       PredType::NATIVE_INT);
	mSegType->insertMember("End", 
			       Offset(_hdf_segdat, _end),
			       PredType::NATIVE_INT);
    }
}

//======================================  Read an HDF5 segment file
void 
seg_iohdf5::read_seg(const string& file, const segID& select, 
		     const string& format) {
    hsize_t dim;

    //----------------------------------  Define the segment definer data type
    defineTypes();

    //----------------------------------  Open the file
    H5File hd_file(file, H5F_ACC_RDONLY );

    //----------------------------------  Read in the definers
    DataSet def_ds(hd_file.openDataSet(definer_name));
    DataSpace dspace = def_ds.getSpace();
    // int ndims = 
    dspace.getSimpleExtentDims( &dim, NULL);
    if (!dim) return;
    segdef_vect defvect(dim); 
    def_ds.read(defvect.data(), *mDefType);

    //----------------------------------  Read in the segments
    DataSet seg_ds(hd_file.openDataSet(segment_name));
    DataSpace sspace = seg_ds.getSpace();
    // int ndims = 
    sspace.getSimpleExtentDims( &dim, NULL);
    segdat_vect segvect(dim); 
    seg_ds.read(segvect.data(), *mSegType);

    int nSegs = segvect.size();
    int def_id = 0;
    segID seg_id(defvect[0].seg_id());
    bool def_ok = seg_id.test(select);
    for (int i=0; i<nSegs; i++) {
	if (segvect[i].def_id() != def_id) {
	    def_id = segvect[i].def_id();
	    seg_id = defvect[def_id].seg_id();
	    def_ok = seg_id.test(select);
	}
	if (def_ok) {
	    int start = segvect[i].start();
	    int end   = segvect[i].end();
	    mSegMap[seg_id].stuff(LockSegment(0, Time(start), Time(end), 0));
	}
    }
}

//======================================  Write an HDF5 segment file
void
seg_iohdf5::write_seg(const seg_map& smap, const segID& select, 
		   const std::string& format, const std::string& file)
{
    //----------------------------------  Define the segment definer data type
    defineTypes();

    //----------------------------------  Get a list of segment types
    segdef_vect defvect;
    segdat_vect segvect;

    //-----------------------------------  Scan number of segments
    string svLast;
    for (const_seg_iter i=smap.begin(); i != smap.end(); ++i) {
	if (!i->first.test(select)) {
	    cout << "Rejected: " << i->first.full_name() 
		 << " filter: " << select.full_name() << endl;
	    continue;
	}
	size_t defID = defvect.size();
	for (size_t j=0; j < defID; ++j) {
	    if (i->first == defvect[j].seg_id()) {
		defID = j;
		break;
	    }
	}
	if (defID == defvect.size()) {
	    defvect.push_back(hdf_segdef(defID, i->first));
	}

	size_t N = i->second.size();
	segvect.reserve(segvect.size() + N);
	for (size_t j=0; j<N; ++j) {
	    hdf_segdat seg(defID, 
			   i->second[j].getStartTime().getS(), 
			   i->second[j].getEndTime().getS()
			   );
	    segvect.push_back(seg);
	}
    }

    if (defvect.empty() || segvect.empty()) {
	cerr << "No suitable segments definitions" << endl;
	return;
    }

    //-----------------------------------  Build the data space
    H5File hd_file(file, H5F_ACC_TRUNC);

    //-----------------------------------  Build the data space
    hsize_t nDefs = defvect.size();
    DataSpace dspace(1, &nDefs);

    //-----------------------------------  Write definers to the dataset;
    DataSet def_ds(hd_file.createDataSet(definer_name, *mDefType, dspace));
    def_ds.write(defvect.data(), *mDefType );

    //-----------------------------------  Build the data space
    hsize_t nSegs = segvect.size();
    DataSpace sspace(1, &nSegs);

    //-----------------------------------  Write definers to the dataset;
    DataSet seg_ds(hd_file.createDataSet(segment_name, *mSegType, sspace));
    seg_ds.write(segvect.data(), *mSegType );


}
