#include "ParseLine.hh"
#include "ReadTrend.hh"
#include <fstream>
#include <string>
#include <stdexcept>
#include <iostream>
#include <iomanip>

//======================================  AvgChan
template<class T>
class Ordinator {
public:
  typedef typename std::vector<T>              type_vector;
  typedef typename type_vector::iterator       type_iter;
  typedef typename type_vector::const_iterator const_type_iter;
  typedef unsigned long                        size_type;
public:
  Ordinator(size_type N=0);
  virtual ~Ordinator(void) {}
  void findMax(size_type N, const T* data);
  void findMin(size_type N, const T* data);
  void setSize(size_type N);
  T operator[](size_type i) const;
  void clear(void);
private:
  size_type   mSize;
  type_vector mVect;
};

template<class T>
inline
Ordinator<T>::Ordinator(size_type N) {
    setSize(N);
}

template<class T>
inline void
Ordinator<T>::clear(void) {
    mVect.clear();
}

template<class T>
inline T
Ordinator<T>::operator[](size_type i) const {
    if (i >= mVect.size()) throw std::out_of_range("argument > vector size");
    return mVect[i];
}

template<class T>
inline void
Ordinator<T>::findMax(size_type N, const T* v) {
    size_type vSiz = mVect.size();
    size_type inx0 = 0;
    if (!vSiz && N) {
       mVect.push_back(*v);
       inx0++;
       vSiz = mVect.size();
    }
    for (size_type i=inx0 ; i<N ; i++) {
        T x = v[i];
	if ( x > mVect[vSiz-1]) {
	    for (size_type j=0 ; j<vSiz ; j++) {
	        T y = mVect[j];
	        if (x > y) {
		    mVect[j] = x;
		    x = y;
		}
	    }
	}
	if (vSiz < mSize) {
	    mVect.push_back(x);
	    vSiz = mVect.size();
	}
    }
}

template<class T>
inline void
Ordinator<T>::findMin(size_type N, const T* v) {
    size_type vSiz = mVect.size();
    size_type inx0 = 0;
    if (!vSiz && N) {
       mVect.push_back(*v);
       inx0++;
       vSiz = mVect.size();
    }
    for (size_type i=inx0 ; i<N ; i++) {
        T x = v[i];
	if ( x < mVect[vSiz-1]) {
	    for (size_type j=0 ; j<vSiz ; j++) {
	        T y = mVect[j];
	        if (x < y) {
		    mVect[j] = x;
		    x = y;
		}
	    }
	}
	if (vSiz < mSize) {
	    mVect.push_back(x);
	    vSiz = mVect.size();
	}
    }
}

template<class T>
inline void
Ordinator<T>::setSize(size_type N) {
    mSize = N;
    mVect.reserve(N);
}

//======================================  AvgChan
class AvgChan {
public:
  AvgChan(const std::string& chan, const std::string& text);
  const char* getChannel(void) const;
  void gobble(const TSeries& ts);
  void print(std::ostream& out) const;
private:
  std::string mChannel;
  std::string mText;
  double mAvg;
  int mPct5;
  Ordinator<double> mMin;
  Ordinator<double> mMax;
};

inline const char*
AvgChan::getChannel(void) const {
    return mChannel.c_str();
}

//======================================  Averager class
class TrendAvg {
public:
  TrendAvg(int argc, const char* argv[]);
  ~TrendAvg(void);
  bool configure(const char* file);
  void doit(void);
  const TSeries& findTS(const std::string& name) const;
private:
  typedef std::vector<AvgChan>      chan_vect;
  typedef chan_vect::iterator       chan_iter;
  typedef chan_vect::const_iterator const_chan_iter;
private:
  bool mOK;
  std::string mFile;
  std::string mTrendDir;
  Time mStartGPS;
  Time mEndGPS;
  chan_vect mVect;
  ReadTrend::string_vect  mChans;
  ReadTrend::type_vect    mTypes;
  ReadTrend::tseries_vect mTSout;
  std::ostream* mOut;
  bool mDebug;
};

using namespace std;

//======================================  Main program
int 
main(int argc, const char* argv[]) {
    try {
        TrendAvg(argc, argv).doit();
    } catch (std::exception& e) {
        cerr << "Caught exception: " << e.what() << endl;
    } catch (...) {
        cerr << "Caught unidentified exception" << endl;
    }
}


//======================================  Average channel constructor
AvgChan::AvgChan(const std::string& chan, const std::string& text) 
  : mChannel(chan), mText(text), mAvg(0.0), mPct5(0)
{}

//======================================  Calculate statistics
void 
AvgChan::gobble(const TSeries& ts) {

    //----------------------------------  Get the data
    int N     = ts.getNSample();
    if (!N) return;
    double* x = new double[N];
    ts.getData(N, x);
    mAvg = *x;
    for (int i=1 ; i<N ; ++i) mAvg += x[i];
    mAvg /= double(N);

    //----------------------------------  Get the 95% limits
    mPct5  = (N+19) / 20;
    mMin.setSize(mPct5);
    mMin.findMin(N, x);

    mMax.setSize(mPct5);
    mMax.findMax(N, x);
    delete[] x;
}

//======================================  Print the results
void
AvgChan::print(std::ostream& out) const {
  out << mText << setw(15) << mAvg;
  if (mPct5) out << setw(15) << mMin[mPct5-1] << setw(15) << mMax[mPct5-1];
  out << endl;
}

//======================================  Trend average constructor
TrendAvg::TrendAvg(int argc, const char* argv[]) 
  : mOK(false), mTrendDir("/export/DMTtrends/DataQual"), mDebug(false)
{
    const char* conf = "TrendAvg.cfg";
    std::string out("-");
    bool syntax(false);
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
	if (argi == "-conf") {
	    conf = argv[++i];
	} else if (argi == "-debug") {
	    mDebug = true;
	} else if (argi == "-end") {
	    mEndGPS = Time(strtol(argv[++i], 0, 0));
	} else if (argi == "-out") {
	    out = argv[++i];
	} else if (argi == "-start") {
	    mStartGPS = Time(strtol(argv[++i], 0, 0));
	} else if (argi == "-td") {
	    mTrendDir = argv[++i];
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	}
    }
    if (syntax) {
        cerr << "Syntax: TrendAvg [-start <GPS.] [-end <GPS>] [-conf <file>]" 
	     << " [-out <file>] [-td <trend-dir>]" << endl;
        return;
    }
    if (configure(conf)) return;
    if (out == "-") mOut = &cout;
    else            mOut = new ofstream(out.c_str(), ios_base::out);
    mOK = true;
}

TrendAvg::~TrendAvg(void) {
}

bool
TrendAvg::configure(const char* file) {
    ParseLine pl(file);
    if (!pl.isOpen()) return true;
    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;
        if (nArg >= 2) mVect.push_back(AvgChan(pl[0], pl[1]));
	else           mVect.push_back(AvgChan(pl[0], pl[0]));
    }
    return false;
}

const TSeries&
TrendAvg::findTS(const string& file) const {
  unsigned long N=mChans.size();
    for (unsigned long i=0 ; i < N ; i++) {
        if (mChans[i] == file) return mTSout[i];
    }
    throw logic_error("Channel name not found");
}

void 
TrendAvg::doit(void) {
    if (!mOK) return;

    for (chan_iter i=mVect.begin() ; i != mVect.end() ; i++) {
        mChans.push_back(i->getChannel());
        mTypes.push_back(ReadTrend::kMean);
	mTSout.push_back(TSeries());
    }

    ReadTrend rt(mTrendDir.c_str());
    if (mDebug) {
        rt.setDebug();
	cout << "Getting data for GPS " << mStartGPS.getS() << "-" 
	     << mEndGPS.getS() << endl;
	cout << "Channel list: " << endl;
	for (unsigned int i=0 ; i<mChans.size() ; ++i) cout<< mChans[i]<< endl;
    }
    rt.getSeries(mChans, mTypes, mStartGPS, mEndGPS-mStartGPS, mTSout);

    for (chan_iter i=mVect.begin() ; i != mVect.end() ; i++) {
        i->gobble(findTS(i->getChannel()));
	i->print(*mOut);
    }
}
