/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef MATCHTRIG_HH
#define MATCHTRIG_HH

#include "Dacc.hh"
#include "SigFlag.hh"
#include "TrigBase.hh"
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrSimEvent.hh"

#include "TH1.h"
#include "TCanvas.h"
#include <vector>
#include "xsil/MetaIO.hh"

class  TFile;

//======================================  Efficiency plotting package
class Plotter {
public:
    Plotter(const char* title);
    ~Plotter(void);
    void hPlot(const TH1D& h, const char* opt=0);
    void avgPlot(const TH1D& n, const TH1D& x, const TH1D& xx, const char* t=0);
    void effPlot(const TH1D& n, const TH1D& d, const char* t=0);
    void resPlot(const TH1D& n, const TH1D& x, const TH1D& xx, const char* t=0);

    TH1D getSig(const TH1D& n, const TH1D& x, const TH1D& xx);

    void update(void);
    void setFile(const char* file);
private:
    std::string mFile;
    TCanvas mCanvas;
};

double getParVal(const FrameCPP::FrSimEvent& e, const std::string& name);

/*  {\large {\bf Introduction}} \\
 *  
 *  @memo Triggered data processing base class.
 *  @author John G. Zweizig
 *  @version 1.0; Last Modified January 24, 2004
 */
class MatchTrig {
public:
    /*  The class constructor gets all the command line arguments in the 
     *  usual Unix/C argument list format. It sets up the environment 
     *  for processing and opens the frame input stream as directed by 
     *  the command line arguments.
     *  @memo Class constructor.
     *  @param argc Number of arguments specified.
     *  @param argv Pointers to arguments.
     */
    MatchTrig(int argc, const char *argv[]);

    /**  Close the input stream and go away.
     *  @memo Class destructor.
     */
    virtual ~MatchTrig(void);

    /*  #MainLoop# reads in data and invokes #ProcessData# for each data 
     *  epoch successfully read in. #MainLoop# exits when a SIGINT or 
     *  SIGTERM signal is caught, or when #finish# is called by a subsidiary 
     *  function (usually by #ProcessData#).This function is provided by the 
     *  base class.
     *  @memo Loop over data epochs.
     */
    void MainLoop(void);

    /*  #finish# stops processing by setting a flag that breaks the main 
     *  loop. Note that #finish# returns to the calling function (generally 
     *  the user's #ProcessData#) which must exit normally for the process 
     *  to terminate.
     *  @memo Stop data processing and go away cleanly.
     */
    void finish(void) {mActive = false;}

    /*  #Debug# returns the debug level as specified by the "-debug nnn" 
     *  argument. If "-debug" isn't specified in the command line the 
     *  debug level defaults to 0.
     *  @memo Get the debug level.
     */
    int Debug(void) const {return mDebug;}

    /*  Return a constant reference to the data accessor.
     *  @memo Get the data accessor.
     */
    const Dacc& getDacc(void) const {return mIn;}

    /*  Get the data stride. The data stride is the length of data to be 
     *  read for each call to ProcessData.
     *  @memo Get the data stride.
     *  @return The data stride interval.
     */
    const Interval getStride(void) const {return mStride;}

    /*  Test if the process is active. This is returne false if 
     *  process termination is underway.
     *  @return true if terimation is underway.
     */
    bool isActive(void) const {return mActive;}

    /*  Read in a trigger.
     */
    int readTrigger(void);
    int readGDSTrigger(void);
    int readSingleBurst(void);

    /*  Return a reference to the data accessor.
     *  @memo Get the data accessor.
     */
    Dacc& getDacc(void) {return mIn;}
 
    //--------------------------------------  Member classes
private:
    class ParHist {
    public:
	ParHist(const std::string& name, double xMin, double xMax);
	ParHist(const ParHist& p);
	~ParHist(void);
	ParHist& operator=(const ParHist& p);
	double getParVal(const FrameCPP::FrSimEvent& e) const;
	void   histEvent(const FrameCPP::FrSimEvent& e);
	void   histTrig(const FrameCPP::FrSimEvent& e, double dt, double dA, double dF);
	void   writeHistos(TFile& f);
	void   genPlots(Plotter& c);
    private:
	std::string _name;
	double _xMin;
	double _xMax;
	TH1D*  _EventHist;
	TH1D*  _TrigHist;
	TH1D*  _dAHist;
	TH1D*  _dtHist;
	TH1D*  _dFHist;
	TH1D*  _sAHist;
	TH1D*  _stHist;
	TH1D*  _sFHist;
    };
    typedef std::vector<ParHist> ParList;
    typedef ParList::iterator par_iter;
    typedef ParList::const_iterator const_par_iter;

    class trigEvt {
    public:
	trigEvt(const trig::TrigBase& t);
	~trigEvt(void) {}
	double getDtLast(void) const;
	double getEvtAmpl(void) const;
	double getEvtFreq(void) const;
	double getFDiff(void) const; 
	double getSNR(void) const; 
	double getTDiff(void) const;
	double getTDiff(const FrameCPP::FrSimEvent& e) const;
	double getTrigAmpl(void) const;
	double getTrigFreq(void) const;
	Time   getTrigTime(void) const;
	bool   isValid(void) const;
	bool   fValid(void) const;
	const  trig::TrigBase& refTrig(void) const;
	const  FrameCPP::FrSimEvent& refEvent(void) const;
	void setEvent(const FrameCPP::FrSimEvent& e, const Time& tLast);

    private:
	trig::TrigBase mTrig;
	FrameCPP::FrSimEvent mEvent;
	bool mValidEvt;
	double tDiff;
	double dtLast;
    };

    typedef std::vector<trigEvt> TriggerList;
    typedef TriggerList::const_iterator const_trig_iter;
    typedef TriggerList::iterator trig_iter;

private:
    ///  Internal run flag
    bool     mActive;

    ///  Frame Input object
    Dacc  mIn;

    ///  Length of epoch.
    Interval mEpoch;

    ///  Offset of epoch from trigger time.
    Interval mOffset;

    ///  Length of stride.
    Interval mStride;

    ///  Matching mode
    enum match_mode {
	best_time,
	best_snr
    } matchMode;

    ///  Current trigger
    trig::TrigBase mTrigger;

    SigFlag mTerm;
  
    ///  Debug Level
    int  mDebug;
  
    ///  Number of triggers read
    int  mTrigRead;
  
    ///  Number of triggers read
    int  mTrigSelect;
  
    ///  Number of triggers processed
    int  mTrigProc;
  
    ///  Number of processed strides
    int  mTrigMatch;

    ///  metaio package environment structure.
    xsil::MetaIO metaio;
    enum {
	kNone,
	kGDSTrig,
	kSngl_burst} mTableType;

    TriggerList mTrigList;

    //-------------------------  Histograms
    enum HistID {
	hEventAmpl,
	hEffvsAmpl,
	hTrigAmpl,
	hDeltaT,
	hDeltaTvsA,
	hDeltaTSqvsA,
	hAmplvsA,
	hDeltaFvsA,
	hDeltaFSqvsA,
	hEventDt,
	hEffvsDt,
	hEffvsF,
	hDeltaFvsF,
	hDeltaFSqvsF,
	hNumHist
    };
    TH1D* mHisto[hNumHist];
    std::string mHistOut;

    Plotter mPlot;

    ParList mParList;

    //---------------------------  Time diff parameters
    double maxDiff;            //  Max delta-t for trig/event match
    double tLimit;             //  Max delta-t histogrammed.
    double dtmax;              //  maximum trigger time delta histogrammed.
    Time   tLast;
};

//======================================  inline functions.
#include "ldas/ldasconfig.hh"
#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif
inline Time
GDSTime(const GPSTime& t) {
    return Time(t.GetSeconds(), t.GetNanoseconds());
}

inline double 
MatchTrig::trigEvt::getTDiff(const FrameCPP::FrSimEvent& e) const {
    return double(mTrig.getPeakTime() - GDSTime(e.GetGTime()));
}

inline double 
MatchTrig::trigEvt::getTrigAmpl(void) const {
    return mTrig.getIntensity();
}

inline double 
MatchTrig::trigEvt::getDtLast(void) const {
    return dtLast;
}

inline double 
MatchTrig::trigEvt::getEvtAmpl(void) const {
    return mEvent.GetAmplitude();
}

inline bool 
MatchTrig::trigEvt::isValid(void) const {
    return mValidEvt;
}

inline double 
MatchTrig::trigEvt::getTDiff(void) const {
    return tDiff;
}

inline const trig::TrigBase& 
MatchTrig::trigEvt::refTrig(void) const {
    return mTrig;
}

inline Time
MatchTrig::trigEvt::getTrigTime(void) const {
    return mTrig.getTime();
}

inline double
MatchTrig::trigEvt::getTrigFreq(void) const {
    return mTrig.getFrequency();
}

inline const FrameCPP::FrSimEvent& 
MatchTrig::trigEvt::refEvent(void) const {
    return mEvent;
}

#endif     //  MatchTrig_HH
