/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Framecmp compares the data portion of two frames.
//


#include "Dacc.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrVect.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrAdcData.hh"
#include <string>
#include <iostream>
#include <iomanip>
#include <map>

using namespace std;
using FrameCPP::FrameH;
using FrameCPP::FrAdcData;
using FrameCPP::FrRawData;
using FrameCPP::FrProcData;
using FrameCPP::FrVect;

#ifdef FCPP_SHARED_PTRS
static frvect_pointer vnull;
#else
static frvect_pointer vnull = 0;
#endif

typedef pair<frvect_pointer, frvect_pointer> ListData;
typedef map<std::string, ListData> ScanList;
typedef ScanList::iterator ScanIter;

//======================================  Dump data from two FrVects
void 
dump(frvect_pointer v, int ov=0, frvect_pointer x=vnull, 
     int ox=0, int nByte=0, bool diff_only=true) {
    int nData = v->GetNData();
    if (nByte) {
        switch (v->GetType()) {
	case FrVect::FR_VECT_2S:
	    nData = nByte/2;
	    break;
	case FrVect::FR_VECT_4S:
	case FrVect::FR_VECT_4U:
	    nData = nByte/4;
	    break;
	case FrVect::FR_VECT_4R:
	    nData = nByte/4;
	    break;
	case FrVect::FR_VECT_8R:
	    nData = nByte/8;
	    break;
	default:
	    return;
	}
    }
    for (int i=0 ; i<nData ; i+=8) {
        int nMax = i+8;
	if (nMax > nData) nMax = nData;

	//------------------------------  Test for differences
#ifdef FCPP_SHARED_PTRS
	const char* d  = ((const char*)v->GetData().get()) + ov;
#else
	const char* d  = ((const char*)v->GetData()) + ov;
#endif
	bool diff = false;
	if (x) {
#ifdef FCPP_SHARED_PTRS
	    const char* dx = ((const char*)x->GetData().get()) + ox;
#else
	    const char* dx = ((const char*)x->GetData()) + ox;
#endif
	    switch (v->GetType()) {
	    case FrVect::FR_VECT_2S:
	        for (int j=i ; j<nMax ; j++) {
		    diff |= *((short*)d + j) != *((short*)dx + j);
		}
		break;
	    case FrVect::FR_VECT_4S:
	    case FrVect::FR_VECT_4U:
	        for (int j=i ; j<nMax ; j++) {
		    diff |= *((int*)d + j) != *((int*)dx + j);
		}
		break;
	    case FrVect::FR_VECT_4R:
	        for (int j=i ; j<nMax ; j++) {
		    diff |= *((float*)d + j) != *((float*)dx + j);
		}
		break;
	    case FrVect::FR_VECT_8R:
	        for (int j=i ; j<nMax ; j++) {
		    diff |= *((double*)d + j) != *((double*)dx + j);
		}
		break;
	    default:
	        return;
	    }
	    if (diff)            cout << "|";
	    else if (!diff_only) cout << " ";
	}

	//------------------------------  Dump the values
	if (diff || !diff_only) {
	    cout << setw(6) << i;
	    for (int j=i ; j<nMax ; j++) {
	        switch (v->GetType()) {
		case FrVect::FR_VECT_2S:
		    cout << setw(9) << *((short*)d + j);
		    break;
		case FrVect::FR_VECT_4S:
		case FrVect::FR_VECT_4U:
		    cout << setw(9) << *((int*)d + j);
		    break;
		case FrVect::FR_VECT_4R:
		    cout << setw(9) << *((float*)d + j);
		    break;
		case FrVect::FR_VECT_8R:
		    cout << setw(9) << *((double*)d + j);
		    break;
		default:
		    return;
		}
	    }
	    cout << endl;
	}
    }
}

//======================================  Frame comparison main function
int
main(int argc, const char* argv[]) {
    int  verbose = 0;
    bool exact   = false;
    bool syntax  = false;
    bool dumpall = false;

    for (int i=1 ; i<argc-2 ; i++) {
        string argi = argv[i];
        if (argi == "-v") {
	    verbose++;
	} else if (argi == "-e") {
	    exact = true;
	} else if (argi == "-a") {
	    dumpall = true;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }

    if (syntax || argc < 3 || *argv[argc-2] == '-' || *argv[argc-1] == '-') { 
        cerr << "Command Syntax: " << endl;
	cerr << "framecmp [-v] [-e] [-a] <file1> <file2>" << endl;
	return 1;
    }

    Dacc In1;
    In1.setTOCMode(false);
    In1.addFile(argv[argc-2]);
    Dacc In2;
    In2.setTOCMode(false);
    In2.addFile(argv[argc-1]);

    bool file_1_OK = In1.synch() == 0;
    if (!file_1_OK) {
        cerr << "framecmp: Unable to open file 1" << endl;
        return 1;
    }
    bool file_2_OK = In2.synch() == 0;
    if (!file_2_OK) {
        cerr << "framecmp: Unable to open file 2" << endl;
        return 2;
    }

    while (file_1_OK && file_2_OK) {
        //------------------------------  Check frames have same start time
        Time time1 = In1.getCurrentTime();
        Time time2 = In2.getCurrentTime();
	Interval dT1 = In1.getDt();
	Interval dT2 = In2.getDt();
	Time stop1 = time1 + dT1;
	Time stop2 = time2 + dT2;
        if (verbose>1) cout << "Frame times, File 1: " << time1.getS()
			    << ":" << dT1 << " File2: " << time2.getS() 
			    << ":" << dT2 << endl;

	Time tStart = (time2 > time1) ? time2 : time1;
	Time tStop  = (stop1 < stop2) ? stop1 : stop2;

	if (exact && (dT1 != dT2 || Almost(time1, time2))) {
	    cout << "Exact comparison not possible with different sized frames"
		 << endl;
	    return 2;
	}
	if (tStart > tStop || Almost(tStart, tStop)) {
	    if (stop1 < stop2) {
		cout << "Skipping frame from file 1." << endl;
		file_1_OK = !In1.nextFrame();
		continue;
	    } else {
		cout << "Skipping frame from file 2." << endl;
		file_2_OK = !In2.nextFrame();
		continue;
	    }
	}

        if (verbose) cout << "Comparing file: " << In1.getFile() 
                          << " with file: " << In2.getFile() << endl;

        //------------------------------  Build a list of channel names
        ScanList al;
        frrawdata_pointer raw(In1.getFrame()->GetRawData());
	if (!raw) {
	    if (verbose) cout << "File 1 has no raw data." << endl;
	} else {
	    FrRawData::const_firstAdc_iterator adcIter=raw->RefFirstAdc().begin();
	    for ( ; adcIter != raw->RefFirstAdc().end() ; adcIter++ ) {
	        fradcdata_pointer ap(*adcIter);
		al[ap->GetName()] = ListData(ap->RefData().front(), vnull);
		if (verbose > 3) cout << "Add channel " << ap->GetName()
				      << " from file 1" << endl;
	    }
	}
	for (FrameH::const_procData_iterator 
	       i=In1.getFrame()->RefProcData().begin() ;
	     i != In1.getFrame()->RefProcData().end() ; i++) {
	    al[(*i)->GetName()] = ListData((*i)->RefData().front(), vnull);
	    if (verbose > 3) cout << "Add channel " << (*i)->GetName()
				  << " from file 1" << endl;
	}

        raw = In2.getFrame()->GetRawData();
	if (!raw) {
	    cout << "File 2 has no raw data." << endl;
	} else {
	    FrRawData::const_firstAdc_iterator adcIter = raw->RefFirstAdc().begin();
	    for ( ; adcIter != raw->RefFirstAdc().end() ; adcIter++ ) {
	        fradcdata_pointer ap(*adcIter);
		string name = ap->GetName();
		ScanIter li = al.find(name);
		if (li == al.end()) al[name] = ListData(vnull, ap->RefData().front());
		else                li->second.second = ap->RefData().front();
		if (verbose > 3) cout << "Add channel " << name
				      << " from file 2" << endl;
	    }
	}
	for (FrameH::const_procData_iterator 
	       i=In2.getFrame()->RefProcData().begin() ;
	     i != In2.getFrame()->RefProcData().end() ; i++) {
	    string name = (*i)->GetName();
	    ScanIter li = al.find(name);
	    if (li == al.end()) al[name]=ListData(vnull, (*i)->RefData().front());
	    else                li->second.second = (*i)->RefData().front();
	    if (verbose > 3) cout << "Add channel " << name
				  << " from file 2" << endl;
	}

        //------------------------------  Scan
        int nMiss1   = 0;
        int nMiss2   = 0;
        int nTotChan = 0;
        int nCmpFail = 0;
        for (ScanIter li=al.begin() ; li != al.end() ; li++) {
            nTotChan++;
            if (!li->second.first) {
                if (verbose>1) cout << " Channel: " << li->first 
                                    << " not in frame 1" << endl;
                nMiss1++;
            } else if (!li->second.second) {
                if (verbose>1) cout << " Channel: " << li->first 
                                    << " not in frame 2" << endl;
                nMiss2++;
            } else {
                frvect_pointer vec1(li->second.first);
                frvect_pointer vec2(li->second.second);
                if (!exact || vec1->GetCompress() != vec2->GetCompress()) {
                    vec1->Uncompress();
                    vec2->Uncompress();
                }

		int b1off=0, b2off=0;
		int nbyt1 = vec1->GetNBytes();
		int nbyt2 = vec2->GetNBytes();
		if (!exact) {
		    b1off = int(nbyt1*double(tStart-time1)/dT1+0.5);
		    nbyt1 = int(nbyt1*double(tStop -time1)/dT1+0.5)-b1off;
		    b2off = int(nbyt2*double(tStart-time2)/dT2+0.5);
		    nbyt2 = int(nbyt2*double(tStop -time2)/dT2+0.5)-b2off;
		}
		if (nbyt1 != nbyt2) {
		    cout << "Data length differs for Channel: " 
			 << li->first << " File 1 length: " << nbyt1
			 << " File 2 length: " << nbyt2 << endl;
		    nCmpFail++;
		    continue;
		}
		if (verbose > 3) {
		    cout << "Comparing channel " << li->first 
			 << " offsets = " << b1off << "," << b2off 
			 << " length = " << nbyt1 << endl;
		}
		if (
#ifndef FCPP_SHARED_PTRS
		    memcmp(vec1->GetData()+b1off, 
			   vec2->GetData()+b2off, nbyt1)
#else
		    memcmp(vec1->GetData().get()+b1off, 
			   vec2->GetData().get()+b2off, nbyt1)
#endif
		    ) {
                    cout << "Data for Channel: " << li->first << " differs." 
                         << endl;
                    nCmpFail++;
                    if (verbose > 1) {
			cout << "Data at " << tStart << " from file " 
			     << In1.getFile() << endl;
		        dump(vec1, b1off, vec2, b2off, nbyt1, !dumpall);
			cout << "Data at " << tStart << " from file " 
			     << In2.getFile() << endl;
			dump(vec2, b2off, vec1, b1off, nbyt1, !dumpall);
		    }
		}
	    }
	}

	//------------------------------  Print statistics
	if (verbose) {
	    cout << "Frame comparison statistics: " << endl;
	    cout << "Combined channels found:       " << nTotChan << endl;
	    cout << "Channels missing from frame 1: " << nMiss1   << endl;
	    cout << "Channels missing from frame 2: " << nMiss2   << endl;
	    cout << "Comparison failures:           " << nCmpFail << endl;
	} else {
	    cout << "GPS: " << time1 << " chans-combined: " << nTotChan
		 << " miss-1: " << nMiss1 << " miss-2: " << nMiss2
		 << " failed: " << nCmpFail << endl;
	}

        //------------------------------  Go On to the next frames.
	if (tStop == time1+dT1) file_1_OK = In1.nextFrame() == 0;
	if (tStop == time2+dT2) file_2_OK = In2.nextFrame() == 0;
    }
    return 0;
}
