/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Dacc.hh"
#include "SigFlag.hh"
#include "TrigBase.hh"
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrSimEvent.hh"

#include <iostream>
#include <fstream>

double getParVal(const FrameCPP::FrSimEvent& e, const std::string& name);

/**  {\large {\bf Introduction}} \\
  *  
  *  @memo Triggered data processing base class.
  *  @author John G. Zweizig
  *  @version 1.1; Last Modified October 11, 2007
  */
class trigTest {
public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  @memo Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
    trigTest(int argc, const char *argv[]);

    /**  Close the input stream and go away.
      *  @memo Class destructor.
      */
    virtual ~trigTest(void);

    /**  #MainLoop# reads in data and invokes #ProcessData# for each data 
      *  epoch successfully read in. #MainLoop# exits when a SIGINT or 
      *  SIGTERM signal is caught, or when #finish# is called by a subsidiary 
      *  function (usually by #ProcessData#).This function is provided by the 
      *  base class.
      *  @memo Loop over data epochs.
      */
    void MainLoop(void);

    /**  #finish# stops processing by setting a flag that breaks the main 
      *  loop. Note that #finish# returns to the calling function (generally 
      *  the user's #ProcessData#) which must exit normally for the process 
      *  to terminate.
      *  @memo Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  #Debug# returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  @memo Get the debug level.
      */
    int Debug(void) const {return mDebug;}

    /**  Return a constant reference to the data accessor.
      *  @memo Get the data accessor.
      */
    const Dacc& getDacc(void) const {return mIn;}

    /**  Get the data stride. The data stride is the length of data to be 
      *  read for each call to ProcessData.
      *  @memo Get the data stride.
      *  @return The data stride interval.
      */
    Interval getStride(void) const {return mStride;}

    /**  Test for a trigTest command line argument. All trigTest command line 
      *  arguments take a value in the following argument position.
      *  @memo Test if string is a trigTest command line argument.
      *  @return true if the specified string is a trigTest argument.
      */
    bool isTrigTestArg(const char* arg) const;
    bool isActive(void) const {return mActive;}

    /**  Return a reference to the data accessor.
      *  @memo Get the data accessor.
      */
    Dacc& getDacc(void) {return mIn;}
 
//--------------------------------------  Member data
private:
    ///  Internal run flag
    bool     mActive;

    ///  Frame Input object
    Dacc  mIn;

    ///  Length of stride.
    Interval mStride;

    SigFlag mTerm;
  
    ///  Debug Level
    int  mDebug;

    ///  Number of triggers processed
    int  mTrigProc;

    ///  Output stream
    std::ostream* mOutput;
 };

//======================================  inline functions.
#include "ldas/ldasconfig.hh"
#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif

inline Time
GDSTime(const GPSTime& t) {
    return Time(t.GetSeconds(), t.GetNanoseconds());
}

#include "Interval.hh"
#include "Time.hh"
#include "ParseLine.hh"
#include <stdexcept>
#include <string>

using FrameCPP::FrSimEvent;
using FrameCPP::FrameH;
using namespace std;

//=====================================  Main program
int main(int argc, const char* argv[]) {
    trigTest testPgm(argc,argv);
    if (testPgm.isActive()) testPgm.MainLoop();
    return 0;
}


double getParVal(const FrameCPP::FrSimEvent& e, const std::string& name) {
    const FrSimEvent::ParamList_type& l(e.GetParam() );
    for (FrSimEvent::ParamList_type::const_iterator i=l.begin(); i!=l.end(); i++) {
      if (i->first == name) return i->second;
    }
    throw std::runtime_error("Nonexistant parameter requested");
}


//======================================  trigTest constructor.
trigTest::trigTest(int argc, const char *argv[]) 
    : mActive(false), mDebug(0), mTrigProc(0), mOutput(0)
{

    //----------------------------------  Parse the arguments
    bool syntax=false;
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-debug" || argi == "--debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	    mIn.setDebug(mDebug);
	} else if (argi == "-infile") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -infile." << endl;
		syntax= true;
	        break;
	    }
	    mIn.addFile(argv[i]);
	} else if (argi == "-inlist") {
	    if (++i >= argc) {
	        cerr << "No argument supplied for -inlist." << endl;
		syntax= true;
	        break;
	    }
	    ParseLine pl(argv[i]);
	    if (!pl.isOpen()) {
	        cerr << "Unable to open file " << argv[i] << endl;
		syntax = true;
		continue;
	    }
	    while (pl.getLine() >= 0) {
	        if (pl.getCount() > 0) mIn.addFile(pl[0]);
	    }
	} else if (argi == "-o" || argi == "--output") {
	    if (i == argc-1 || *(argv[i+1]) == '-') {
		cerr << "Missing output file" << endl;
		syntax = true;
	    } else {
		ofstream* out = new ofstream(argv[++i]);
		if ( ! (out && out->is_open())) {
		    cerr << "Unable to open output file: " << argv[i] << endl;
		    syntax = true;
		} else {
		    mOutput = static_cast<ostream*>(out);
		}
	    }
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Get the table file
    if (syntax) {
        cerr << "Error in command line. Syntax is:" << endl;
	cerr << "SimList {-infile <files> | -inlist <list-file>} " 
	     << "[-debug <debug-level>]" << endl;
	cerr << "        [{--output | -o} <out-file>]" << endl;
	finish();
	return;
    }

    //----------------------------------  Default output to cout
    if (!mOutput) mOutput = &cout;

    //----------------------------------  Read entire data.
    mIn.setTOCMode(false);

    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    //----------------------------------  Open a frame stream (FrameCPP)
    mActive = true;
}

//======================================  trigTest object destructor.
trigTest::~trigTest(void) {

    //----------------------------------  DMT has terminater
    cout << "Events read:   " << mTrigProc   << endl;
 
    //----------------------------------  Close the Input file/partition.
    mIn.close();
}

//======================================  Main processing loop function.
void
trigTest::MainLoop(void) {

    //----------------------------------  Loop until terminated.
    while(mActive && !mTerm) {

        //------------------------------  Check for an attention interrupt.
        int rc = mIn.nextFrame();
	if (rc) {
	    mActive = false;
	    continue;
	}
	frameh_pointer frame(mIn.getFrame());

	//------------------------------  Loop over events
	for (FrameH::const_simEvent_iterator i=frame->RefSimEvent().begin();
	     i != frame->RefSimEvent().end() && !mTerm ; i++) {
	    *(mOutput) << (*i)->GetGTime() << " " << (*i)->GetComment() 
		       << endl;
	    mTrigProc++;
	}
    }
}
