//////////////////////////////////////////////////////////////////////////
//  									//
//  XsilEvent								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include "events/XsilEvent.hh"
#include "events/Iterator.hh"
#include "events/Event.hh"
#include "events/Layout.hh"
#include "events/LayoutInfo.hh"
#include "events/Column.hh"
#include "events/Value.hh"
#include "events/IfoSet.hh"
#include "events/Name.hh"

#include <strings.h>
#include <iostream>

namespace xml {
   using namespace std;
   using namespace events;

//______________________________________________________________________________
   xsilEvent::xsilEvent (const char* name, const events::Event& event, 
                     int level)
   : xsil_base (name, level), 
   fBeg (StdIteratorImp<const Event*>(&event)),
   fEnd (StdIteratorImp<const Event*>(&event + 1)), fMax (-1)
   {
      fCur = fBeg;
   }

//______________________________________________________________________________
   xsilEvent::xsilEvent (const char* name, const events::Event* event, 
                     int n, int level)
   : xsil_base (name, level), 
   fBeg (StdIteratorImp<const Event*>(event)),
   fEnd (StdIteratorImp<const Event*>(event + n)), fMax (-1)
   {
      fCur = fBeg;
   }

//______________________________________________________________________________
   std::ostream& xsilEvent::write (std::ostream &os) const 
   {
      typedef std::pair <ColumnType::Enum, Column> column_item;
      typedef vector<column_item> column_list;
   
      fCur = fBeg;
      int n = fMax;
      while (((fMax < 0) || (n > 0)) && (fCur != fEnd)) {
         // Create new table
         Type type;
         if (!fCur->GetLayout().GetType (type)) {
            return os;
         }
         string tname = "gds:events:";
         tname += type.GetName();
         os << xsilTableBegin (fName, tname.c_str(), fLevel) << endl;
         os << xsilComment ("Event Analysis Tool", fLevel + 1) << endl;
         // Create column information
         column_list cols;
         for (ColumnInfoList::const_iterator i = 
             fCur->GetLayout().GetColumnList().begin() + 4;
             i != fCur->GetLayout().GetColumnList().end(); ++i) {
            if ((i->GetType() == ColumnType::kInt) ||
               (i->GetType() == ColumnType::kReal) || 
               (i->GetType() == ColumnType::kComplex) || 
               (i->GetType() == ColumnType::kTime) || 
               (i->GetType() == ColumnType::kString) || 
               (i->GetType() == ColumnType::kEvent)) {
               cols.push_back (column_item 
                              (i->GetType(), Column (i->GetName())));
            }
         }
         // Write first few fixed columns
         os << xsilTableColumn<const char*> 
            (ColumnType::kColumnNameName, fLevel + 1) << endl;   
         os << xsilTableColumn<Time> 
            (ColumnType::kColumnTimeName, fLevel + 1) << endl;   
         os << xsilTableColumn<const char*> 
            (ColumnType::kColumnIfoName, fLevel + 1) << endl;   
         // Write rest of column information
         for (column_list::const_iterator i = cols.begin();
             i != cols.end(); ++i) {
            switch (i->first) {
               case ColumnType::kInt:
                  os << xsilTableColumn<int> 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
               case ColumnType::kReal:
                  os << xsilTableColumn<double> 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
               case ColumnType::kComplex:
                  os << xsilTableColumn<std::complex<double> > 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
               case ColumnType::kTime:
                  os << xsilTableColumn<Time> 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
                  break;
               case ColumnType::kString:
                  os << xsilTableColumn<const char*> 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
               case ColumnType::kEvent:
                  os << xsilTableColumn<xsil_table> 
                     (i->second.GetName(), fLevel + 1) << endl;
                  break;
               default:
                  break;
            }
         }
         // Start writing table data
         os << xsilTableDataBegin ("Events", fLevel + 1) << endl;
         // Now write all events of the same type
         bool notfinished;
         do {
            os << xsilIndent (fLevel + 2);
            // Write first few fixed columns
            os << xsilTableEntry<const char*> (fCur->GetName().GetName())
               << xsilTableEntryDelimiter();
            os << xsilTableEntry<Time>(fCur->GetTime()) 
               << xsilTableEntryDelimiter();
            os << xsilTableEntry<const char*>(fCur->GetIfoStr().c_str());
            if (!cols.empty()) {
               os << xsilTableEntryDelimiter();
            }
            // Write column values
            bool eventcol = false;
            for (column_list::const_iterator i = cols.begin();
                i != cols.end(); ) {
               switch (i->first) {
                  // Int
                  case ColumnType::kInt:
                     {
                        Value val;
                        int x;
                        if ((i->second.Get (*fCur, val)) &&
                           (val.Write (x))) {
                           os << xsilTableEntry<int>(x);
                        }
                        break;
                     }
                  // Real
                  case ColumnType::kReal:
                     {
                        Value val;
                        double x;
                        if ((i->second.Get (*fCur, val)) &&
                           (val.Write (x))) {
                           os << xsilTableEntry<double>(x);
                        }
                        break;
                     }
                  // Complex
                  case ColumnType::kComplex:
                     {
                        Value val;
                        std::complex<double> x;
                        if ((i->second.Get (*fCur, val)) &&
                           (val.Write (x))) {
                           os << xsilTableEntry<complex<double> >(x);
                        }
                        break;
                     }
                  // Time
                  case ColumnType::kTime:
                     {
                        Value val;
                        Time x;
                        if ((i->second.Get (*fCur, val)) &&
                           (val.Write (x))) {
                           os << xsilTableEntry<Time>(x);
                        }
                        break;
                     }
                  // String
                  case ColumnType::kString:
                     {
                        Value val;
                        std::string x;
                        if ((i->second.Get (*fCur, val)) &&
                           (val.Write (x))) {
                           os << xsilTableEntry<const char*> 
                              (x.c_str(), x.size());
                        }
                        break;
                     }
                  // Event
                  case ColumnType::kEvent:
                     {
                        const Event* x = i->second.GetEvent (*fCur);
                        if (x) {
                           os << endl << xsilEvent (i->second.GetName(), 
                                                *x, fLevel + 3);
                           eventcol = true;
                        }
                        break;
                     }
                  default:
                     break;
               }
               ++i;
               if (i != cols.end()) {
                  os << xsilTableEntryDelimiter();
                  if (eventcol) os << endl << xsilIndent (fLevel + 3);
               }
            }
            // go to next event in list
            if (fMax >= 0) --n;
            ++fCur;
            // last event?
            notfinished = (((fMax < 0) || (n > 0)) && (fCur != fEnd));
            // check type of new event
            if (notfinished) {
               Type curtype;
               if (!fCur->GetLayout().GetType (curtype)) {
                  break;
               }
               // need a new table if not of the same type
               if (curtype != type) notfinished = false;
            }
            // Add a delimiter if more events are witten
            if (notfinished) {
               os << xsilTableEntryDelimiter();
            }
            os << endl;
         } while (notfinished);
         // Write table trailer
         os << xsilTableDataEnd (fLevel + 1) << endl;
         os << xsilTableEnd (fLevel);
      }
      return os;
   }

//______________________________________________________________________________
   xsilHandlerEvent::xsilHandlerEvent (const attrlist *attr,
                     xsilEventAdd* parent)
   : fParent (parent), fEventRow (0), fEventCol (0)
   {
      // get the type information
      attrlist::const_iterator type = attr->find (xmlType);
      if (type != attr->end()) {
         std::string s = type->second.c_str();
      
         string::size_type pos;
         if ((pos = s.find_last_of (":")) != string::npos) {
            s.erase (0, pos + 1);
         }
         if (!s.empty()) {
            fLayout.SetType (Type (s.c_str()));
            return;
         }
      }
   
      // Check Name attribute.
      attrlist::const_iterator name = attr->find (xmlName);
      if (name != attr->end()) {
      
         std::string line = name->second + ":";
         std::vector<string> element;
         std::string::size_type pos, prev_pos;
         pos = prev_pos = 0;
      	// parse Name attribute.
         while( ( pos = line.find_first_of(":", pos)) != string::npos ) {
            std::string word = line.substr( prev_pos, pos - prev_pos);
            if ( word.size() ) element.push_back(word.c_str());
            prev_pos = ++pos;
         }
      
         if (element.size() == 3) {
            if( strcasecmp(element[1].c_str(), "result_table") != 0 ) {
            // pick up middle part of NAME attribute
            // (ex. gds_triggergroup:gds_trigger:table).
            // In the case of result_table, get layout type in commnet tag.
               fLayout.SetType (Type (element[1].c_str()));
            }
         }
         else fLayout.SetType(Type ("Unknown_Type"));
      }
   
   }

//______________________________________________________________________________
   xsilHandlerEvent::~xsilHandlerEvent() 
   {
   }

//______________________________________________________________________________
   bool xsilHandlerEvent::CommentHandler (const std::string& comment)
   {
   
      Type t;
      if (!fLayout.GetType(t)) { // check if Type already has been given to the layout.
      
         std::string word;
         std::string::size_type pos = 0, prev_pos = 0;
         bool find_FROM = false;
         bool find_TYPE = false;
         while( ( pos = comment.find_first_of(" ",pos)) != string::npos && !find_TYPE) {
            word = comment.substr( prev_pos, pos - prev_pos);
            if ( word.size() ) {
               if (strcasecmp(word.c_str(), "FROM") == 0 && !find_FROM) find_FROM = true;
               else if (find_FROM) find_TYPE = true;
            }
            prev_pos = ++pos;
         }
      
         if (word.size()) {
            fLayout.SetType (Type (word));
         }
         else {
            fLayout.SetType (Type("Unknown_LDAS"));
         }
      
      }
   
      return xsilHandler::CommentHandler (comment);
   }

//______________________________________________________________________________
   bool xsilHandlerEvent::TableEntryHandler (int row, int col, 
                     int type, void* x)
   { 
      if (type == gds_table) {
         fEventCol = col;
         fEventRow = row;
         return true;
      }
      else {
         return xsilHandler::TableEntryHandler (row, col, type, x);
      }
   }

//______________________________________________________________________________
   bool xsilHandlerEvent::HandleTableColumn (int col, 
                     const std::string& name,
                     int type, const attrlist& attr)
   { 
      // Add a column
      if (col >= 0) {
         ColumnType::Enum t = ColumnType::kInvalid;
         switch (type) {
            default:
               fCols.push_back (column_item (t, Column ("")));
               return false;
            case gds_bool:
            case gds_int8:
            case gds_int16:
            case gds_int32:
            case gds_int64:
               t = ColumnType::kInt;
               break;
            case gds_float32:
            case gds_float64:
               t = ColumnType::kReal;
               break;
            case gds_complex32:
            case gds_complex64:
               t = ColumnType::kComplex;
               break;
            case gds_string:
            case gds_channel:
               t = ColumnType::kString;
               break;
            case gds_time:
               t = ColumnType::kTime;
               break;
            case gds_table:
               t = ColumnType::kEvent;
               break;
         };
      
      
      
      //vvvvvvvv mito vvvvvvvv
         std::string colname;
         string::size_type pos;
         pos = name.find_last_of(":");
         if ( pos != string::npos && pos + 1 < name.size() ) {
            colname = name.substr(pos + 1);
         }
         else colname = name;
      
         redirect_info redir = col_normal;
         switch (t) {
            case ColumnType::kInt:
               if (colname.size() > 8 && 
                  strcasecmp(colname.substr(colname.size() - 8).c_str(), "_TIME_NS") == 0) {
                  redir = col_time_ns;
                  colname = colname.substr(0, colname.size() - 3);
                  t = ColumnType::kTime;
               }
               else if (colname.size() > 5 && 
                       strcasecmp(colname.substr(colname.size() - 5).c_str(), "_TIME") == 0) {
                  redir = col_time;
                  t = ColumnType::kTime;
               }
               break;
            case ColumnType::kString:
               if (strcasecmp(colname.c_str(), "NAME") == 0 || 
                  strcasecmp(colname.c_str(), "SEARCH") == 0) {
                  redir = col_name;
                  colname = ColumnType::kColumnNameName;
               }
               else if (strcasecmp(colname.c_str(), "IFO") == 0 || 
                       strcasecmp(colname.c_str(), "IFOS") == 0) {
                  redir = col_ifo;
                  colname = ColumnType::kColumnIfoName;
               }
               break;
            default:
               break;
         }
      //^^^^^^^^^^ mito ^^^^^^^^^^^ 
      
         fCols.push_back (column_item (t, Column(colname), redir));
      
      }
      // No more columns
      else {
      
      //vvvvvvvv mito vvvvvvvv
      	// look for time stamp
         bool find_tstamp = false;
         bool find_tstamp_ns = false;
         for ( column_list::iterator itr = fCols.begin(); 
             itr != fCols.end(); ++itr ) {
            if ( strcasecmp( itr->fColumn.GetName(), "TIME" ) == 0 ) {
               find_tstamp = true;
            }
         }
      
         if ( !find_tstamp ) {
            for ( column_list::iterator itr = fCols.begin(); 
                itr != fCols.end(); ++itr ) {
               if ( itr->fType == ColumnType::kTime ) {
                  if ( strcasecmp( itr->fColumn.GetName(), "START_TIME" ) == 0 ) {
                     itr->fColumn = Column(ColumnType::kColumnTimeName);
                     if ( itr->fRedirect == col_time ) find_tstamp = true;
                     else if ( itr->fRedirect == col_time_ns ) find_tstamp_ns = true;
                  }
               }
               if ( find_tstamp && find_tstamp_ns ) 
                  break;
            }
         }
      
         if ( !find_tstamp ) {
            for ( column_list::iterator itr = fCols.begin(); 
                itr != fCols.end(); ++itr ) {
               if ( itr->fType == ColumnType::kTime ) {
                  if ( strcasecmp( itr->fColumn.GetName(), "END_TIME" ) == 0 ) {
                     itr->fColumn = Column(ColumnType::kColumnTimeName);
                     if ( itr->fRedirect == col_time ) find_tstamp = true;
                     else if ( itr->fRedirect == col_time_ns ) find_tstamp_ns = true;
                  }
               }
               if ( find_tstamp && find_tstamp_ns ) 
                  break;
            }
         }
      
         if ( !find_tstamp ) {
            int indx_s = -1;
            int indx_ns = -1;
            int indx = 0;
            std::string ncol;
            for ( column_list::iterator itr = fCols.begin(); 
                itr != fCols.end(); ++itr, ++indx ) {
               std::string n = itr->fColumn.GetName();
               if ( itr->fType == ColumnType::kTime ) {
                  if ( n.size() > 5 && 
                     strcasecmp(n.substr(n.size() - 5).c_str(), "_TIME") == 0 ) {
                     if ( ncol.empty() ) ncol = n;
                     if ( ncol == n ) {
                        if ( itr->fRedirect == col_time ) {
                           indx_s = indx;
                           find_tstamp = true;
                        }
                        else if ( itr->fRedirect == col_time_ns ) {
                           indx_ns = indx;
                           find_tstamp_ns = true;
                        }
                     }
                  }
               }
               if ( find_tstamp && find_tstamp_ns ) 
                  break;
            }
         
            if ( indx_s >= 0 ) {
               fCols[indx_s].fColumn = Column(ColumnType::kColumnTimeName);
               if ( indx_ns >= 0 ) {
                  fCols[indx_ns].fColumn = Column(ColumnType::kColumnTimeName);
               }
            }
         }
      
      
         for ( column_list::iterator itr = fCols.begin(); 
             itr != fCols.end(); ++itr ) {
            if (!fLayout.GetColumn(itr->fColumn.GetName())) {
               fLayout.AddColumn (itr->fColumn.GetName(), itr->fType);
            }
         }
      //^^^^^^^^^^ mito ^^^^^^^^^^^
      
         if (!fLayout.IsRegistered()) {
            fLayout.Register();
         }
      }
      return true;
   }

//__________________________________________________________________________
template <class T>
   bool xsilHandlerEvent::AddColumnValue (int row, int col, const T& p)
   { 
      // Must have a registered layout
      if (!fLayout.IsRegistered() || 
         (col < 0) || (col >= (int)fCols.size()) ||
         (fCols[col].fType == ColumnType::kInvalid)) { // mito
         return false;
      }
      // Initialize event on first column
      if (col == 0) {
         fEvent = Event (fLayout);
      }
      // Write column
      Value val (p);
      fCols[col].fColumn.Set (fEvent, val);
      // If last column add event to parent
      if (col + 1 == (int)fCols.size()) {
         if (fParent) {
            fParent->AddEvent (fEvent);
         }
      }
      return true;
   }

//_______________________________________________________________ mito
template <>
   bool xsilHandlerEvent::AddColumnValue (int row, int col, const Time& p)
   { 
      // Must have a registered layout
      if (!fLayout.IsRegistered() || 
         (col < 0) || (col >= (int)fCols.size()) ||
         (fCols[col].fType == ColumnType::kInvalid)) {
         return false;
      }
      // Initialize event on first column
      if (col == 0) {
         fEvent = Event (fLayout);
      }
   
      Value val;
      fCols[col].fColumn.Get (fEvent, val);
      Time t;
      val.Write(t);
      if (fCols[col].fRedirect == col_time) {
         t.setS(p.getS());
      }
      else if (fCols[col].fRedirect == col_time_ns) {
         t.setN(p.getN());
      }
      else t = p;
      val.Read(t);
      fCols[col].fColumn.Set (fEvent, val);
   
      // If last column add event to parent
      if (col + 1 == (int)fCols.size()) {
         if (fParent) {
            fParent->AddEvent (fEvent);
         }
      }
      return true;
   }

//_______________________________________________________________ mito
template <>
   bool xsilHandlerEvent::AddColumnValue (int row, int col, const events::Name& p)
   { 
      // Must have a registered layout
      if (!fLayout.IsRegistered() || 
         (col < 0) || (col >= (int)fCols.size()) ||
         (fCols[col].fType == ColumnType::kInvalid)) {
         return false;
      }
      // Initialize event on first column
      if (col == 0) {
         fEvent = Event (fLayout);
      }
   
      fEvent.SetName(p);
   
      // If last column add event to parent
      if (col + 1 == (int)fCols.size()) {
         if (fParent) {
            fParent->AddEvent (fEvent);
         }
      }
      return true;
   }


//__________________________________________________________________________
   bool xsilHandlerEvent::HandleTableEntry (int row, int col, int p)
   {
   
   //vvvvvvvvvv mito vvvvvvvvvvv 
      if (fCols[col].fRedirect == col_time) {
         Time t(p, 0);
         return HandleTableEntry(row, col, t);
      }
      else if (fCols[col].fRedirect == col_time_ns) {
         Time t(0, p);
         return HandleTableEntry(row, col, t);
      }
   //^^^^^^^^^^ mito ^^^^^^^^^^^ 
   
      return AddColumnValue (row, col, p);
   }

//__________________________________________________________________________
   bool xsilHandlerEvent::HandleTableEntry (int row, int col, double p)
   {
      return AddColumnValue (row, col, p);
   }

//__________________________________________________________________________
   bool xsilHandlerEvent::HandleTableEntry (int row, int col,
                     const std::complex<double>& p)
   {
      return AddColumnValue (row, col, p);
   }

//__________________________________________________________________________
   bool xsilHandlerEvent::HandleTableEntry (int row, int col, 
                     const Time& p)
   {
      return AddColumnValue (row, col, p);
   }

//__________________________________________________________________________
   bool xsilHandlerEvent::HandleTableEntry (int row, int col,
                     const std::string& p)
   {
   
   //vvvvvvvvvv mito vvvvvvvvvvv 
      if (fCols[col].fRedirect == col_name) {
         return HandleTableEntry(row, col, Name(p));
      }
      if (fCols[col].fRedirect == col_ifo) {
         int ifo;
         IfoSet::GetBits(p.c_str(), ifo);
         return HandleTableEntry(row, col, ifo);
      }
   //^^^^^^^^^^ mito ^^^^^^^^^^^
   
      return AddColumnValue (row, col, p);
   }

//______________________________________________________________________mito
   bool xsilHandlerEvent::HandleTableEntry (int row, int col, 
                     const Name& p)
   {
      return AddColumnValue (row, col, p);
   }

//__________________________________________________________________________
   bool xsilHandlerEvent::AddEvent (const events::Event& event)
   {
      return AddColumnValue (fEventRow, fEventCol, event);
   }

//__________________________________________________________________________
   xsilHandler* queryEventHandler (const xsilHandlerQuery::attrlist& attr,
                     xsilEventAdd* parent)
   {
      // Check Name attribute.
      xsilHandlerQuery::attrlist::const_iterator name = attr.find (xmlName);
      if (name == attr.end()) {
         return new xsilHandlerEvent (&attr, parent);
      }
      // check for ldas table formats
      std::string line = name->second + ":";
      std::vector<string> element;
      std::string::size_type pos, prev_pos;
      pos = prev_pos = 0;
      // parse Name attribute.
      while( ( pos = line.find_first_of(":", pos)) != string::npos ) {
         std::string word = line.substr( prev_pos, pos - prev_pos);
         if ( word.size() ) element.push_back(word.c_str());
         prev_pos = ++pos;
      }
      // filter out process, segment, etc.
      if ((element.size() == 3) &&
         ((strncasecmp (element[1].c_str(), "process", 7) == 0) ||
         (strncasecmp (element[1].c_str(), "summ_", 5) == 0) ||
         // (strncasecmp (element[1].c_str(), "segment", 7) == 0) ||
         (strncasecmp (element[1].c_str(), "search", 6) == 0) ||
         (strcasecmp (element[1].c_str(), "runlist") == 0) ||
         (strncasecmp (element[1].c_str(), "frameset", 8) == 0) ||
         (strncasecmp (element[1].c_str(), "filter", 6) == 0))) {
         return 0;
      }
      else {
         return new xsilHandlerEvent (&attr, parent); 
      }
   }

} // namespace xml
