#ifndef _LIGO_EVENTVALUE_H
#define _LIGO_EVENTVALUE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Value							*/
/*                                                         		*/
/* Module Description: Value of a function				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Argument.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/ColumnType.hh"


namespace events {

   class IfoSet;

/** The event value class contains the number value including its type.
    Typically, the values of event columns are returned as event values.
    A value is also a constant function!

    @memo Defines an event value
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Value : public Function, public ColumnType {
   public:
      /// Value type
      typedef ColumnType::Enum valuetype;
   
      /// Value union
      union Union {
         char	mInt[sizeof(Int)];
         char	mReal[sizeof(Real)];
         char	mComplex[sizeof(Complex)];
         char	mTime[sizeof(time_type)];
         char	mString[sizeof(String)];
      
         struct {
            data_ptr layout;
            data_ptr data;
         } mEvent;
      };
      /// Maximum data size
      static const int kMaxValueSize;
   
      /** Creates an event value.
          @memo Default constructor
       ******************************************************************/
      Value() : mType (kInvalid) {
      }
      /** Copy constructor.
          @memo Copy constructor
       ******************************************************************/
      Value (const Value& val) : mType (kInvalid) {
         *this = val; }
      /** Destructs an event argument.
          @memo Denstructor
       ******************************************************************/
      ~Value();
      /** Assignment operator.
          @memo Assignment operator
       ******************************************************************/
      Value& operator= (const Value& val);
      /** Returns a copy of the event function. This method must be 
          overriden by all descendents.
          @memo Copy the event function
          @return event function copy
       ******************************************************************/
      virtual Value* Copy() const {
         return new Value (*this); }
   
      /** Creates an event value.
          @memo Default constructor
       ******************************************************************/
      Value (Real x);
      /** Creates an event value.
          @memo Constructor (int)
       ******************************************************************/
      Value (Int i);
      /** Creates an event value.
          @memo Constructor (Time)
       ******************************************************************/
      Value (const Time& t);
      /** Creates an event value.
          @memo Constructor (complex)
       ******************************************************************/
      Value (const Complex& c);
      /** Creates an event value.
          @memo Constructor (string)
       ******************************************************************/
      Value (const String& s);
      /** Creates an event value. If the length argument is negative
          the strlen function is used to determine the length of the 
          data.
          @memo Constructor (string)
       ******************************************************************/
      Value (const char* s, int len = -1);
      /** Creates an event value.
          @memo Constructor (event)
       ******************************************************************/
      Value (const Event& e);
      /** Creates an event value.
          @memo Constructor (specified type)
       ******************************************************************/
      Value (Enum type, const void* p);
   
      /** Get value in real format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (Real& x) const;
      /** Get value in int format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (Int& i) const;
      /** Get value in complex format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (Complex& c) const;
      /** Get value in Time format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (time_type& t) const;
      /** Get value in string format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (std::string& s) const;
      /** Get value in string format. The length has to be set to the
          maximum size of the character buffer p before the call. It
          will return the actual length of the string (non counting
          a terminating NULL character). The returned string is always
          NULL terminated regardless whether the string contains
          ASCII or binary data. Do not use strlen to determine the
          length of the string if it contains binary data!
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (char* p, int& len) const;
      /** Get value in event format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (Event& e) const;
      /** Get value in the specified format.
          @return True if conversion possible
          @memo Get value
       ******************************************************************/
      bool Write (Enum type, data_ptr p) const;
   
      /** Set value in real format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (Real x);
      /** Set value in int format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (Int i);
      /** Set value in complex format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (const Complex& x);
      /** Set value in Time format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (const time_type& t);
      /** Set value in string format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (const std::string& s);
      /** Set value in string format. If the length argument is negative
          the strlen function is used to determine the length of the 
          data.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (const char* p, int len = -1);
      /** Set value in event format.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (const Event& e);
      /** Set value in the specified format. If the data pointer is null,
          the value is set to the default value of this type.
          @return True if conversion possible
          @memo Set value
       ******************************************************************/
      bool Read (Enum type, const_data_ptr p);
   
      /** Type of value.
          @memo Value type
       ******************************************************************/
      valuetype Type () const {
         return mType; }
   
      /** A value is a constant function.
          @memo Function value
          @param arg Event argument list
          @param x value (return)
          @return True if function could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument&, Value& x) const {
         x = *this;
         return mType != kInvalid; }
   
      /** Math operator for value (plus equal).
          @memo Value math (plus equal)
          @return itself
       ******************************************************************/
      Value& operator += (const Value& val);
      /** Math operator for value (minus equal).
          @memo Value math (minus equal)
          @return itself
       ******************************************************************/
      Value& operator -= (const Value& val);
      /** Math operator for value (times equal).
          @memo Value math (times equal)
          @return itself
       ******************************************************************/
      Value& operator *= (const Value& val);
      /** Math operator for value (over equal).
          @memo Value math (over equal)
          @return itself
       ******************************************************************/
      Value& operator /= (const Value& val);
      /** Math operator for value (mod equal).
          @memo Value math (mod equal)
          @return itself
       ******************************************************************/
      Value& operator %= (const Value& val);
      /** Math operator for value (or equal).
          @memo Value math (or equal)
          @return itself
       ******************************************************************/
      Value& operator |= (const Value& val);
      /** Math operator for value (and equal).
          @memo Value math (and equal)
          @return itself
       ******************************************************************/
      Value& operator &= (const Value& val);
      /** Math operator for value (xor equal).
          @memo Value math (xor equal)
          @return itself
       ******************************************************************/
      Value& operator ^= (const Value& val);
      /** Math operator for value (shift right equal).
          @memo Value math (shift right)
          @return itself
       ******************************************************************/
      Value& operator >>= (const Value& val);
      /** Math operator for value (shift left equal).
          @memo Value math (shift left equal)
          @return itself
       ******************************************************************/
      Value& operator <<= (const Value& val);
   
   private:
      /// Value type
      valuetype			mType;
      /// Data array
      char			mData[sizeof(Union)];
   };


/** @name Value operators
    The following math operations are supported:
    \begin{verbatim}
    \end{verbatim}
    @memo Defines math operators for values
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

   /** Math operator for value (unary plus).
       @memo Value math (unary plus)
       @return result
    *********************************************************************/
   inline Value operator+ (const Value& val) {
      return val; }
   /** Math operator for value (unary minus).
       @memo Value math (unary minus)
       @return result
    *********************************************************************/
   Value operator- (const Value& val);
   /** Math operator for value (prefix increment).
       @memo Value math (prefix increment)
       @return result
    *********************************************************************/
   Value& operator++ (Value& val);
   /** Math operator for value (prefix decrement).
       @memo Value math (prefix decrement)
       @return result
    *********************************************************************/
   Value& operator-- (Value& val);
   /** Math operator for value (postfix increment).
       @memo Value math (postfix increment)
       @return result
    *********************************************************************/
   inline Value operator++ (Value& val, int) {
      Value tmp = val; ++val; 
      return tmp; }
   /** Math operator for value (postfix decrement).
       @memo Value math (postfix decrement)
       @return result
    *********************************************************************/
   inline Value operator-- (Value& val, int) {
      Value tmp = val; --val; 
      return tmp; }
   /** Math operator for value (plus).
       @memo Value math (plus)
       @return result
    *********************************************************************/
   inline Value operator+ (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp += v2; 
      return tmp; }
   /** Math operator for value (minus).
       @memo Value math (minus)
       @return result
    *********************************************************************/
   inline Value operator- (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp -= v2; 
      return tmp; }
   /** Math operator for value (times).
       @memo Value math (times)
       @return result
    *********************************************************************/
   inline Value operator* (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp *= v2; 
      return tmp; }
   /** Math operator for value (over).
       @memo Value math (over)
       @return result
    *********************************************************************/
   inline  Value operator/ (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp /= v2; 
      return tmp; }
   /** Math operator for value (mod).
       @memo Value math (mod)
       @return result
    *********************************************************************/
   inline Value operator% (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp %= v2; 
      return tmp; }
   /** Math operator for value (not).
       @memo Value math (not)
       @return result
    *********************************************************************/
   Value operator~ (const Value& val);
   /** Math operator for value (or).
       @memo Value math (or)
       @return result
    *********************************************************************/
   inline Value operator| (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp |= v2; 
      return tmp; }
   /** Math operator for value (and).
       @memo Value math (and)
       @return result
    *********************************************************************/
   inline Value operator& (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp &= v2; 
      return tmp; }
   /** Math operator for value (xor).
       @memo Value math (xor)
       @return result
    *********************************************************************/
   inline Value operator^ (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp ^= v2; 
      return tmp; }
   /** Math operator for value (shift left).
       @memo Value math (shift left)
       @return result
    *********************************************************************/
   inline Value operator<< (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp <<= v2; 
      return tmp; }
   /** Math operator for value (shift right).
       @memo Value math (shift right)
       @return result
    *********************************************************************/
   inline Value operator>> (const Value& v1, const Value& v2) {
      Value tmp = v1; tmp >>= v2; 
      return tmp; }

//@}
/** @name Value math functions
    The following math functions are supported:
    \begin{verbatim}
    \end{verbatim}
    @memo Defines math functions for values
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
   /** Math function for value (abs).
       @memo Value math (abs)
       @return result
    *********************************************************************/
   Value abs (const Value& val);
   /** Math function for value (sqrt).
       @memo Value math (sqrt)
       @return result
    *********************************************************************/
   Value sqrt (const Value& val);
   /** Math function for value (pow).
       @memo Value math (pow)
       @return result
    *********************************************************************/
   Value pow (const Value& v1, const Value& v2);
   /** Math function for value (exp).
       @memo Value math (exp)
       @return result
    *********************************************************************/
   Value exp (const Value& val);
   /** Math function for value (log).
       @memo Value math (log)
       @return result
    *********************************************************************/
   Value log (const Value& val);
   /** Math function for value (log10).
       @memo Value math (log10)
       @return result
    *********************************************************************/
   Value log10 (const Value& val);
   /** Math function for value (sin).
       @memo Value math (sin)
       @return result
    *********************************************************************/
   Value sin (const Value& val);
   /** Math function for value (cos).
       @memo Value math (cos)
       @return result
    *********************************************************************/
   Value cos (const Value& val);
   /** Math function for value (tan).
       @memo Value math (tan)
       @return result
    *********************************************************************/
   Value tan (const Value& val);
   /** Math function for value (asin).
       @memo Value math (asin)
       @return result
    *********************************************************************/
   Value asin (const Value& val);
   /** Math function for value (acos).
       @memo Value math (acos)
       @return result
    *********************************************************************/
   Value acos (const Value& val);
   /** Math function for value (atan).
       @memo Value math (atan)
       @return result
    *********************************************************************/
   Value atan (const Value& val);
   /** Math function for value (atan2).
       @memo Value math (atan2)
       @return result
    *********************************************************************/
   Value atan2 (const Value& v1, const Value& v2);
   /** Math function for value (sinh).
       @memo Value math (sinh)
       @return result
    *********************************************************************/
   Value sinh (const Value& val);
   /** Math function for value (cosh).
       @memo Value math (cosh)
       @return result
    *********************************************************************/
   Value cosh (const Value& val);
   /** Math function for value (tanh).
       @memo Value math (tanh)
       @return result
    *********************************************************************/
   Value tanh (const Value& val);
   /** Math function for value (ceil).
       @memo Value math (ceil)
       @return result
    *********************************************************************/
   Value ceil (const Value& val);
   /** Math function for value (floor).
       @memo Value math (floor)
       @return result
    *********************************************************************/
   Value floor (const Value& val);
   /** Math function for value (conj).
       @memo Value math (conj)
       @return result
    *********************************************************************/
   Value conj (const Value& val);
   /** Math function for value (polar).
       @memo Value math (polar)
       @return result
    *********************************************************************/
   Value polar (const Value& v1, const Value& v2);
   /** Math function for value (real).
       @memo Value math (real)
       @return result
    *********************************************************************/
   Value real (const Value& val);
   /** Math function for value (imag).
       @memo Value math (imag)
       @return result
    *********************************************************************/
   Value imag (const Value& val);
   /** Math function for value (arg).
       @memo Value math (arg)
       @return result
    *********************************************************************/
   Value arg (const Value& val);
   /** Math function for value (norm).
       @memo Value math (norm)
       @return result
    *********************************************************************/
   Value norm (const Value& val);

   /** Comparison function for values (equal).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (equal)
       @return result
    *********************************************************************/
   Value equal (const Value& v1, const Value& v2);
   /** Comparison function for values (unequal).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (unequal)
       @return result
    *********************************************************************/
   Value unequal (const Value& v1, const Value& v2);
   /** Comparison function for values (less).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (less)
       @return result
    *********************************************************************/
   Value less (const Value& v1, const Value& v2);
   /** Comparison function for values (less equal).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (less equal)
       @return result
    *********************************************************************/
   Value lessequal (const Value& v1, const Value& v2);
   /** Comparison function for values (greater).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (greater)
       @return result
    *********************************************************************/
   Value greater (const Value& v1, const Value& v2);
   /** Comparison function for values (greater equal).
       Return is a bool converted to an Int if successful.
       @memo Value comparision (greater equal)
       @return result
    *********************************************************************/
   Value greaterequal (const Value& v1, const Value& v2);

//@}
/** @name Value string functions
    The following string functions are supported:
    \begin{verbatim}
    \end{verbatim}
    @memo Defines string functions for values
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
   /** String function for value (bin2hex).
       @memo Value string processing (bin2hex)
       @return result
    *********************************************************************/
   Value bin2hex (const Value& val);
   /** String function for value (hex2bin).
       @memo Value string processing (hex2bin)
       @return result
    *********************************************************************/
   Value hex2bin (const Value& val);
//@}


/** @name Conversion functions
    The following conversion functions are supported:
    \begin{verbatim}
    IVal (Function) - integer value
    RVal (Function) - floating point value
    CVal (Function) - complex value
    TVal (Function) - time value
    IfoVal (IfoSet) - integer value
    \end{verbatim}
    @memo Defines conversion functions for values
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

/** Converts to a 32 bit integer value.
    @memo Integer conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class IVal : public Function {
   public:
      /// Constructor
      explicit IVal (const Function& f) : mPtr (f) {
      }
      explicit IVal (const FunctionPtr& f) : mPtr (f) {
      }
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual IVal* Copy() const {
         return new IVal (*this); }
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param x Value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Argument
      FunctionPtr	mPtr;
   };

/** Converts to a double precision floating point value.
    @memo Real conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class RVal : public Function {
   public:
      /// Constructor
      explicit RVal (const Function& f) : mPtr (f) {
      }
      explicit RVal (const FunctionPtr& f) : mPtr (f) {
      }
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual RVal* Copy() const {
         return new RVal (*this); }
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param x Value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Argument
      FunctionPtr	mPtr;
   };

/** Converts to a double precision complex floating point value.
    @memo Complex conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class CVal : public Function {
   public:
      /// Constructor
      explicit CVal (const Function& f) : mPtr (f) {
      }
      explicit CVal (const FunctionPtr& f) : mPtr (f) {
      }
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual CVal* Copy() const {
         return new CVal (*this); }
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param x Value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Argument
      FunctionPtr	mPtr;
   };

/** Converts to a time value.
    @memo Time conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class TVal : public Function {
   public:
      /// Constructor
      explicit TVal (const Function& f) : mPtr (f) {
      }
      explicit TVal (const FunctionPtr& f) : mPtr (f) {
      }
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual TVal* Copy() const {
         return new TVal (*this); }
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param x Value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Argument
      FunctionPtr	mPtr;
   };

/** Converts to a string value.
    @memo Time conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class SVal : public Function {
   public:
      /// Constructor
      explicit SVal (const Function& f) : mPtr (f) {
      }
      explicit SVal (const FunctionPtr& f) : mPtr (f) {
      }
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual SVal* Copy() const {
         return new SVal (*this); }
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param x Value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Argument
      FunctionPtr	mPtr;
   };

/** Converts a interferometer set to value.
    @memo Interferometer set conversion
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class IfoVal : public Value {
   public:
      /// Ifo set type
      typedef ColumnType::ColumnIfo_t ifoset_type;
      /// Constructor
      explicit IfoVal (const char* ifoset);
      /// Constructor
      explicit IfoVal (const std::string& ifoset);
      /// Constructor
      explicit IfoVal (const IfoSet& ifoset);
      /** Returns a copy of the conversion function. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual IfoVal* Copy() const {
         return new IfoVal (*this); }
   };

//@}

}

#endif // _LIGO_EVENTVALUE_H
