#ifndef _LIGO_EVENTMATH_H
#define _LIGO_EVENTMATH_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Math							*/
/*                                                         		*/
/* Module Description: Math operations with event functions		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Math.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Value.hh"


namespace events {

/** Event columns can be used in math expression which can be used
    everywhere an event function is needed. For example one can write 
    \begin{verbatim}
    5 < sqrt (pwr (Column ("Amplitude(0)"), 2) +
              pwr (Column ("Amplitude(1)"), 2) )
    \begin{verbatim}
    to define an event condition which checks whether the vector 
    sum of the amplitudes of two events is larger than 5.
   
    @memo Defines the event math routines
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Math : public Function {
   public:
      /** Math operations supported by the function.
          @memo Math operations.
       ******************************************************************/
      enum operation {
      /// Positive value
      opPos = 0,
      /// Negative value
      opNeg,
      /// Increment (prefix)
      opInc,
      /// Decrement (prefix)
      opDec,
      /// Addition
      opAdd,
      /// Subtraction
      opSub,
      /// Multiplication
      opMul,
      /// Division
      opDiv,
      /// Modulus
      opMod,
      /// Absolute value
      opAbs,
      /// Square root
      opSqrt,
      /// Power
      opPow,
      /// Exponential
      opExp,
      /// Natural logarithm
      opLog,
      /// Logarithm base 10
      opLog10,
      /// Sine
      opSin,
      /// Cosine
      opCos,
      /// Tangent
      opTan,
      /// Arc sine
      opAsin,
      /// Arc cosine
      opAcos,
      /// Arc tangent
      opAtan,
      /// Arc tangent (2 arguments)
      opAtan2,
      /// Sine hyp
      opSinh,
      /// Cosine hyp
      opCosh,
      /// Tangent hyp
      opTanh,
      /// Ceiling
      opCeil,
      /// Floor
      opFloor,
      /// Negate ~
      opNot,
      /// And &
      opAnd,
      /// Or |
      opOr,
      /// Xor ^
      opXor,
      /// Left shift <<
      opLeft,
      /// Right shift >>
      opRight,
      /// Complex conjugate
      opConj,
      /// Complex number in polar notation
      opPolar,
      /// Real part
      opReal,
      /// Imaginary part
      opImag,
      /// Argument
      opArg,
      /// Norm
      opNorm
      };
   
      /** Creates an math expression of a single value (unary operator).
          @memo Constructor
       ******************************************************************/
      Math (const Function& f, operation op) 
      : mOp (op), mPtr1 (f), mPtr2 (0) {
      }
      Math (const FunctionPtr& f, operation op) 
      : mOp (op), mPtr1 (f), mPtr2 (0) {
      }
      /** Creates an math expression of two values (binary operator).
          @memo Constructor
       ******************************************************************/
      Math (const Function& f1, const Function& f2, 
           operation op)
      : mOp (op), mPtr1 (f1), mPtr2 (f2) {
      }
      Math (const FunctionPtr& f1, const FunctionPtr& f2, 
           operation op)
      : mOp (op), mPtr1 (f1), mPtr2 (f2) {
      }
   
      /** Returns a copy of the event column math object. This method 
          must be overriden by all descendents.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      virtual Math* Copy() const {
         return new Math (*this); }
   
      /** Returns the evaluated math expression.
          @memo Math expression value
          @param arg Event argument list
          @param val Fucntion value (return)
          @return true if successful
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   protected:
      /** Default constructor.
          @memo Default constructor
       ******************************************************************/
      Math () : mOp (opPos) {
      }
   
   private:
      /// Operation
      operation 	mOp;
      /// First argument if exists
      FunctionPtr	mPtr1;
      /// Second argument if exists
      FunctionPtr	mPtr2;
   };

/** @name Math operators
    The following math operations and functions are supported:
    \begin{verbatim}
    Unary: 
    +     -     ~     ++    --
    abs   sqrt  exp   log   log10 sin   cos   tan   acos  asin  atan  
    sinh  cosh  tanh  ceil  floor
    
    Binary:
    +     -     *     /     %     &     |     ^     <<    >>
    pow   atan2 
    \end{verbatim}
    For unary operators and functions the argument is always a function. 
    For binary operators and functions the argument one of the arguments
    is a function whereas the other is either a function, or one of
    the column types Int, Real, Complex or Time.

    For strings the following operators and functions are supported:
    \begin{verbatim}
    \end{verbatim}

    For events no functions or operators are supported.

    @memo Defines event function math operators
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
//@Include: MathOps.hh
//@}
#include "MathOps.hh"

/** @name Shift a function
    When a function is shifted the specified number of events are removed
    from the argument list and the function is evaluated on the remaining
    event arguments.
   
    @memo Defines the function shift operation
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

/** Shifted function.
    @memo Shifted function
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ShiftFunction : public Function {
   public:
      /** Creates a shifted function.
          @memo Constructor
          @param func Function to be shifted
          @param howmuch Number of events to shift
       ******************************************************************/
      explicit ShiftFunction (const Function& func, int howmuch = 1) :
      mShift (howmuch), mPtr (func) {
      }
      explicit ShiftFunction (const FunctionPtr& func, int howmuch = 1) :
      mShift (howmuch), mPtr (func) {
      }
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual ShiftFunction* Copy() const {
         return new ShiftFunction (*this); }
   
      /** Evaluate the function.
          @memo Evaluate
          @param arg Event argument list
          @param val Fucntion value (return)
          @return True if function could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, Value& val) const;
   
   private:
      /// Shift amount
      int		mShift;
      /// Condition 
      FunctionPtr 	mPtr;
   };

/** Shift function
    @memo Shift a function
    @param func Function to be shifted
    @param howmuch Number of events to shift
    @return Shifted function
 ************************************************************************/
   ShiftFunction Shift (const Function& func, int howmuch = 1);
   ShiftFunction Shift (const FunctionPtr& func, int howmuch = 1);
 
   inline ShiftFunction Shift (const Function& func, int howmuch) {
      return ShiftFunction (func, howmuch); }
   inline ShiftFunction Shift (const FunctionPtr& func, int howmuch) {
      return ShiftFunction (func, howmuch); }
//@}


}

#endif // _LIGO_EVENTMATH_H
