#ifndef _LIGO_EVENTINDEXLIST_H
#define _LIGO_EVENTINDEXLIST_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: IndexList						*/
/*                                                         		*/
/* Module Description: A list with an index				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: IndexList.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/
#ifndef __CINT__
#include <string>
#include <vector>
#include <map>
#include <algorithm>
#include "refcount/ref_ptr.hh"


namespace events {


   struct IndexSort {
      bool operator() (const std::string& s1, const std::string& s2) const;
   };


/** Defines a smart pointer with reference counting which can be
    used as an element for IndexList.
   
    @memo Defines a smart pointer for the index list
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class IndexEntryPtr : public gdsbase::ref_ptr<T> {
   public:
      /** Creates a NULL pointer.
          @memo Default constructor
       ******************************************************************/
      IndexEntryPtr() : gdsbase::ref_ptr<T> (0) {
      }
      /** Creates an pointer and adopts the object.
          @memo Constructor
          @param obj Object to be adopted
          @param array Set true if array delete has to be used
       ******************************************************************/
      explicit IndexEntryPtr (T* obj) : gdsbase::ref_ptr<T> (obj) {
      }
      /// Equality operator
      bool operator== (const IndexEntryPtr& p) const {
         return (p.get() == this->get()) ||
            (p.get() && this->get() && (*this->get() == *p)); }
      /// Inequality operator
      bool operator!= (const IndexEntryPtr& p) const {
         return !(*this == p);}
      /// Less operator
      bool operator< (const IndexEntryPtr& p) const {
         return (this->get() && !p.get()) ||
            (p.get() && this->get() && (*this->get() < *p)); }
      /// get name
      const char* GetName () const {
         return (this->get() ? this->get()->GetName() : 0); }
   };



/** Defines a list with a by default case-insensitive index. 
   
    @memo Defines a list with a case-insensitive index
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T, class Cmp = IndexSort>
   class IndexList {
   public:
      /// Element type
      typedef T element_type;
      /// Base list
      typedef std::vector<T> base_list;
      /// Iterator 
      typedef typename base_list::iterator iterator;
      /// Const Iterator 
      typedef typename base_list::const_iterator const_iterator;
      /// Sort order
      typedef Cmp sort_order;
      /// Index list
      typedef std::map<std::string, int, Cmp> index_type;
      /// Index iterator 
      typedef typename index_type::iterator index_iterator;
      /// Const index iterator 
      typedef typename index_type::const_iterator const_index_iterator;
   
      /** Creates an empty index list
          @memo Default constructor
       ******************************************************************/
      IndexList() {
      }
      /** Copy constructor
          @memo Copy constructor
       ******************************************************************/
      IndexList (const IndexList& index) {
         *this = index; }
      /** Assignment operator
          @memo Assignment operator
       ******************************************************************/
      IndexList& operator= (const IndexList& index) {
         if (this != &index) {
            mList = index.mList; RebuildIndex(); }
         return *this; }
      /** Return the list
          @memo Index
       ******************************************************************/
      const base_list& List() const {
         return mList; }
      /** Return the beginning of the list
          @memo begin
       ******************************************************************/
      typename base_list::iterator begin() {
         return mList.begin(); }
      /** Return the beginning of the list
          @memo begin
       ******************************************************************/
      const_iterator begin() const {
         return mList.begin(); }
      /** Return the end of the list
          @memo end
       ******************************************************************/
      typename base_list::iterator end() {
         return mList.end(); }
      /** Return the end of the list
          @memo end
       ******************************************************************/
      const_iterator end() const {
         return mList.end(); }
      /** Return the index
          @memo Index
       ******************************************************************/
      const index_type& Index() const {
         return mIndex; }
      /** Get the element (uses the index)
          @memo Get
       ******************************************************************/
      T* Get (const char* s) {
         typename index_type::iterator i;
         return (s && ((i = mIndex.find (s)) != mIndex.end())) ? 
            &mList[i->second] : 0; }
      /** Get the element (uses the index)
          @memo Get
       ******************************************************************/
      const T* Get (const char* s) const {
         typename index_type::const_iterator i;
         return (s && ((i = mIndex.find (s)) != mIndex.end())) ? 
            &mList[i->second] : 0; }
      /** Get the element position (uses the index)
          @memo Get position
       ******************************************************************/
      typename base_list::iterator GetPos (const char* s) {
         typename index_type::const_iterator i;
         return (s && ((i = mIndex.find (s)) != mIndex.end())) ? 
            mList.begin() + i->second : mList.end(); }
      /** Get the element position (uses the index)
          @memo Get position
       ******************************************************************/
      const_iterator GetPos (const char* s) const {
         typename index_type::const_iterator i;
         return (s && ((i = mIndex.find (s)) != mIndex.end())) ? 
            mList.begin() + i->second : mList.end(); }
      /** Element access by position
          @memo Element access
       ******************************************************************/
      T& operator[] (int pos) {
         return mList[pos]; }
      /** Element access by position
          @memo Element access
       ******************************************************************/
      const T& operator[] (int pos) const {
         return mList[pos]; }
   
      /** Adds an element to the end of the list
          @memo Add
       ******************************************************************/
      void Add (const T& el) {
         T* t = Get (el.GetName());
         if (t != 0) {
            *t = el; } 
         else {
            mList.push_back (el); AddToIndex (mList.end() - 1); } 
      }
      /** Inserts an element
          @memo Insert
       ******************************************************************/
      void Insert (const T& el) {
         typename base_list::iterator i = GetPos (el.GetName());
         if (i != mList.end()) {
            Remove (i); }
         i = std::lower_bound (mList.begin(), mList.end(), el);
         i = mList.insert (i, el); 
         RebuildIndexAfterInsert (i); AddToIndex (i);
      }
      /** Removes an element from the list
          @memo Remove
       ******************************************************************/
      void Remove (typename base_list::iterator pos) {
         if (pos != mList.end()) {
            RemoveFromIndex (pos); mList.erase (pos); 
            RebuildIndexAfterRemove (pos); }
      }  
      /** Removes an element from the list
          @memo Remove
       ******************************************************************/
      void Remove (int pos) {
         Remove (mList.begin()+pos); }
      /** Removes an element from the list
          @memo Remove
       ******************************************************************/
      void Remove (const char* name) {
         Remove (GetPos (name)); }  
      /** Clears the list
          @memo Clear
       ******************************************************************/
      void Clear() {
         mIndex.clear(); mList.clear(); }
      /** Size of the list
          @memo Size
       ******************************************************************/
      int Size() const {
         return mList.size(); }
      /** Empty list?
          @memo Empty?
       ******************************************************************/
      bool Empty() const {
         return mList.empty(); }
   
      /** Sort the base list
          @memo Sort
       ******************************************************************/
      void Sort () {
         std::sort (mList.begin(), mList.end()); RebuildIndex(); }
   
   protected:
      /** Add an entry to the index
          @memo Index add
       ******************************************************************/
      void AddToIndex (typename base_list::iterator pos) {
         mIndex[pos->GetName()] = pos - mList.begin(); }
   
      /** Remove an entry from the index
          @memo Index remove
       ******************************************************************/
      void RemoveFromIndex (typename base_list::iterator pos) {
         mIndex.erase (pos->GetName()); }
      /** Rebuild index
          @memo Rebuild index
       ******************************************************************/
      void RebuildIndex() {
         mIndex.clear();
         for (typename base_list::iterator i= mList.begin(); 
	      i != mList.end(); ++i) {
            AddToIndex (i); } 
      }
      /** Rebuild index after insert
          @memo Rebuild index after insert
       ******************************************************************/
      void RebuildIndexAfterInsert (typename base_list::iterator pos) {
         typename index_type::mapped_type ref = pos - mList.begin();
         for (index_iterator i= mIndex.begin(); i != mIndex.end(); ++i) {
            if (i->second >= ref) ++i->second; } 
      }
      /** Rebuild index after remove
          @memo Rebuild index after remove
       ******************************************************************/
      void RebuildIndexAfterRemove (typename base_list::iterator pos) {
         typename index_type::mapped_type ref = pos - mList.begin();
         for (index_iterator i= mIndex.begin(); i != mIndex.end(); ++i) {
            if (i->second > ref) --i->second; } 
      }
   
   private:
      /// Base list
      base_list		mList;
      /// Index
      index_type	mIndex;
   };


}
#endif // __CINT__
#endif // _LIGO_EVENTINDEXLIST_H
