#ifndef _LIGO_EVENTFUNCPTR_H
#define _LIGO_EVENTFUNCPTR_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: FunctionPtr						*/
/*                                                         		*/
/* Module Description: Defines a smart pointer for an event columns	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: FunctionPtr.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Function.hh"


namespace events {

/** A smart pointer for an event column. When copying an event column 
    smart pointer, a new event column is created which is an exact copy
    of the original.
   
    @memo Smart pointer for event lists.
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class FunctionPtr {
   public:
      /** Creates a pointer to an event function. The event function gets 
          adopted.
          @memo Default constructor
       ******************************************************************/
      explicit FunctionPtr (Function* ptr = 0) : mPtr (0) {
         Reset (ptr); }
      /** Creates a new copy of the event function. The event function is 
          not adopted.
          @memo Constructor
       ******************************************************************/
      FunctionPtr (const Function& func) : mPtr (0) {
         Reset (func); }
      /** Copies an event function.
          @memo Copy constructor
       ******************************************************************/
      FunctionPtr (const FunctionPtr& ptr) : mPtr (0) {
         *this = ptr; }
      /** Destructs the event function.
          @memo Destructor
       ******************************************************************/
      ~FunctionPtr () {
         Reset (0); }
   #if 0
      /** Type converison into an event function.
          @memo type conversion
       ******************************************************************/
      operator Function& () {
         return *this; }
      /** Type converison into an event function.
          @memo type conversion
       ******************************************************************/
      operator const Function& () const {
         return *this; }
   #endif
      /** Assignment. Makes a new pointer with a true copy.
          @memo Assignment operator
       ******************************************************************/
      FunctionPtr& operator= (const FunctionPtr& ptr) {
         if (this != &ptr) Reset (ptr.Get() ? ptr->Copy() : 0); 
         return *this; }
      /** Get a pointer to the event function.
          @memo Get
       ******************************************************************/
      Function* Get () {
         return mPtr; }
      /** Get a pointer to the event function.
          @memo Get
       ******************************************************************/
      const Function* Get () const {
         return mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      Function& operator*() {
         return *mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      const Function& operator*() const {
         return *mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      Function* operator->() {
         return mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      const Function* operator->() const {
         return mPtr; }
      /** Release the event function.
          @memo Release
       ******************************************************************/
      Function* Release () {
         Function* e = mPtr; mPtr = 0; 
         return e; }
      /** Resets the event function. The new function gets adopted.
          @memo Reset
       ******************************************************************/
      void Reset (Function* ptr = 0) {
         if (mPtr) delete mPtr; 
         mPtr = ptr; }
      /** Resets the event function. The new function is copied.
          @memo Reset
       ******************************************************************/
      void Reset (const Function& func) {
         if (mPtr) delete mPtr; 
         mPtr = func.Copy(); }
   
   private:
      /** Event function pointer.
          @memo Pointer to event function
       ******************************************************************/
      Function* mPtr;
   };


}

#endif // _LIGO_EVENTFUNCPTR_H
