#ifndef _LIGO_EVENTCLUSTER_H
#define _LIGO_EVENTCLUSTER_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Cluster							*/
/*                                                         		*/
/* Module Description: Event cluster					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Cluster.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <vector>
#include "events/TimeWindow.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/Event.hh"


namespace events {

   class Event;


/** Describes an event cluster condition. An event cluster is recognized,
    if the number of events which fullfill the event condition and
    which lie within the specfied time window around the analyzed
    event exceeds the threshold value. The current events are not
    considered.
   
    @memo Defines an event cluster condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Cluster : public Condition {
   public:
   #ifndef __CINT__
      /// Event list
      typedef std::vector < const Event* > eventlist;
   #endif
      /** Creates a default event cluster condition. All events are
          considered. The default threshold is three.  The default
          window is 1s.
          @memo Default constructor
       ******************************************************************/
      Cluster() : mThreshold (3), mWindow (1.0) {
      }
      /** Creates an event cluster condition. All events are
          considered.
          @memo Constructor
   	  @param threshold Cluster threshold (number of events)
   	  @param duration Duration of veto window
   	  @param offset Time offset for veto start (relative to event)
       ******************************************************************/
      Cluster (int threshold, const TimeWindow& window)
      : mThreshold (threshold), mWindow (window) {
      }
      /** Creates an event cluster condition. All events are
          considered.
          @memo Constructor
   	  @param threshold Cluster threshold (number of events)
   	  @param duration Duration of veto window
   	  @param offset Time offset for veto start (relative to event)
       ******************************************************************/
      Cluster (int threshold, const Interval& duration, 
              const Interval& offset)
      : mThreshold (threshold), mWindow (duration, offset) {
      }
      /** Creates an event cluster condition.
          @memo Constructor
   	  @param cond Event condition for cluster participants
   	  @param threshold Cluster threshold (number of events)
   	  @param duration Duration of veto window
       ******************************************************************/
      Cluster (const Condition& cond, int threshold, 
              const TimeWindow& window)
      : mCond (cond), mThreshold (threshold), mWindow (window) {
      }
      Cluster (const ConditionPtr& cond, int threshold, 
              const TimeWindow& window)
      : mCond (cond), mThreshold (threshold), mWindow (window) {
      }
      /** Creates an event cluster condition.
          @memo Constructor
   	  @param cond Event condition for cluster participants
   	  @param threshold Cluster threshold (number of events)
   	  @param duration Duration of veto window
   	  @param offset Time offset for veto start (relative to event)
       ******************************************************************/
      Cluster (const Condition& cond, int threshold, 
              const Interval& duration, const Interval& offset)
      : mCond (cond), mThreshold (threshold), mWindow (duration, offset) {
      }
      Cluster (const ConditionPtr& cond, int threshold, 
              const Interval& duration, const Interval& offset)
      : mCond (cond), mThreshold (threshold), mWindow (duration, offset) {
      }
   
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual Cluster* Copy() const {
         return new Cluster (*this); }
   
      /** Evaluate the condition. This function must be overriden
          by descendents.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
   #ifndef __CINT__
      /** Evaluates the cluster condition. Returns the number of clusters 
          found as well as a list of them.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @param list Events belonging to the cluster (return)
          @return True if condition could be evaluated
       ******************************************************************/
      int GetClusters (const Argument& arg, eventlist& list) const;
   #endif
      /** Get the event condition.
          @memo Get event condition
       ******************************************************************/
      const ConditionPtr& GetCondition () const {
         return mCond; }
      /** Set the cluster threshold.
          @memo Set threshold
       ******************************************************************/
      void SetThreshold (int threshold) {
         mThreshold = threshold; }
      /** Get the cluster threshold.
          @memo Get threshold
       ******************************************************************/
      int GetThreshold () const {
         return mThreshold; }
      /** Set the time window.
          @memo Set time window
       ******************************************************************/
      void SetWindow (const TimeWindow& window) {
         mWindow = window; }
      /** Set the time window.
          @memo Set time window
          @param width Window duration
          @param offset Window offset
       ******************************************************************/
      void SetWindow (const Interval& width, const Interval& offset) {
         mWindow = TimeWindow (width, offset); }
      /** Set the time offset.
          @memo Set time offset
       ******************************************************************/
      void SetOffset (const Interval& offset) {
         mWindow.SetOffset (offset); }
      /** Get the time offset.
          @memo Get time offset
       ******************************************************************/
      Interval GetOffset () const {
         return mWindow.GetOffset(); }
      /** Set the time duration.
          @memo Set time duration
       ******************************************************************/
      void SetWidth (const Interval& duration) {
         mWindow.SetOffset (duration); }
      /** Get the time duration.
          @memo Get time duration
       ******************************************************************/
      Interval GetWidth () const {
         return mWindow.GetWidth(); }
   
   private:
      /// Event condition 
      ConditionPtr	mCond;
      /// Cluster threshold
      int		mThreshold;
      /// Time window
      TimeWindow	mWindow;
   };


}

#endif // _LIGO_EVENTCLUSTER_H
