//////////////////////////////////////////////////////////////////////////
//  									//
//  Chain								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include <stdio.h>
#include <fstream>
#include "events/Chain.hh"
#include "events/Algorithm.hh"


namespace events {

//______________________________________________________________________________
   class IteratorImpChain : public IteratorImp {
   
   public:
      // List iterator
      typedef List::iterator list_iterator;
      // Create event list iterator implementation
      IteratorImpChain (Chain& chain, int lnum, 
      list_iterator i = list_iterator()) 
      : mChain (chain), mListNum (lnum), mIter (i) {
      }
      // Copy
      virtual IteratorImp* Copy() const {
         return new IteratorImpChain (*this); }
      // Get
      virtual Event* Get() const {
         return (mListNum >= 0) ? &*mIter : 0; }
      // Increment
      virtual void Inc() {
         if (mListNum < 0) 
            return;
         ++mIter; 
         while ((mIter == mChain.GetList(mListNum).End()) &&
         (mListNum + 1 < mChain.N())) {
            mIter = mChain.GetList(++mListNum).Begin();
         }
      }
      // Decrement
      virtual void Dec() {
         if (mListNum < 0) 
            return;
         while ((mIter == mChain.GetList(mListNum).Begin()) &&
         (mListNum > 0)) {
            mIter = mChain.GetList(--mListNum).End();
         }
         --mIter; 
      }
      // Get list iterator
      list_iterator& GetListIterator() {
         return mIter; }
      // Get list number
      int GetListNumber() const {
         return mListNum; }
   private:
      // Refernce to original chain
      Chain&		mChain;
      // List number of current iterator
      int		mListNum;
      // current iterator
      list_iterator	mIter;
   };

//______________________________________________________________________________
   bool Chain::AddList (const char* filename)
   {
      mChain.push_back (List());
      return mChain.back()->Load (filename);
   }

//______________________________________________________________________________
   bool Chain::AddList (const List& eventlist)
   {
      mChain.push_back (eventlist);
      return true;
   }

//______________________________________________________________________________
   bool Chain::RemoveList (int lnum)
   {
      if ((lnum < 0) || (lnum >= N())) {
         return false;
      }
      mChain.erase (mChain.begin() + lnum);
      return true;
   }

//______________________________________________________________________________
   void Chain::Merge ()
   {
      if (N() <= 1) {
         return;
      }
      for (eventchain::iterator i = mChain.begin() + 1;
      i != mChain.end(); ++i) {
         for (List::iterator j =(*i)->Begin(); j != (*i)->End(); ++j) {
            mChain[0]->PushBack (Event());
            std::swap (mChain[0]->Back(), *j);
         }
      }
      while (N() > 1) {
         mChain.pop_back();
      }
   }

//______________________________________________________________________________
   bool Chain::Configure (const char* filename)
   {
   // :TODO:
      return false;
   }

//______________________________________________________________________________
   bool Chain::Save (const char* filename, int perfile, 
   int maxevents) const
   {
      // write a single file
      if (perfile <= 0) {
         std::ofstream out (filename);
         if (!out) {
            return false;
         }
         Write (out, Begin(), End(), maxevents);
         return !!out;
      }
      // write multiple files
      else {
         ConstIterator next = Begin();
         for (int i = 0; (next != End()) && maxevents; ++i) {
            char name[4*1024];
            sprintf (name, "%s.%i", filename, i);
            std::ofstream out (name);
            if (!out) {
               return false;
            }
            next = Write (out, next, End(), perfile);
            if (maxevents >= 0) {
               maxevents -= perfile;
               if (maxevents < 0) maxevents = 0;
            }
            if (!out) {
               return false;
            }
         }
         return true;
      }
   }
//______________________________________________________________________________
   bool Chain::Restore (const char* filename) {
      Clear ();
      return AddList (filename);
   }

//______________________________________________________________________________
   int Chain::Size() const {
      int n = 0;
      for (eventchain::const_iterator i = mChain.begin();
      i != mChain.end(); ++i) {
         n += (*i)->Size();
      }
      return n;
   }

//______________________________________________________________________________
   bool Chain::Empty() const
   {
      for (eventchain::const_iterator i = mChain.begin();
      i != mChain.end(); ++i) {
         if (!(*i)->Empty()) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool Chain::operator== (const Chain& l) const
   {
      if (Size() != l.Size()) {
         return false;
      }
      const_iterator j = l.Begin();
      for (const_iterator i = Begin(); i != End(); ++i, ++j) {
         if (*i != *j) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool Chain::CheckOrder() const
   {
      return events::CheckOrder (Begin(), End());
   }

//______________________________________________________________________________
   void Chain::Swap (Chain& l)
   {
      std::swap (mChain, l.mChain);
   }

//______________________________________________________________________________
   Chain::iterator Chain::LowerBound (const Event& e)
   {
      // :TODO: more efficient algorithm
      return std::lower_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Chain::const_iterator Chain::LowerBound (const Event& e) const
   {
      // :TODO: more efficient algorithm
      return std::lower_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Chain::iterator Chain::UpperBound (const Event& e)
   {
      // :TODO: more efficient algorithm
      return std::upper_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   Chain::const_iterator Chain::UpperBound (const Event& e) const
   {
      // :TODO: more efficient algorithm
      return std::upper_bound (Begin(), End(), e);
   }

//______________________________________________________________________________
   void Chain::Sort()
   {
      Merge();
      if (N() == 1) {
         GetList(0).Sort();
      }
   }

//______________________________________________________________________________
   void Chain::Sort (const Function& func, bool ascending)
   {
      Merge();
      if (N() == 1) {
         GetList(0).Sort (func, ascending);
      }
   }

//______________________________________________________________________________
   Chain::reference Chain::At (size_type idx)
   {
      size_type pos = 0;
      eventchain::iterator i = mChain.begin();
      for (; i != mChain.end(); ++i) {
         if (idx < pos + (*i)->Size()) {
            break;
         }
         pos += (*i)->Size();
      }
      if (i == mChain.end()) {
         return *(value_type*)0;
      } 
      else {
         return (*i)->At(idx - pos);
      }
   }

//______________________________________________________________________________
   Chain::const_reference Chain::At (size_type idx) const
   {
      size_type pos = 0;
      eventchain::const_iterator i = mChain.begin();
      for (; i != mChain.end(); ++i) {
         if (idx < pos + (*i)->Size()) {
            break;
         }
         pos += (*i)->Size();
      }
      if (i == mChain.end()) {
         return *(value_type*)0;
      } 
      else {
         return (*i)->At(idx - pos);
      }
   }


//______________________________________________________________________________
   Chain::reference Chain::Front()
   {
      eventchain::iterator i = mChain.begin();
      while ((i != mChain.end()) && (*i)->Empty()) {
         ++i;
      }
      return (*i)->Front();
   }

//______________________________________________________________________________
   Chain::const_reference Chain::Front() const
   {
      eventchain::const_iterator i = mChain.begin();
      while ((i != mChain.end()) && (*i)->Empty()) {
         ++i;
      }
      return (*i)->Front();
   }

//______________________________________________________________________________
   Chain::reference Chain::Back()
   {
      eventchain::reverse_iterator i = mChain.rbegin();
      while ((i != mChain.rend()) && (*i)->Empty()) {
         ++i;
      }
      return (*i)->Back();
   }

//______________________________________________________________________________
   Chain::const_reference Chain::Back() const
   {
      eventchain::const_reverse_iterator i = mChain.rbegin();
      while ((i != mChain.rend()) && (*i)->Empty()) {
         ++i;
      }
      return (*i)->Back();
   }

//______________________________________________________________________________
   Chain::iterator Chain::Begin()
   {
      if (N() == 0) {
         return iterator (IteratorImpChain (*this, -1));;
      }
      size_type lnum = 0;
      while ((lnum + 1 < mChain.size()) && mChain[lnum]->Empty()) {
         ++lnum;
      }
      return iterator (IteratorImpChain 
         (*this, lnum, mChain[lnum]->Begin()));
   }

//______________________________________________________________________________
   Chain::const_iterator Chain::Begin() const
   {
      return const_iterator (((Chain*)this)->Begin());
   }

//______________________________________________________________________________
   Chain::iterator Chain::End()
   {
      if (N() == 0) {
         return iterator (IteratorImpChain (*this, -1));
      }
      eventchain::iterator i = mChain.end() - 1;
      return iterator (IteratorImpChain (*this, N() - 1, (*i)->End()));
   }

//______________________________________________________________________________
   Chain::const_iterator Chain::End() const
   {
      return const_iterator (((Chain*)this)->End());
   }

//______________________________________________________________________________
   Chain::iterator Chain::Insert (const Event& event)
   {
      iterator pos = UpperBound (event);
      return Insert (pos, event);
   }

//______________________________________________________________________________
   Chain::iterator Chain::Insert (iterator pos, const Event& event)
   {
      IteratorImpChain* imp = 
         dynamic_cast<IteratorImpChain*> (pos.GetImplementation());
      if (imp == 0) {
         return End();
      }
      if (imp->GetListNumber() < 0) {
         PushBack (event);
         return --End();
      }
      else {
         List::iterator j = mChain[imp->GetListNumber()]->Insert (
            imp->GetListIterator(), event);
         return iterator (IteratorImpChain (*this, 
            imp->GetListNumber(), j));
         //return pos;
      }
   }

//______________________________________________________________________________
   void Chain::Insert (iterator beg, iterator end)
   {
      for (iterator i = beg; i != end; ++i) {
         Insert (*i);
      }
   }

//______________________________________________________________________________
   void Chain::PushBack (const Event& event)
   {
      if (N() == 0) {
         AddList (List());
      }
      mChain.back()->PushBack (event);
   }

//______________________________________________________________________________
   Chain::iterator Chain::Erase (iterator pos)
   {
      IteratorImpChain* imp = 
         dynamic_cast<IteratorImpChain*> (pos.GetImplementation());
      if ((imp == 0) || (imp->GetListNumber() < 0)) {
         return End();
      }
      List::iterator i = 
         mChain[imp->GetListNumber()]->Erase (imp->GetListIterator());
      if (i != mChain[imp->GetListNumber()]->End()) {
         return iterator (IteratorImpChain (*this, 
            imp->GetListNumber(), i));
      }
      else if (imp->GetListNumber() + 1 == N()) {
         return End();
      }
      else {
         int lnum = imp->GetListNumber() + 1;
         while ((lnum + 1 < N()) && mChain[lnum]->Empty()) {
            ++lnum;
         }
         return iterator (IteratorImpChain 
            (*this, lnum, mChain[lnum]->Begin()));
      }
   }

//______________________________________________________________________________
   Chain::iterator Chain::Erase (iterator beg, iterator end)
   {
      int n = 0;
      for (iterator i = beg; i != end; ++i) ++n;
      iterator j = beg;
      for (int i = 0; i < n; ++i) {
         j = Erase (j);
      }
      return j;
      // iterator ret = End();
      // iterator i = end;
      // while (i != beg) {
         // ret = Erase (--i);
      // }
      // return ret;
   }

//______________________________________________________________________________
   void Chain::PopBack()
   {
      while ((N() > 0) && mChain.back()->Empty()) {
         mChain.pop_back();
      }
      if (N() > 0) {
         mChain.back()->PopBack();
      }
   }

}
