/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xmlform							*/
/*                                                         		*/
/* Module Description: web server class for monitor data		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 4Jan02   D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: xmlform.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_XMLFORM_H
#define _LIGO_XMLFORM_H

#include "web/webserver.hh"
#include "Time.hh"
#include <iosfwd>
#include <string>

namespace web {

   const int DEFAULT_THREADS = 5;


/** @name Web server class for editing XML parameter files
    This web server displays parameter xml files as html forms.

    @memo XML paramter web server
    @author Written March 2002 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xmlform_server : public basic_server
   {
   public:
      /// File listing
      typedef std::map<std::string, std::string> listing;
      /// Post list
      typedef std::map<std::string, std::string> post_list;
   
      /** Constructs a XML paramter web server.
          
          @memo Constructor
          @param argn Number of cmd line arguments
          @param argv list of cmd line arguments
          @param basedir Directory base for xml files
          @param ext File extension for xml files
       ******************************************************************/
      xmlform_server (int argn, char** argv,
                     const char* title = "XML Parameter Configuration",
                     const char* basedir = ".", 
                     const char* ext = "xml") 
      : basic_server (argn, argv, DEFAULT_THREADS),
      fTitle (title), fBaseDir (basedir), fXmlExt (ext)
      {
      }
   
   protected:
      /** Process a GET request.
          Can be overriden by descendents.
          @memo Process GET request
          @param request requested page
          @param answer Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestGet (const request_t& request,
                        response_t& response);
      /** Process a POST request.
          Can be overriden by descendents.
          @memo Process POST request
          @param request requested page
          @param answer Web page (return)
          @return True if request could be processed
       ******************************************************************/
      virtual bool RequestPost (const request_t& request,
                        response_t& response);
   
      /** Updates the list of xml files.
          @memo Update xml file names
          @return True if successful
       ******************************************************************/
      virtual bool UpdateListing();
   
      /** Get the update time.
          @memo Update time
          @return Time of last update
       ******************************************************************/
      Time GetUpdateTime() const;
   
      /** Parse the POST return string of a form.
          @memo Parse post message
          @param prm form paramters (return)
          @param p Post message
          @param len Length of post message
          @return True if successful
       ******************************************************************/
      virtual bool parsePost (post_list& list, const char* p, int len);
   
      /** Generates the index page.
          @memo Generate index page
          @param os Output stream
          @return True if successful
       ******************************************************************/
      virtual bool index (std::ostream& os);
      /** Generates the XML forms.
          @memo Generate XML forms
          @param os Output stream
          @param filename Name of XML file
          @return True if successful
       ******************************************************************/
      virtual bool form (std::ostream& os, const std::string& filename);
      /** Generates the XML forms.
          @memo Generate XML forms
          @param os Output stream
          @param filename Name of XML file
          @param container Name of XML container
          @return True if successful
       ******************************************************************/
      virtual bool form (std::ostream& os, const std::string& filename,
                        const std::string& container);
   
      /// Title
      std::string	fTitle;
      /// Base directory list
      std::string	fBaseDir;
      /// XML file extension
      std::string	fXmlExt;
      /// XML file listing
      listing		fListing;
      /// Time stamp when listing was last read
      Time		fListingTime;
   };

}
#endif // _LIGO_XMLFORM_H
