#ifndef _LIGO_MONITORAPI_H
#define _LIGO_MONITORAPI_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: monitorapi						*/
/*                                                         		*/
/* Module Description: monitor API					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 03Jun00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: monitorapi.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "Time.hh"
#ifndef __CINT__
#include "xml/Xsil.hh"
#endif
#include <string>
#include <map>

   class TSeries;
   class FSeries;
   class FSpectrum;
   class Histogram1;
   class ParameterDescriptor;
   class PlotDescriptor;

namespace calibration {
   class Descriptor;
}

namespace monapi {

   class monaccess;


/** @name TLGMonitor
    This header exports class used to get data from data monitor 
    servers.

    @memo Data monitor interface
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/

//@{

#ifndef __CINT__
   const char* const kSeriesTypenames[4] = 
   {"TSeries", "FSeries", "FSpectrum", "Histogram1"};
   const char* const kSeriesTypenamesAlt[4] = 
   {"TimeSeries", "FrequencySeries", "Spectrum", "Histogram1D"};
#endif

/** Describes a data object of a data monitor server.
   
    @memo Monitor datum
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class TLGMonitorDatum : public std::pair<std::string, std::string> {
   public:
      /// name type
      typedef std::pair<std::string, std::string> name_t;
   
      /// Monitor update type
      enum MonitorUpdate_t {
      /// Never
      kMonUpdateNever = 0,
      /// Update only upon user request
      kMonUpdateUser = 1,
      /// Update whenever data has changed
      kMonUpdateWhenChanged = 2,
      /// Upadate in fixed intervals
      kMonUpdateRepeat = 3
      };
   
      enum SeriesType {
      TimeSeries = 0,
      FrequencySeries = 1,
      Spectrum = 2,
      Histogram1D = 3
      };
   
   protected:
      /// Data type
      SeriesType 	fType;
      /// Update type
      MonitorUpdate_t	fUpdate;
      /// Update interval
      double		fUpdateInterval;
      /// Time of last update
      Time		fUpdateTime;
      /// Graph Type
      std::string	fGraph;
      /// A channel name
      std::string	fAChn;
      /// A channel name
      std::string	fBChn;
      /// Parameter descriptor
      ParameterDescriptor*	fParam;
      /// Calibration descriptor
      calibration::Descriptor*	fCal;
      /// Monitor client
      monaccess*	fMC;
   
      /// Initialization routine 
      virtual void Init();
      /// Analyze option string
      virtual void SetOpt (const char* opt);
   public:
   
      /// Default constructor
      TLGMonitorDatum () : fUpdate (kMonUpdateNever) {
         Init();
      }
      /// Copy constructor
      TLGMonitorDatum (const TLGMonitorDatum& mon) 
      : fUpdate (kMonUpdateNever), fParam(0), fCal (0) {
         *this = mon;
      }
   
      /// Constructor
      TLGMonitorDatum (const char* monitor, const char* dobj,
                      SeriesType type = TimeSeries, 
                      const char* opt = 0)
      : name_t (monitor, dobj), fType (type), fUpdate (kMonUpdateRepeat) {
         Init(); Connect (); SetOpt (opt); }
      /// Desctructor
      virtual ~TLGMonitorDatum();
      /// Assignment operator
      TLGMonitorDatum& operator= (const TLGMonitorDatum& mon);
   
      /// lookup monitor server name
      virtual bool Connect ();
      /// Get the monitor name
      virtual const char* GetMonitorName() const {
         return first.c_str(); }
      /// Set the monitor name
      virtual void SetMonitorName (const char* n) {
         first = n; }
      /// Get data object name
      virtual const char* GetDataObjectName() const {
         return second.c_str(); }
      /// Set data object name
      virtual void SetDataObjectName (const char* n) {
         second = n; }
      /// Get the graph type
      virtual const char* GetGraphType() const {
         return fGraph.c_str(); }
      /// Set the graph type
      virtual void SetGraphType (const char* graph) {
         fGraph = graph; }
      /// Get the A channel name
      virtual const char* GetAChannel() const {
         return fAChn.c_str(); }
      /// Set the A channel name
      virtual void SetAChannel (const char* a) {
         fAChn = a; }
      /// Get the B channel name
      virtual const char* GetBChannel() const {
         return fBChn.empty() ? 0 : fBChn.c_str(); }
      /// Set the B channel name
      virtual void SetBChannel (const char* b) {
         fAChn = b ? b : ""; }
      /// Set data object name, type and options
      virtual void Set (const char* monitor, const char* dobj,
                       SeriesType type, const char* opt = 0) {
         ((name_t&)*this) = name_t (monitor, dobj); fType = type;
         SetOpt (opt); }
   
      /// Get type
      virtual SeriesType GetType() const {
         return fType; }
      /// Set type
      virtual void SetType (SeriesType mtype) {
         fType = mtype; }
      /// Get type name
      virtual std::string GetTypename() const {
         return kSeriesTypenames[fType]; }
      /// Get monitor update option
      virtual MonitorUpdate_t GetUpdateOpt() const {
         return fUpdate; }
      /// Get monitor update option
      virtual void SetUpdateOpt (MonitorUpdate_t up) {
         fUpdate = up; }
      /// Get monitor update time
      virtual Time GetUpdateTime() const {
         return fUpdateTime; }
      /// Get monitor update interval
      virtual double GetUpdateInterval() const {
         return fUpdateInterval; }
      /// Get monitor update interval
      virtual void SetUpdateInterval (double ival) {
         fUpdateInterval = ival; }
      /// True if ready to poll new data
      virtual bool Ready() const;
      /// Updates the time stamp
      virtual void Touch();
      /// Get data array: descriptor will be owned by caller!
      virtual PlotDescriptor* GetData ();
   
      /** Write a monitor object to output stream.
          @memo XML output of monitor object
          @param os output stream
          @param mon monitor object
          @param index monitor number
          @return true if successful
       ******************************************************************/
      bool write (std::ostream& os, int index = -1) const;
   };


/** Describes a list of data monitor objects.
   
    @memo Monitor datum list
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class TLGMonitorDatumList {
   private:
      TLGMonitorDatumList (const TLGMonitorDatumList&);
      TLGMonitorDatumList& operator= (const TLGMonitorDatumList&);
   
   public:
      /// auto pointer to monitor
      typedef TLGMonitorDatum::name_t name_t;
      /// auto pointer to monitor
      typedef TLGMonitorDatum* data_ptr;
      /// list
      typedef std::map <name_t, data_ptr> data_list;
      /// iterator
      typedef data_list::iterator iterator;
      /// const iterator
      typedef data_list::const_iterator const_iterator;
   
      /// Default constructor
      TLGMonitorDatumList () {
      }
      /// Destructor
      virtual ~TLGMonitorDatumList () {
         clear(); }
      /// Begin
      iterator begin() {
         return fList.begin(); }
      const_iterator begin() const {
         return fList.begin(); }
      /// End
      iterator end() {
         return fList.end(); }
      const_iterator end() const {
         return fList.end(); }
      /// Empty
      bool empty() const {
         return fList.empty(); }
   
      /// Add a monitor datum to list (will be adopted)
      void add (TLGMonitorDatum* mon);
      /// Remove a monitor datum from list 
      void remove (const TLGMonitorDatum::name_t& mon);
      /// Find a monitor datum from list 
   
      iterator find (const TLGMonitorDatum::name_t& mon) {
         return fList.find (mon); }
      /// Clear all elements
      void clear();
   
   private:
      /// Data list
      data_list		fList;
   };


/** Write a monitor object to output stream.
    @memo XML output of monitor object
    @param os output stream
    @param mon monitor object
    @return output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, 
                     const TLGMonitorDatum& mon);

/** Write a monitor list to output stream.
    @memo XML output of monitor list
    @param os output stream
    @param mon monitor object
    @return output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, 
                     const TLGMonitorDatumList& mlist);

#ifndef __CINT__
/** XSil handler for monitor object.
   
    @memo monitor object handler
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandlerMonitor : public xml::xsilHandler {
   protected:
      /// Monitor list
      TLGMonitorDatumList*	fMList;
      /// Monitor record	
      TLGMonitorDatum*		fMon;
   public:
      /// Constructor
      explicit xsilHandlerMonitor (TLGMonitorDatumList& mon,
                        TLGMonitorDatum::SeriesType mtype);
      /// Destructor
      virtual ~xsilHandlerMonitor();
   
      /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
      /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
      /// short parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
      /// int parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const int& p, int N = 1);
   #ifndef __CINT__
      /// long parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const long long& p, int N = 1) {
         return false; }
   #endif
      /// float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const float& p, int N = 1) {
         return false; }
      /// double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const double& p, int N = 1);
      /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
      /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
      /// string parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr, const std::string& p);
      /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name,
                        const attrlist& attr,
                        unsigned long sec, unsigned long nsec) {
         return false; }
   };


/** Xsil monitor handler query class.
    The query will return a handler, if the data object name starts
    with 'Monitor' and the type is recognized.

    @memo Xsil monitor handler query.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandlerQueryMonitor : public xml::xsilHandlerQuery {
   protected:
      /// Monitor list
      TLGMonitorDatumList*	fMon;
   public:
      /// Constructor
      xsilHandlerQueryMonitor (TLGMonitorDatumList* mon) 
      : fMon (mon) {
      }
      /// returns a handler for the specified object (or 0 if not)
      virtual xml::xsilHandler* GetHandler (const attrlist& attr);
   };
#endif


/** Manages a name list of monitor servers and available data objects.
   
    @memo Monitor name manager
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class TLGMonitorMgr {
   public:
      /// Series type
      typedef TLGMonitorDatum::SeriesType SeriesType;
      /// Data obejct element type: pair of type and option
      struct dobject {
         /// Type
         SeriesType	fType;
         /// Option
         std::string	fOption;
      };
      /// List of data objects (element = id)
      typedef std::map<std::string, dobject> dobjlist;
      /// class for storing an object list
      class dobjects {
      protected:
         /// initialized?
         bool fInit;
         /// list of data objects
         dobjlist fDObjects;
      public:
         /// Constructor
         dobjects () : fInit (false) {
         }
         /// Get init
         bool Initialized () const {
            return fInit; }
         /// Set init
         void SetInit (bool state = true) {
            fInit = state; }
         /// clear
         void clear() {
            fInit = false; fDObjects.clear(); }
         /// insert 
         void insert (dobjlist::value_type obj) {
            fDObjects.insert (obj); }
         /// find 
         dobjlist::const_iterator find (const std::string& name) const {
            return fDObjects.find (name); }
         /// begin iterator
         dobjlist::const_iterator begin() const {
            return fDObjects.begin(); }
         /// end iterator
         dobjlist::const_iterator end() const {
            return fDObjects.end(); }
      };
      /// List of services (each element is a data object list)
      typedef std::map<std::string, dobjects> servicelist;
   
   protected:
      /// Monitor client
      monaccess*	fMC;
      /// List of available servers and data objects
      servicelist	fServices;
   
   public: 
      /// Constructor
      TLGMonitorMgr (bool connect = false) : fMC (0) {
         if (connect) UpdateAll(); }
      /// Desctructor
      virtual ~TLGMonitorMgr () {
      }
      /// Get list of services
      const servicelist& List() const {
         return fServices; }
      /// Update list of servers/services
      virtual bool UpdateServices ();
      /// Checks if data objects of a service have been obtained
      virtual bool Initialized (const char* service);
      /// Update list of specified data object
      virtual bool UpdateDObjects (const char* service);
      /// Update list of services and all data objects
      virtual bool UpdateAll();
   };


//@}
}

#endif
