/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FRWRITER_HH
#define FRWRITER_HH

#include "Time.hh"
#include "Interval.hh"
#include "Channel.hh"
#include "FrVectRef.hh"

namespace FrameCPP {
    namespace Common {
	class OFrameStream;
	class FrameBufferInterface;
    }
}

#include <string>
#include <iosfwd>
#include <list>
#include <vector>

#define DACC_ONLDEV "/online/"

class ChanList;   // Hide the channel list from root.
class TSeries;
class FSeries;
class FSpectrum;
class FrStatDataRef;

/**  %FrWriter methods are used to create frames from TSeries objects and 
  *  other data. The %FrWriter API is used as follows. A frame writer
  *  is created and then opened with the open() method. Any TSeries that
  *  are to be written to all output frames are introduced to the frame
  *  writer with the addChannel() function. For each frame to be written 
  *  user must build the frame with buildFrame(), add any additional 
  *  TSeries data with addSeries() or other data using the FrameCPP API
  *  and the frame pointer from getFrame() and write it out with 
  *  writeFrame(). After all frames have been written, the frame writer 
  *  is closed with close() and/or deleted.
  *
  *  A sample code segment is shown below.
  *  \verbatim
     TSeries ts(Now(), Interval(1./16384.), 16384, Sine(1000.0));
     FrWriter fw("TestFrame", -1);
     fw.buildFrame(ts.getStartTime(), Interval(1.0));
     fw.addRawSeries("Sine_1kHz", ts);
     fw.addHistory("FrWriterExample", Now(), "FrWriter usage example");
     fw.addWriterHistory();
     fw.open("TestFrame.gwf", true);
     fw.writeFrame();
     fw.close();     \endverbatim
  *
  *  The above code creates a frame in file "TestFrame.F" with a frame ID 
  *  of "TestFrame" and a run number of -1. A single time raw data channel
  *  named "Sine_1kHz" with a 1kHz sine wave is written to the frame. History
  *  records are written describing the program used (specified explicitly
  *  with addHistory()) and the %FrWriter version (specified by 
  *  addWriterHistory()). The negative run number specified in the 
  *  constructor indicates that the frame contains Monte Carlo generated data.
  *
  *  If the file name is of the form "/online/\<partition\>" the frame data 
  *  will be written to a Ligo shared memory partition. Apart from the file
  *  names, writing to shared memory is handled identically to writing to a
  *  file. There is currently no mechanism available to hold an output stream
  *  open while switching shared memory buffers so the frame writer must be
  *  close and re-opened at the end of each frame.
  *  @memo DMT frame writer API.
  *  @author  John G Zweizig
  *  @version 1.3; Last modified April 27, 2006
  */
class FrWriter 
{
  public:
    typedef frameh_pointer frame_ptr_type;
    typedef FrameCPP::Common::OFrameStream frame_ostream;

  public:
    //--------------------------------------  Constructors and destructors.
    /**  Default (Null) constructor.
      */
    FrWriter(void);

    /**  Construct a frame writer and connect it to a specified destination 
      *  file. Any frame name may be used. The run number should follow the 
      *  frame standard run number convention. - negative run numbers for 
      *  simulated data.
      *  \brief Construct a frame writer.
      *  \param Name Frame name.
      *  \param Run  Current run number.
      */
    FrWriter(const char* Name, int Run);

    /**  Destroy a frame writer
      *  \brief Destructor.
      */
    ~FrWriter(void);

    //---------------------------------------  Status manipulation
    /**  If the debug flag is set, explanatory messages will be printed 
      *  when an error is encountered.
      *  @memo Set the debug flag.
      *  \param debug Debug print level.
      */
    void setDebug(int debug=0);

    //--------------------------------------  Channel list manipulation
    /**  The specified channel is added to the list of channels to be
      *  processed by buildFrame. A channel may be included in the list 
      *  only once. Subsequent calls to addChannel() will replace the 
      *  original request. \a TSptr specifies the location of a pointer
      *  to the data. Note that the data are not copied until buildFrame
      *  is called. Core must be taken that the specified pointer (\a TSptr) 
      *  must therefore still exists and points to valid data at the time
      *  buildFrame() is called. The data are written with the same type 
      *  as the %TSeries.
      *  @memo  Add a channel to the request list.
      *  @param Name  Name of channel to collect data from.
      *  @param TSptr Address of %TSeries pointer.
      */
    void addChannel(const char* Name, TSeries** TSptr=0);

    /**  The named channel is removed from the request list.
      *  \brief Remove a channel from the request list.
      *  \param Name Name of channel to be removed from the request list.
      */
    void rmChannel(const char* Name);

    //--------------------------------------  File list manipulation
    /**  The file is opened for output. If a file is already 
      *  open, it will be closed, and a new file will be opened. 
      *  \brief Open a file.
      *  \param File name of output file.
      *  \param toc  If true, write a table of contents.
      */
    int open(const char* File, bool toc=false);

#ifndef __CINT__
    /**  The specified framewriter is used for output. If a file is already
      *  open the current frame writer is closed and deleted.
      *  @memo Open a file specified by a framewriter.
      *  \param writer Pointer to frame writer.
      */
    int open(frame_ostream* writer);
#endif

    //----------------------------------  Data processing.
    /**  The current data file is closed and the frame writer is deleted.
      *  @memo Close the current data file.
      */
    void close(void);

    /**  The current frame is deleted without being written.
      *  @memo Erase the current data frame.
      */
    void erase(void);

    /**  Data from TSeries specified by addChannel() are copied to the
      *  output frame.
      *  @memo Fill the requested TSeries.
      */
    int writeFrame(void);

    /**  A frame is constructed and data from all TSeries specified with
      *  addChannel are copied into the frame.
      *  @memo Create a data frame from the data.
      *  @return Zero if valid frame read in, otherwise -1.
      *  @param Start Frame start time.
      *  @param dT    Frame length.
      */
    int buildFrame(const Time& Start=Time(0), const Interval& dT=0.0);

    /**  Add a frequency series to the frame from a FSeries.
      *  @memo Add a processed data frequency series to the frame.
      *  @param Name Name of ProcData to be added.
      *  @param fs   Frequency series containing pre-processed data.
      */
    void addFrequencySeries(const char* Name, const FSeries& fs);

    /**  Add a history record as defined by the arguments to the current frame.
      *  @memo Add a history record.
      *  @param Name Name of History record.
      *  @param t    Time of post processing.
      *  @param cmt  Additional comments.
      */
    void addHistory(const std::string& Name, const Time& t, 
		    const std::string& cmt);

    /**  Add a processed data structure.
      *  @memo Add a processed data structure.
      *  @param Name    Name of ProcData to be added.
      *  @param comment FrProcData comment.
      *  @param type    FrProcData type code
      *  @param subtype FrProcData subtype code
      *  @param Start   Start time.
      *  @param End     End time.
      *  @param fShift  Heterodyne frequency
      *  @param fPhase  Heterodyne phase
      *  @param fRange  Frequency range
      *  @param bw      Bandwidth
      *  @param vect    Vector reference.
      */
    void addProcData(const std::string& Name, const std::string& comment, 
		     int type, int subtype, const Time& Start, const Time& End,
		     double fShift, double fPhase, double fRange, double bw,
		     FrVectRef& vect);

    /**  Add a pre-processed data series to the frame from a TSeries.
      *  @memo Add a pre-processed data structure to the frame.
      *  @param Name Name of ProcData to be added.
      *  @param TS   Time series containing pre-processed data.
      */
    void addProcSeries(const char* Name, const TSeries& TS);

    /**  Add a raw data channel to the frame from a TSeries.
      *  @memo Add an ADC data structure.
      *  @param Name Name of ADC to be added.
      *  @param TS   Time series containing Adc data.
      */
    void addRawSeries(const char* Name, const TSeries& TS);

    /**  Add a simulated data channel to the frame from a TSeries.
      *  @memo Add a simulated data structure.
      *  @param Name    Name of Simulated channel to be added.
      *  @param Comment Description of event
      *  @param Inputs  Description of input data
      *  @param GPSMax  GPS time of event maximum
      *  @param before  Interval before maximum
      *  @param after   Interval after maximum
      *  @param Amp     Event amplitude
      *  @param v       Other parameter values.
      */
    void addSimEvent(const char* Name, const char* Comment, 
		     const char* Inputs, const Time& GPSMax, Interval before, 
		     Interval after, float Amp, 
		     const std::vector< std::pair<std::string, float> >& v);

    /**  Add a simulated data channel to the frame from a %TSeries.
      *  @memo Add a simulated data structure.
      *  @param Name    Name of Simulated channel to be added.
      *  @param TS      Time series containing simulated data.
      */
    void addSimSeries(const char* Name, const TSeries& TS);

    /**  Add a raw data channel to the frame from a TSeries.
      *  This is equivalent to addRawSeries(), but kept for backward 
      *  compatibility.
      *  @memo Add an ADC data structure.
      *  @param Name Name of ADC to be added.
      *  @param TS   Time series containing Adc data.
      */
    void addSeries(const char* Name, const TSeries& TS);

    /**  Add a spectrum from a FSpectrum object to the frame in a 
      *  FrProcData object.
      *  @memo Add a spectrum to the frame.
      *  @param Name Name of ProcData to be added.
      *  @param sp   FSpectrum containing pre-processed data.
      */
    void addSpectrum(const char* Name, const FSpectrum& sp);

    /**  Add an FrStat object.
      *  @memo Add a static data structure to the output frame.
      *  @param s Static data reference.
      */
    void addStatic(FrStatDataRef& s);

    /**  Write an FSeries as an FrStatData structure.
      *  @memo Write a static frequency series.
      *  @param Name     Name of FrStatData to be added.
      *  @param version  Static data version.
      *  @param GpsStart Start of data validity.
      *  @param GpsEnd   End of data validity.
      *  @param det      Pointer to FrDetector for this FSeries.
      *  @param fs       FSeries containing pre-processed data.
      */
    void addStatic(const std::string& Name, unsigned long version,
		   const Time& GpsStart, const Time& GpsEnd, void* det,
		   const FSeries& fs);

    /**  Write a TSeries as an FrStatData structure.
      *  @memo Write a static time series.
      *  @param Name Name of FrStatData to be added.
      *  @param version  Static data version.
      *  @param GpsStart Start of data validity.
      *  @param GpsEnd   End of data validity.
      *  @param det      Pointer to FrDetector for this TSeries.
      *  @param ts   TSeries containing pre-processed data.
      */
    void addStatic(const std::string& Name, unsigned long version,
		   const Time& GpsStart, const Time& GpsEnd, void* det,
		   const TSeries& ts);

    /**  Add a detector record with the standard parameters for the instrument
      *  specified by the prefix.
      *  @memo Add a FrameCPP Detector record.
      *  @param iName Ifo prefix. Only the first two characters are used.
      *  @return Pointer to the inserted FrDetector object.
      */
    void* addStdDetector(const std::string& iName);

    /**  Add a history record with the current version of FrameCPP.
      *  @memo Add a FrameCPP History record.
      */
    void addWriterHistory(void);

     //---------------------------------  Accessors
    /**  Returns the level of debug information that was requested.
      *  @memo Get the debug flag.
      *  @return Debug print level.
      */
    int getDebug(void) const;

    /**  Returns a pointer to a string containing the current input file 
      *  name. If no file is open, a pointer to the next file name is 
      *  returned.
      *  @memo Get the input file name.
      *  \return File name pointer or NULL.
      */
    const char* getFile(void) const;

#if 0
    //  method not implemented or used.

    // /**  An identifier of the form \<ifo-id\>-\<GPS-time\>.F is generated 
    //  *  for the current Frame and copied to the specified string.
    //  *  @memo Copy a Frame ID into the specified string.
    //  *  @return String containing the current frame identifier.
    //  */
    // std::string getFrameID(void) const;
#endif

    /**  Returns the time at which data collection is to start.
      *  @memo Get the Start time.
      *  \return Data collection start time.
      */
    Time getTime(void) const;

    /**  Test whether the %FrWriter is open. The %FrWriter is considered to 
      *  be open if an open OFrameStream is pointed to by mWriter.
      *  @memo Test open status.
      *  @return true if the frame writer is open.
      */
    bool is_open(void) const;

    /**  A formatted list of requested channels, their decimation factors, 
      *  and latest time copied is printed to the specified output stream. 
      *  \brief List requested channels.
      *  \param out Reference to the output stream to which the channel 
      *             list will be written.
      *  \return Output stream reference.
      */
    std::ostream& list(std::ostream& out) const;

    /**  Set the check-sum mode for structure and file output.
      *  \brief Set check-sum modes
      *  \param ck_struct Structure checksum mode (0=None, 1=CRC32).
      *  \param ck_file   File checksum mode (0=None, 1=CRC32).
      */
    void setChecksum(int ck_struct=1, int ck_file=1);

    /**  Set the compression mode for data vectors.
      *  \brief Set compression mode
      *  \param cmode Enumerated compression mode.
      */
    void setCompress(FrVectRef::compression_mode cmode);

    /**  Set run name string.
      *  \brief Set run name.
      *  \param name Run name string.
      */
    void setName(const std::string& name);

    /**  Set run number and reset the frame count.
      *  \brief Set a new run number.
      *  \param run New run number.
      */
    void setRunID(int run);

    //----------------------------------  Copy data from another frame
#ifndef __CINT__
    /**  Copy the detector information (both processed and simulation) 
      *  from an existing frame.
      *  @memo Copy detector information.
      *  @param f Frame containing detector information to be copied.
      */
    void copyDetectors(const FrameCPP::FrameH& f);

    /**  Copy the history information from an existing frame.
      *  @memo Copy history information.
      *  @param f Frame containing history information to be copied.
      */
    void copyHistory(const FrameCPP::FrameH& f);

    /**  Copy the auxiliary data from an existing frame.
      *  \brief Copy axillary data.
      *  \param f Frame containing auxiliary data to be copied.
      */
    void copyAuxData(const FrameCPP::FrameH& f);

    /**  Copy the adc data from an existing frame.
      *  @memo Copy adc data.
      *  @param f Frame containing adc data to be copied.
      *  @param Name Name of adc to be copied.
      */
    void addRawAdc(frame_ptr_type f, const char* Name);

    /**  Add adc data specified in an FrAdc structure.
      *  @memo Copy raw data from an FrAdc structure.
      *  @param adc Adc structure containing data to be copied.
      */
    void addRawAdc(const FrameCPP::FrAdcData& adc);
#endif

  private:
    /**  mFile contains the output data file name
      *  @memo Output data file name.
      */
    std::string mFile;

    /**  Debug mode flag.
      */
    int mDebug;

    /**  Time of current (Next) Frame.
      *  @memo Time of current (Next) Frame.
      */
    Time mTime;

    /**  Frame sequence number from this writer.
      *  @memo Frame sequence number.
      */
    int mFrameID;

    /**  Frame Name.
      *  @memo Frame name.
      */
    std::string mName;

    /**  Run number.
      *  @memo Frame sequence number.
      */
    int mRun;

    /**  Time interval (length) of the frame.
      *  @memo Time interval.
      */
    Interval mFrLength;

    /**  List of channels to be written.
      *  @memo List of channels to be written.
      */
    std::list<Channel> mChanList;
    typedef std::list<Channel>::iterator chan_iter;
    typedef std::list<Channel>::const_iterator const_chan_iter;

#if LDAS_VERSION_NUMBER < 109360
    /**  Pointer to output stream object.
      *  @memo Pointer to output stream object.
      */
    std::ostream* mStream;
#else
    /**  Pointer to output stream buffer object.
      *  @memo Pointer to output stream buffer object.
      */
    FrameCPP::Common::FrameBufferInterface* mFrameBuf;
#endif

    /**  Pointer to current frame writer.
      *  @memo Pointer to current frame writer.
      */
    frame_ostream* mWriter;

    /**  Pointer to current frame. If no frame has been read, mFrame is
      *  set to NULL.
      *  @memo Pointer to current frame.
      */
    frame_ptr_type mFrame;

    /**  List of hanging FrStatData structures to clean up.
     */
    std::vector<frstatdata_pointer> mStatTBD;

    FrVectRef::compression_mode mCompress;
    bool mOnline;
    int  mChkSumStruct;
    int  mChkSumFile;
};

//======================================  Inline methods
#ifndef __CINT__
inline Time 
FrWriter::getTime(void) const {
    return mTime;
}

inline bool 
FrWriter::is_open(void) const {
    return mWriter;
}

inline void 
FrWriter::addSeries(const char* Name, const TSeries& TS) {
    addRawSeries(Name, TS);
}
#endif

#endif   //  FRWRITER_HH
