/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Sandbox data access implementation
//
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif
#include <strings.h>

#define GENERAL_FSTREAM 1

#include "DaccIn.hh"
#include "FSeries.hh"
#include "TSeries.hh"
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrHistory.hh"
#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif
#include "framecpp/Common/Verify.hh"
#if LDAS_VERSION_NUMBER >= 109360
#include "framecpp/Common/IOStream.hh"
using FrameCPP::Common::FrameBuffer;
template<class T>
T*
stream_ptr(FrameCPP::Common::FrameBufferInterface* p) {
    return dynamic_cast< FrameBuffer<T>* >(p);
}
#endif
#include "framecpp/IFrameStream.hh"
#include "FrVectRef.hh"
#include "FrStatDataRef.hh"
#include "DVecType.hh"
#ifndef DMTOFFLINE
#include "iSMbuf.hh"
#endif
#include <iostream>
#ifndef GENERAL_FSTREAM
#include <fstream>
typedef std::filebuf filebuffer_type;
#else
  #if LDAS_VERSION_NUMBER < 200000
    #include "general/fstream.hh"
    typedef General::filebuf filebuffer_type;
  #else
    #include "ldastoolsal/fstream.hh"
    typedef LDASTools::AL::filebuf filebuffer_type;
  #endif
#endif

using FrameCPP::FrAdcData;
using FrameCPP::FrameH;
using FrameCPP::IFrameStream;
using FrameCPP::FrProcData;
using FrameCPP::FrRawData;
using FrameCPP::FrSimData;
#include "framecpp/FrStatData.hh"
using FrameCPP::FrStatData;
#include <sstream>

using namespace std;

//======================================  Time object conversion functions.
inline Time
getTime(const GPSTime& t) {
    return Time(t.getSec(), t.getNSec());
}

inline GPSTime
getGPS(const Time t) {
    return GPSTime(t.getS(), t.getN());
}

//======================================  Default (Null) constructor.
DaccIn::DaccIn(void) 
  : mDebug(0), mT0(0), mFramesInFile(0), mFrameNo(0), mTOCMode(s_TOC),
 
#if LDAS_VERSION_NUMBER < 109360
    mSbuf(0), mStream(0), 
#else
    mFrameBuf(0),
#endif
#ifdef FCPP_SHARED_PTRS
    mReader(0), mTotalFrames(0)
#else
    mReader(0), mFrame(0), mTotalFrames(0)
#endif
{
#if LDAS_VERSION_NUMBER > 119020
    FrameCPP::Initialize();
#endif
    mSource = s_none;
}

//======================================  Construct and connect to source.
DaccIn::DaccIn(IFrameStream* reader) 
  : mDebug(0), mT0(0), mFramesInFile(0), mFrameNo(0), 
#if LDAS_VERSION_NUMBER < 109360
     mSbuf(0), mStream(0),
#else
    mFrameBuf(0),
#endif
#ifdef FCPP_SHARED_PTRS
    mReader(reader), mTotalFrames(0)
#else
    mReader(reader), mFrame(0), mTotalFrames(0)
#endif
{
    mSource = s_none;
    setTOCMode(false);
#if LDAS_VERSION_NUMBER > 119020
    FrameCPP::Initialize();
#endif
}

//======================================  Destructor.
DaccIn::~DaccIn(){
    close();
}

//======================================  Specify number of online buffers
void
DaccIn::setBuffer(int N) {
#ifndef DMTOFFLINE
    if (!isOnline()) return;
#if LDAS_VERSION_NUMBER < 109360
    dynamic_cast<iSMbuf*>(mSbuf)->setBCount(N);
#else
    stream_ptr<iSMbuf>(mFrameBuf)->setBCount(N);
#endif
#endif
}

//======================================  Set the TOC mode
void
DaccIn::setTOCMode(bool toc) {
    if (toc) mTOCMode = s_TOC;
    else     mTOCMode = s_noTOC;
}

//======================================  Get Time span of current frame.
Interval
DaccIn::getDt(void) const {
    if (!haveFrame()) return Interval(0.0);
    return Interval(mFrame->GetDt());
}

//======================================  Build the Frame name.
string
DaccIn::getFrameID(void) const {
    //----------------------------------  Find last '/'
    string::size_type inx = mLastFrame.find_last_of("/");
    if (inx == string::npos) return mLastFrame;

    //----------------------------------  Return portion after last '/'
    return mLastFrame.substr(inx+1);
}

//======================================  Read in static data
int 
DaccIn::getStaticData(const std::string& name,  const std::string& det, 
		      const Time& startGps, const Time& endGps) {
    if (!mReader) return 0;

#if LDAS_VERSION_NUMBER < 109360
    mReader->ReadFrStatData(name, det, getGPS(startGps), getGPS(endGps), 
			    FrStatData::LATEST_VERSION, mStatData);
#else
    mReader->ReadFrStatData(name, det, getGPS(startGps), getGPS(endGps), 
			    FrameCPP::Common::FrStatData::Query::LATEST_VERSION,
			    mStatData);
#endif

    int N = mStatData.size();
    for (int i=0; i<N; i++) mStatData[i];
    if (mDebug) {
        cerr << "Number of FrStatdata structures: " << N << endl;
	if (mDebug > 1) {
	    cerr << "Name  gps-start gps-stop version" << endl;
	    for (int i=0; i<N; i++) {
#if LDAS_VERSION_NUMBER < 109360
	        const FrStatData* p = mStatData[i];
		cerr << p->GetName()    << " " << p->GetTimeStart() << " "
		     << p->GetTimeEnd() << " " << p->GetVersion()   << endl;
#elif !defined(FCPP_SHARED_PTRS)
	        const FrStatData* p = dynamic_cast<FrStatData*>(mStatData[i]);
		cerr << p->GetName()    << " " << p->GetTimeStart() << " "
		     << p->GetTimeEnd() << " " << p->GetVersion()   << endl;
#else
	        FrStatDataRef p(mStatData[i]);
		cerr << p.getName()    << " " << p.getStartTime() << " "
		     << p.getEndTime() << " " << p.getVersion()   << endl;
#endif
	    }
	}
    }
    return N;
}

//======================================  Get a static FSeries
FSeries
DaccIn::getStaticFSeries(const std::string& name, const Time& gps) {
    FrStatDataRef p = findStat(name, gps);
    if (!p.null() && string(p.getRepresentation()) == "freq_series") {
	FrVectRef v = p.getFrVect();
	double   f0 = v.getDimX0(0);
	double   dF = v.getDimDx(0);
	Interval dT = double(p.getEndTime() - p.getStartTime());
	return FSeries(f0, dF, p.getStartTime(), dT, v.getDVector());
    }
    return FSeries();
}

//======================================  Get a static TSeries
TSeries
DaccIn::getStaticTSeries(const std::string& name, const Time& gps) {
    FrStatDataRef p = findStat(name, gps);
    if (!p.null() && string(p.getRepresentation()) == "time_series") {
	FrVectRef  v = p.getFrVect();
	Time      t0 = p.getStartTime() + Interval(v.getDimX0(0));
	Interval  dT = v.getDimDx(0);
	return TSeries(t0, dT, v.getDVector());
    }
    return TSeries();
}

//======================================  Connect to a data source.
bool 
DaccIn::getHistory(const std::string& name, Time& t, std::string& com) const {
    if (haveFrame()) {
        for (FrameH::history_iterator itr=mFrame->RefHistory().begin();
	     itr != mFrame->RefHistory().end();
	     itr++) {
	    if ( (*itr)->GetName() == name) {
	        t   = Time((*itr)->GetTime());
		com =  (*itr)->GetComment();
		return true;
	    }
	}
    }
    return false;
}

//======================================  Connect to a data source.
int
DaccIn::open(void) {
    if (isOpen()) return 0;
#if LDAS_VERSION_NUMBER < 109360
    if (!mStream)
#else
    if (!mFrameBuf)
#endif
    {
	if (mFile.empty()) {
	    cout << "No more requested files" << endl;
	    return -1;
	}
	int rc = openFile(mFile.first()); // prints error and pops file name
	if (rc) return rc;
    }
    int rc = openReader();
    if (rc) {
        cerr << "Unable to open reader on file: " << getFile() << endl;
        closeFile();               // discard file.
    }
    return rc;
}

//======================================  Disconnect from the data source.
void
DaccIn::close(void) {
    endFrame();
    closeReader();
    closeFile();
}

//======================================  Set the default stride.
void 
DaccIn::setDebug(int level) {
    mDebug = level;
}

//======================================  Clean up when frame no longer needed
void 
DaccIn::endFrame(void) {
    if (!haveFrame()) return;

    //-------------------------- Delete the frame and bump the frame number.
    deleteFrame();

#ifndef DMTOFFLINE
    //-------------------------- Note that the following code attempts to free
    //                           a partition buffer for online code. It isn't
    //                           really correct in that the reader is deleted
    //                           and buffer freed for non-TOC access after the
    //                           first buffer.
    if (isOnline()) {
        if (mTOCMode != s_TOC || mFramesInFile <= mFrameNo) {
#if LDAS_VERSION_NUMBER < 109360
	    dynamic_cast<iSMbuf*>(mSbuf)->relse();
#else
	    stream_ptr<iSMbuf>(mFrameBuf)->relse();
#endif
	    closeReader();
	}
    }
#endif
}

//======================================  Get the next frame  
int
DaccIn::nextFrame(void) {
    if (haveFrame()) deleteFrame();

    while (!haveFrame()) {
        if (!isOpen() && open()) return -4;
	int rc = readFrame();
	if (!rc) {
	    continue;
	} else if (!isOnline()) {
	    close();
	} else {
#ifndef DMTOFFLINE
#if LDAS_VERSION_NUMBER < 109360
	    dynamic_cast<iSMbuf*>(mSbuf)->relse();
#else
	    stream_ptr<iSMbuf>(mFrameBuf)->relse();
#endif
#endif
	    closeReader();
	}
    }

    mT0 = ::getTime(mFrame->GetGTime());
    frrawdata_pointer raw = getFrame()->GetRawData();
    if (raw) mAdcIter = raw->RefFirstAdc().begin();
    mProcIter = getFrame()->RefProcData().begin();
    mSimIter  = getFrame()->RefSimData().begin();
    mTotalFrames++;
    return 0;
}

//======================================  Wait for an online frame to arrive
bool
DaccIn::waitData(bool nowait) const {
#ifndef DMTOFFLINE
    if (!isOnline()) return true;
#if LDAS_VERSION_NUMBER < 109360
    return dynamic_cast<iSMbuf*>(mSbuf)->waitBuf(nowait);
#else
    return stream_ptr<iSMbuf>(mFrameBuf)->waitBuf(nowait);
#endif
#else
    return true;
#endif
}

//======================================  Search entire list for an adc
fradcdata_pointer
DaccIn::findAdc(const std::string& name) const {
#ifdef FCPP_SHARED_PTRS
    fradcdata_pointer p;
#else
    fradcdata_pointer p = 0;
#endif
    if (mFrame) {
	frrawdata_pointer raw = getFrame()->GetRawData();
	if (raw) {
	    mAdcIter = raw->RefFirstAdc().find(name);
	    if (mAdcIter != raw->RefFirstAdc().end()) p = *mAdcIter;
	}
	if (!p && (mTOCMode == s_TOC)) p = readAdc(name);
    }
    return p;
}

//======================================  Search list from previous point
int
DaccIn::findAdcOrdered(const std::string& name, fradcdata_pointer& p) const {
    int rc = 1; // assume the most likely
#ifdef FCPP_SHARED_PTRS
    p.reset();
#else
    p = 0;
#endif
    if (mDebug > 3) cout << "Searching for Adc: " << name << "...";

    //----------------------------------  Find adc with TOC
    if (mTOCMode == s_TOC) {
	p = findAdc(name);
    }

    //----------------------------------  Look for Adc in list
    else {
	//------------------------------  Get raw data structure
	frrawdata_pointer raw = mFrame->GetRawData();
	if (!raw) {
	    if (mDebug) cerr << "No Raw data in frame!" << endl;
	}

	//------------------------------  Search after current pointer
	else {
	    mAdcIter = raw->RefFirstAdc().find(name, mAdcIter);
	    if (mAdcIter != raw->RefFirstAdc().end()) {
		p = *mAdcIter;
	    } else {
		p  = findAdc(name);
		rc = -1;
	    }
	}
    }
    if (!p) rc = 0;
    if (mDebug > 3) {
	if (rc) cout << " Found!" << endl;
	else    cout << " Not found!" << endl;
    }
    return rc;
}

//======================================  Search entire list for ProcData
frprocdata_pointer 
DaccIn::findProc(const std::string& name) const {
#ifdef FCPP_SHARED_PTRS
    frprocdata_pointer p;
#else
    frprocdata_pointer p = 0;
#endif
    if (mFrame) {
	mProcIter = mFrame->RefProcData().find(name);
	if (mProcIter != mFrame->RefProcData().end()) p = *mProcIter;
	if (!p && (mTOCMode == s_TOC)) p = readProc(name);
    }
    return p;
}

//======================================  Search list from previous point
int
DaccIn::findProcOrdered(const std::string& name, frprocdata_pointer& p) const {
    int rc = 1;
#ifdef FCPP_SHARED_PTRS
    p.reset();
#else
    p = 0;
#endif
    if (mDebug > 3) cout << "Searching for FrProc: " << name << " ...";
    if (!mFrame) {
        if (mDebug) cout << "No frame available!" << endl;
    } else if (mTOCMode == s_TOC) {
	p = findProc(name);
    } else {
	const FrameCPP::FrameH::procData_type& pCntr = mFrame->RefProcData();
	mProcIter = pCntr.find(name, mProcIter);
	if (mProcIter != pCntr.end()) {
	    p = *mProcIter;
	} else {
	    p = findProc(name);
	    rc = -1;
	}
    }
    if (!p) rc = 0;

    if ( mDebug > 3) {
	if (rc) cout << " Found!" << endl;
	else    cout << " Not found!" << endl;
    }
    return rc;
}

//======================================  Search entire list for SimData
frsimdata_pointer 
DaccIn::findSim(const std::string& name) const {
#ifdef FCPP_SHARED_PTRS
    frsimdata_pointer p;
#else
    frsimdata_pointer p = 0;
#endif
    if (mFrame) {
	mSimIter = mFrame->RefSimData().find(name);
	if (mSimIter != mFrame->RefSimData().end()) p = *mSimIter;
	if (!p && (mTOCMode == s_TOC)) p = readSim(name);
    }
    return p;
}

//======================================  Search list from previous point
int
DaccIn::findSimOrdered(const std::string& name, frsimdata_pointer& p) const {
    int rc = 1;
#ifdef FCPP_SHARED_PTRS
    p.reset();
#else
    p = 0;
#endif
    if (mDebug > 3) cout << "Searching for FrSim: " << name << " ...";
    if (!mFrame) {
        if (mDebug) cout << "No frame available!" << endl;
    } else if (mTOCMode == s_TOC) {
	p = findSim(name);
    } else {
	const FrameCPP::FrameH::simData_type& pCntr = mFrame->RefSimData();
	mSimIter = pCntr.find(name, mSimIter);
	if (mSimIter != pCntr.end()) {
	    p = *mSimIter;
	} else {
	    p = findSim(name);
	    rc = -1;
	}
    }
    if (!p) rc = 0;

    if ( mDebug > 3) {
	if (rc) cout << " Found!" << endl;
	else    cout << " Not found!" << endl;
    }
    return rc;
}

//======================================  Find a specified static structure
FrStatDataRef 
DaccIn::findStat(const std::string& name, const Time& gps, int vsn) {
    if (mDebug > 1) cerr << "Look for FrStatData: " << name << " version: "
			 << vsn << " for GPS: " << gps.getS() << "... ";
    FrStatDataRef psav;
    Time tLast(0);
    int  pvsn(-1);
    int N = mStatData.size();
    for (int i=0; i<N; i++) {
#ifdef FCPP_SHARED_PTRS
        FrStatDataRef p(mStatData[i]);
#else
        FrStatDataRef p(dynamic_cast<FrStatDataRef::stat_type*>(mStatData[i]));
#endif
	if (name != p.getName()) continue;
	int cvsn = p.getVersion();
	if ((vsn < 0 && cvsn >= pvsn) || cvsn == vsn) {
	    if (!gps && p.getStartTime() > tLast) {
	        psav  = p;
		pvsn  = cvsn;
		tLast = p.getStartTime();
	    } else if (gps >= p.getStartTime() && gps < p.getEndTime()) {
	        psav = p;
		pvsn = cvsn;
	    }
	}
    }
    if (mDebug > 1) {
        if (psav.null()) cerr << "Not Found!" << endl;
	else             cerr << "found!" << endl;
    }
    return psav;
}

//======================================  File primitives
int 
DaccIn::closeFile(void) {
    if (mDebug > 3) cout << "Closing frame file ...";
#if LDAS_VERSION_NUMBER < 109360
    if (mStream) {
        delete mStream;
	mStream = 0;
    }
    if (mSbuf) {
        delete mSbuf;
        mSbuf = 0;
	mFile.pop_front();
    }
#else
    if (mFrameBuf) {
        delete mFrameBuf;
	mFrameBuf = 0;
	mFile.pop_front();
    }
#endif

    mSource = s_none;
    if (mDebug > 3) cout << " Done" << endl;
    return 0;
}

//======================================  Open a file.
int 
DaccIn::openFile(const std::string& Source) {

    //----------------------------------  Request online data (ignore start)
    int nc = strlen(DACC_ONLDEV);
    if (Source.substr(0, nc) == DACC_ONLDEV) {
	const char* part = Source.c_str() + nc;
#ifndef DMTOFFLINE
        if (mDebug > 3) cout << "Opening partition ..." << endl;

#if LDAS_VERSION_NUMBER < 109360
        iSMbuf* buf = new iSMbuf;
        mSbuf = buf->open(part, ios::in);
	if (!mSbuf) {
	    cerr << "Unable to open partition " << part << endl;
	    mFile.pop_front();
	    return -1;
	}

	mStream = new std::istream(mSbuf);
	if (!mStream) return -1;
#else
	mFrameBuf = new FrameBuffer< iSMbuf >(ios::in);
	if (!mFrameBuf) {
	    cerr << "Unable to construct Framebuffer<iSMbuf> for partition " 
		 << part << endl;
	    mFile.pop_front();
	    return -1;
	}
	if (!stream_ptr<iSMbuf>(mFrameBuf)->open(part, ios::in)) {
	    cerr << "Unable to open partition " << part << endl;
	    mFile.pop_front();
	    delete mFrameBuf;
	    mFrameBuf = 0;
	    return -1;
	}
#endif
	if (mDebug) {
	    cout << "Opened partition " << part << " for frame input." 
		 << endl;
	}
	mSource = s_online;
#else
	cerr << "Shared memory partitions not implemented in offline version"
	     << endl;
	return -1;
#endif

    //----------------------------------  Forget about offline for now
    } else {
        if (mDebug > 3) cout << "Opening file ..." << endl;
#if LDAS_VERSION_NUMBER < 109360
	filebuffer_type* fbuf = new filebuffer_type;
	mSbuf = fbuf->open(Source.c_str(), ios::in);
	if (!mSbuf) {
	    cerr << "Unable to open file " << Source << endl;
	    mFile.pop_front();
	    return -1;
	}

	mStream = new std::istream(mSbuf);
	if (!mStream) return -1;
#else
	FrameBuffer< filebuffer_type >* 
	    fbuf = new FrameBuffer<filebuffer_type>(ios::in);
	fbuf->open(Source.c_str(), ios::in);
	mFrameBuf = fbuf;
#endif
	if (mDebug) {
	    cout << "Opened file " << Source << " for frame input." << endl;
	}
	mSource = s_file;
    }

    return 0;
}

//======================================  Reader primitives
int 
DaccIn::closeReader(void) {
    int rc = 0;
    if (!mReader) return rc;
    mFramesInFile = 0;

    try {
        delete mReader;
    } catch (exception& e) {
        cerr << "Exception in DaccIn::close() when deleting IFrameStream: " 
	     << e.what() << endl;
	rc = -1;
    } catch (...) {
        cerr << "Unidentified exception caught." << endl;
	rc = -1;
    }
#if LDAS_VERSION_NUMBER < 109360
    mReader   = 0;
#else
    mReader   = 0;
    //----------------------------------  delete source file name when the 
    //                                    frambuffer is closed
    if (mSource == s_file) {
	mFile.pop_front();
	mFrameBuf = 0;
    }
#endif
    return rc;
}

//======================================  Open a frame reader
int 
DaccIn::openReader(void) {
    if (mDebug > 3) cout << "Opening stream reader ...";
    try {
#if LDAS_VERSION_NUMBER < 109360
        mReader = new IFrameStream(*mStream);
#else
	mReader = new IFrameStream(mFrameBuf);
	if (isOnline()) mReader->SetAutoDeleteBuffer(false);
#endif
    } catch (exception& e) {
        cerr << "Exception constructing IFrameStream: " << e.what()<< endl;
	return -1;
    } catch (...) {
        cerr << "Unidentified exception caught." << endl;
	return -1;
    }
    if (mDebug > 3) cout << " Done!" << endl;

    //----------------------------------  Set last frame field.
    if (isOnline()) {
#ifndef DMTOFFLINE
	ostringstream oss;
	oss << mFile.first() << "-" << stream_ptr<iSMbuf>(mFrameBuf)->eventid()
	    << ".gwf";
	mLastFrame = oss.str();
#else
	mLastFrame.clear();
#endif
    }
    else {
	mLastFrame =  mFile.first();
    }
    mFrameNo = 0;
    return 0;
}

//======================================  Frame primitives
int 
DaccIn::deleteFrame(void) {
    try {
#ifdef FCPP_SHARED_PTRS
	mFrame.reset();
#else
        delete mFrame;
	mFrame = 0;
#endif
    } catch (exception& e) {
        cerr << "deleteFrame: Exception in frame destructor: " 
	     << e.what() << endl;
    } catch (...) {
        cerr << "deleteFrame: Unexpected exception." << endl;
    }
    mFrameNo++;
    return 0;
}

int 
DaccIn::readFrame(void) {
    if (mDebug > 3) cout << "Reading next frame...";
    if (haveFrame() || !isOpen()) return -1;
    try {
        frrawdata_pointer raw;
        switch (mTOCMode) {
	case s_noTOC:
	    //--------------------------  Read a frame. Versions of framecpp
	    //                            sometimes throws an exception if
	    //                            no more frames in reall mode.
	    try {
		mFrame = mReader->ReadNextFrame();
	    } 

	    //--------------------------  Exception
	    catch (std::exception & e) {
		if (mDebug) cerr << "Exception in readFrame: " << e.what() 
				 << endl;
#ifdef FCPP_SHARED_PTRS
		mFrame.reset();
#else
		mFrame = 0;
#endif
	    }
	
	    //--------------------------  Check for successful read.
	    if (!haveFrame()) {
		if (!mFrameNo) cerr << "File has no frames!" << endl;
		return -1;
	    }

	    raw = mFrame->GetRawData();
	    if (raw) mAdcIter = raw->RefFirstAdc().begin();
	    break;
	case s_TOC:
	    if (mFrameNo && mFramesInFile <= mFrameNo) return -1;
	    unsigned int rflgs = FrameH::DETECT_SIM | FrameH::DETECT_PROC 
	                       | FrameH::HISTORY;
	    mFrame = mReader->ReadFrameH(mFrameNo, rflgs);
	    if (!haveFrame()) {
	        cerr << "Unable to read frame header!" << endl;
#ifdef ISMBUF_TRACE
#if LDAS_VERSION_NUMBER < 109360
		if (mSbuf && isOnline()) {
		    dynamic_cast<iSMbuf*>(mSbuf)->pTrace();
		}
#else
		if (mFrameBuf && isOnline()) {
		    stream_ptr<iSMbuf>(mFrameBuf)->pTrace();
		}
#endif
#endif
		return -1;
	    }
	    //raw = mReader->ReadFrRawData(mFrameNo);
#if LDAS_VERSION_NUMBER < 109360
	    raw = new FrRawData;
#else
	    raw = frrawdata_pointer(new FrRawData);
#endif
	    if (raw) {
#if LDAS_VERSION_NUMBER < 109360
	        mFrame->SetRawData(raw, false, true);
#else
	        mFrame->SetRawData(raw);
#endif
		mAdcIter = raw->RefFirstAdc().begin();
	    }
	    if (!mFrameNo) {
#if LDAS_VERSION_NUMBER < 109360
		mFramesInFile = mReader->GetNumberOfFrames();
#else
		mFramesInFile = mReader->GetTOC()->nFrame();
#endif
	    }
	}
	if (mDebug > 2) cout << "Frame read successfully. ID: " 
			     <<  mFrame->GetFrame() << endl;
    } catch (exception& e) {
#ifdef FCPP_SHARED_PTRS
	mFrame.reset();
#else
        mFrame = static_cast<FrameH*>(0);
#endif
	if (mDebug) {
	    cerr << "Exception reading Frame: " << e.what() << endl;
#if LDAS_VERSION_NUMBER < 109360
	    cerr << "Error occurred at offset " 
		 << mSbuf->pubseekoff(0, ios_base::cur, ios_base::in)
		 << endl;
#else
	    cerr << "Error occurred at offset " 
		 << stream_ptr<filebuffer_type>(mFrameBuf)->pubseekoff(0, ios_base::cur, ios_base::in)
		 << endl;
#endif
	}
	return -1;
    } catch (...) {
#ifdef FCPP_SHARED_PTRS
	mFrame.reset();
#else
        mFrame = static_cast<FrameH*>(0);
#endif
	if (mDebug) cerr << "Unidentified exception caught." << endl;
	return -1;
    }
    if (mDebug > 3) cout << " Done" << endl;
    return 0;
}

fradcdata_pointer
DaccIn::readAdc(const string& name) const {
    frrawdata_pointer raw = mFrame->GetRawData();
#ifndef FCPP_SHARED_PTRS
    if (!raw) {
        raw = new FrRawData();
	mFrame->SetRawData(raw, false, true);
    }
#else
    if (!raw) {
        raw = frrawdata_pointer(new FrRawData());
	mFrame->SetRawData(raw);
    }
#endif
    if (getDebug() > 4) cout << "Slurp frame: " << mFrameNo 
			     << " Adc: " << name << endl;
    fradcdata_pointer p;
    try {
	p = mReader->ReadFrAdcData(mFrameNo, name);
#ifndef FCPP_SHARED_PTRS
	if (p) mAdcIter = raw->RefFirstAdc().append(p, false, true);
#else
	if (p) mAdcIter = raw->RefFirstAdc().append(p);
#endif
	else   mAdcIter = raw->RefFirstAdc().end();
    }

    //----------------------------------  Invalid frame data.
    catch (FrameCPP::Common::VerifyException& e) {
#ifndef FCPP_SHARED_PTRS
	p = 0;
#endif
	cerr << "Exception reading Adc " << name << ": " << e.what() << endl;
    }

    //----------------------------------  Missing data or other error.
    catch (exception& e) {
#ifndef FCPP_SHARED_PTRS
	p = 0;
#endif
	if (mDebug > 2) cerr << "DaccIn: Exception reading Adc " << name 
			     << ": " << e.what() << endl;
    }
    return p;
}

//======================================  Read an FrProcData structure.
frprocdata_pointer
DaccIn::readProc(const string& name) const {
    if (getDebug() > 4) cout << "Slurp frame: " << mFrameNo 
			     << " FrProcData: " << name << endl;
    frprocdata_pointer p;
    try {
	p = mReader->ReadFrProcData(mFrameNo, name);
#ifndef FCPP_SHARED_PTRS
	if (p) mProcIter = mFrame->RefProcData().append(p, false, true);
#else
	if (p) mProcIter = mFrame->RefProcData().append(p);
#endif
	else   mProcIter = mFrame->RefProcData().end();
    } 

    //----------------------------------  Catch invalid frame structure.
    catch (FrameCPP::Common::VerifyException& e) {
#ifndef FCPP_SHARED_PTRS
	p = 0;
#endif
	cerr << "DaccIn: Error in frame structure while reading FrProcData " 
	     << name << ": " << e.what() << endl;
    }

    //----------------------------------  Other exception (missing channel?)
    catch (exception& e) {
#ifndef FCPP_SHARED_PTRS
	p = 0;
#endif
	if (mDebug > 2) cerr << "DaccIn: Exception reading FrProcData " << name 
			     << ": " << e.what() << endl;
    }
    return p;
}

//======================================  Read an FrSimData structure.
frsimdata_pointer
DaccIn::readSim(const string& name) const {
    if (mDebug > 4) cout << "Slurp frame: " << mFrameNo 
			 << " FrSimData: " << name << "... ";
    frsimdata_pointer p;
    try {
        p = mReader->ReadFrSimData(mFrameNo, name);
	if (mDebug > 4) {
	    if (p) cout << "Found!" << endl;
	    else   cout << "Not Found!" << endl;
	}
#ifndef FCPP_SHARED_PTRS
	if (p) mSimIter = mFrame->RefSimData().append(p, false, true);
#else
	if (p) mSimIter = mFrame->RefSimData().append(p);
#endif
  	else   mSimIter = mFrame->RefSimData().end();
    } 

    //----------------------------------  Catch invalid frame structure
    catch (FrameCPP::Common::VerifyException& e) {
#ifndef FCPP_SHARED_PTRS
        p = 0;
#endif
	cerr << "Erroneous frame structure while reading FrSimData "
	     << e.what() << endl;
    }

    //----------------------------------  Other exception (missing structure?)
    catch (exception& e) {
#ifndef FCPP_SHARED_PTRS
        p = 0;
#endif
	if (mDebug > 4) cout << "Failed!" << endl;
	if (mDebug > 2) cerr << "Exception reading FrSimData " << name 
			     << ": " << e.what() << endl;
    }
    return p;
}

#if !defined(DMTOFFLINE) && LDAS_VERSION_NUMBER >= 109360
#include "framecpp/Common/IOStream.icc"
 template class FrameCPP::Common::FrameBuffer< iSMbuf >;
#endif
