/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DACC_API_HH
#define DACC_API_HH

#include "Interval.hh"
#include "Time.hh"
#include <string>
#include <iosfwd>

class TSeries;

/**  %DaccAPI is a purely virtual class that defines the API used by the 
  *  DMT. It will be used to allow the traditional Dacc class and the new
  *  ChanApi class to be intrerchanged transparently.
  */
class DaccAPI {
public:
    /**
     *  Virtual destructor to make everything go away nicely.
     *  \brief Destructor.
     */
    virtual ~DaccAPI(void);

    /**  The specified channel is added to the list of channels to be
      *  processed by fillData(). fillData() looks for the channel name 
      *  first in the FrAdcData list folowed by the FrProcData list. A 
      *  channel may be included in the list 
      *  only once. Subsequent calls to addChannel() will replace the 
      *  original request. \a TSptr is used by the calling program to access 
      *  the data. If the pointer address is \c NULL or not specified, an 
      *  internal pointer is allocated, and the channel is accessed 
      *  exclusively with the refData() method. The TSeries is then created 
      *  and destroyed under Dacc control. If the pointer address is 
      *  specified a non-zero pointer indicates a pre-allocated
      *  TSeries. If the pointer is \c NULL no TSeries is pre-allocated
      *  and the TSeries will be created when fillData() is invoked. Note 
      *  that if the TSeries is not preallocated, the TSeries data vector 
      *  will have the same data type as the channel data (usually short). 
      *  \a Decimate gives the number of samples to be averaged together
      *  before they are entered into the TSeries. If \a Decimate \<= 0, it
      *  is set to 1. If no TSeries is passed to fillData() and the 
      *  Decimation \> 1, a \c float data vector is allocated.
      *  \brief Add a channel to the request list.
      *  \param Name     Name of channel to collect data from.
      *  \param Decimate Decimation factor.
      *  \param TSptr    Address of TSeries pointer.
      */
    virtual void addChannel(const std::string& Name, int Decimate=0, 
			    TSeries **TSptr=0) = 0;

    /**  Set / add a new input data source path.
      *  \brief Set the input data path.
      *  \param path New input file path.
      */
    virtual void addPath(const std::string& path) = 0;

    /**  Add a list of new input data source paths.
      *  \brief Add a list of input data paths.
      *  \param path New input file path.
      */
    virtual void addPathList(const std::string& path) = 0;

    /**
     *  Close the input data file or disconnect from the server as appropriate.
     *  The DaccAPI channel list is left intact, but the courrent path is 
     *  discarded.
     *  \brief Close the data accessor.
     *  \return Standard DaccAPI return codes.
     */
    virtual void close(void) = 0;

    /**  Data from channels specified by addChannel() are copied to TSeries
      *  objects.
      *  \brief  Fill the requested TSeries.
      *  \param Stride Time interval to be read in.
      *  \param start  Start a stride.
      *  \return 
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Frame start not contiguous to previous data.</td>
      *        </tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in current frame.</td>
      *        </tr>
      *    <tr><td>-4</td><td>Error reading frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *    <tr><td>-8</td><td>Signal received while reading.</td></tr>
      *    <tr><td>-9</td><td>Invalid data in structure.</td></tr>
      *  </table>
      */
    virtual int fillData(Interval Stride=Interval(0), bool start=true) = 0;

    /**  Returns the current data collection time i.e. the time of the 
      *  first sample not used by fillData. getCurrentTime() returns 
      *  Time(0) if no frame has been read. Note that data filling may 
      *  not start at getCurrentTime if there are gaps in the data stream
      *  or if fillData() errors occur. Use synch to position the current 
      *  time to the start of the next frame if current frame is not 
      *  available.
      *  \brief  Get the Current time.
      *  \return Start time for next data fill.
      */
    virtual Time getCurrentTime(void) const = 0;

    /**  Returns a pointer to a string containing the current input file 
      *  path. If no file is open, a pointer to the next file name is 
      *  returned.
      *  \brief Get the input file name.
      *  \return Current frame file path.
      */
    virtual const char* getFile(void) const = 0;

    /**  An identifier of the form \c \<ifo-id\>-\<GPS-time\>.F is generated 
      *  for the current Frame and copied to the specified string.
      *  \brief Copy a Frame ID into the specified string.
      *  \return string containing the current frame identifier.
      */
    virtual std::string getFrameID(void) const = 0;

    /**  Returns the time at which data collection started with the
      *  last invocation of fillData.
      *  \brief Get the start time of the last fill.
      *  \return Start time of last fillData() stride.
      */
    virtual Time getFillTime(void) const = 0;

    /**  Returns an offset of the current point in the frame.
      *  \brief Get the current offset into the frame.
      *  \return Current offset in frame data. 
      */
    virtual Interval getOffset(void) const = 0;

    /**  Returns the time interval over which data are collected.
      *  \brief Get the stride interval.
      *  \return Default stride interval.
      */
    virtual Interval getStride(void) const = 0;

    /**  Test to see is the names channel has been requested..
      *  \brief Test if channel was requested.
      *  \return True if the specified channel has been requested.
      *  \param Name Channel name to be tested.
      */
    virtual bool isChannelRead(const std::string& Name) const = 0;

    /**
     *  Test whether the input data set is online i.e. streaming real-time
     *  data.
     *  \brief Test for online data.
     *  \return True if this instance is streaming real-time data;
     */
    virtual bool isOnline(void) const = 0;

    /**  A formatted list of requested channels, their decimation factors, 
      *  and latest time copied is printed to the specified output stream. 
      *  \brief List requested channels.
      *  \param out STL output stream to which channels are listed.
      *  \return Output stream reference.
      */
    virtual std::ostream& list(std::ostream& out) const = 0;

    /**  Write a list of input files to the specified STL output stream.
      *  \brief Write frame file list.
      *  \param out STL output stream to receive list
      *  \return Output stream reference.
      */
    virtual std::ostream& listFiles(std::ostream&) const = 0;

    /**
     *  Open the input data file or connect to the appropriate server
     *  to receive the first listed data.
     *  \brief Open the data accessor.
     *  \return Standard DaccAPI return codes.
     */
    virtual int open(void) = 0;

    /**  Returns a pointer to the last %TSeries filled for the named channel.
      *  \brief Get a pointer to the data.
      *  \param name Channel name string pointer.
      *  \return Constant pointer to current data %TSeries.
      */
    virtual const TSeries* refData(const std::string& name) const = 0;

    /**  Returns a pointer to the last %TSeries filled for the named channel.
      *  \brief Get a pointer to the data.
      *  \param name Channel name string pointer.
      *  \return Constant pointer to current data %TSeries.
      */
    virtual TSeries* refData(const std::string& name) = 0;

    /**  The named channel is removed from the request list.
      *  \brief Remove a channel from the request list.
      *  \param Name Name of channel to be removed.
      */
     virtual void rmChannel(const std::string& Name) = 0;

    /**  Frames are read from the specified input file(s) until one 
      *  containing the specified time is found or a time later than 
      *  than \a STime is encountered. If \a STime is not specified or 
      *  specified as %Time(0), the result is equivalent to synch().
      *  The time of the first sample can be obtained with getCurrentTime().
      *  \brief Skip to the specified time.
      *  \param STime Next GPS to be read.
      *  \return Return value indicates the status as follows:
      *    * 0:  Successful completion
      *    * -4: No more data frames.
      *    * -8: No data currently available and NoWait was specified.
      */
    virtual int seek(Time STime=Time(0)) = 0;

    /**  If the debug flag is set, explanatory messages will be printed 
      *  when an error is encountered.
      *  \brief Set the debug flag.
      *  \param N Debug print level.
      */
    virtual void setDebug(int N) = 0;

    /**  Set or clear the IgnoreMissingChannel flag. If this flag is set, a 
      *  call to fillData will succede (\e i.e the return code will be 0)
      *  even if one of more channels are not found.
      *  \brief Set the "Ignore missing channel" flag.
      *  \param yn  New setting for the "Ignore missing channel" flag.
      */
    virtual void setIgnoreMissingChannel(bool yn) = 0;

    /**  The default stride is used if a time interval is not specified
      *  when fillData() is invoked.
      *  \brief Define the default Stride.
      *  \param Dt Default data stride length.
      */
     virtual void setStride(Interval Dt) = 0;

    /**  The frame that will provide that the data for the next read operation 
      *  is read into memory. If unread input data are already in the buffer,
      *  no further data are read or discarded. The current time is set to the 
      *  start of the newly read frame.
      *  \brief Synchronize the current frame buffer.
      *  \return Return value indicates the status as follows:
      *    * 0:  Successful completion
      *    * -4: No more data frames.
      *    * -8: No data currently available and NoWait was specified.
      */
    virtual int synch(void) = 0;

    /**  Test whether data are available. The function will optionally block 
      *  until data are available or an interrupt occurs.
      *  \brief Test for data
      *  \param wait If true the method will block until data are available
      *  \return True if data are available.
      */
    virtual bool testData(bool wait) const = 0;
};

//======================================  Inline DaccAPI methods
inline
DaccAPI::~DaccAPI(void) {
}

#endif // !defiend(DACC_API_HH)
