#include "TrigMgrLog.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "html/color.hh"
#include "html/compound.hh"
#include "html/genAttr.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"
#include "html/writer.hh"

#include <fstream>

using namespace html;
using namespace std;

//======================================  Construct the html log document
TrigMgrLog::TrigMgrLog(const std::string& name) 
  : mFile("TrigMgrLog.html"), mTrigLog(name.c_str()), mLogUpdate(false)
{

    //----------------------------------  Set global style parameters
    mTrigLog.setBackgroundColor(color("white"));
    mTrigLog.setRefresh(15.0);

    //----------------------------------  Define a style for each priority
    StyleClass sty;
    mTrigLog.setStyle("pr_i", sty.setColor(color("green")));
    mTrigLog.setStyle("pr_w", sty.setColor(color("orange")));
    mTrigLog.setStyle("pr_e", sty.setColor(color("magenta")));
    mTrigLog.setStyle("pr_s", sty.setColor(color("red")));

    //----------------------------------  Text
    compound hdrtxt;
    hdrtxt.addObject(text("This page contains a list of the most recent 100 ")); 
    hdrtxt.addObject(text("triggers recorded by the trigger manager. The ")); 
    hdrtxt.addObject(text("trigger severity is indicated by color as follows: ")); 
    table sevtbl;
    sevtbl.addColumn("Severity");
    sevtbl.addColumn("Meaning");
    int i = sevtbl.addRow();
    sevtbl.insertData(i, 0, "Information");
    sevtbl.refCell(i, 0).addAttr("class", genAttr("pr_i"));
    sevtbl.insertData(i, 1, "Status information only");
    i = sevtbl.addRow();
    sevtbl.insertData(i, 0, "Warning");
    sevtbl.refCell(i, 0).addAttr("class", genAttr("pr_w"));
    sevtbl.insertData(i, 1, "No effect is expected on GW data");
    i = sevtbl.addRow();
    sevtbl.insertData(i, 0, "Error");
    sevtbl.refCell(i, 0).addAttr("class", genAttr("pr_e"));
    sevtbl.insertData(i, 1, "Trigger should be considered during GW analysis.");
    i = sevtbl.addRow();
    sevtbl.insertData(i, 0, "Severe");
    sevtbl.refCell(i, 0).addAttr("class", genAttr("pr_s"));
    sevtbl.insertData(i, 1, "Data are not valid for GW analysis");
    hdrtxt.addObject(sevtbl);
    mTrigLog.addObject(hdrtxt);

    //----------------------------------  Add a header
    text ttl("Trigger Log");
    ttl.setSize(size(2));
    block* cb = dynamic_cast<block*>(&mTrigLog.addObject(block("center")));
    mLogDate  = dynamic_cast<text* >(&(cb->addObject(ttl)));

    //----------------------------------  Build the table
    mLogTable = dynamic_cast<table*>(&mTrigLog.addObject(table()));
    mLogTable->addColumn("GPS");
    mLogTable->addColumn("Trigger");
    mLogTable->addColumn("Sub-ID");
    mLogTable->addColumn("Process");
    mLogTable->addColumn("Size");
    mLogTable->addColumn("Duration");

    //----------------------------------  Initialize the time stamp
    mLogStamp = Now();
}

//======================================  Destroy the html log
TrigMgrLog::~TrigMgrLog(void) {}

//======================================  Construct the html log document
void
TrigMgrLog::addTrigger(const trig::TrigBase& t, const trig::TrigProc& p) {
    int nRow = mLogTable->getNRow();
    if (nRow >= Max_Log_Rows) {
        mLogTable->eraseRow(Max_Log_Rows-1, nRow-Max_Log_Rows+1);
    }
    mLogTable->insertRow(0);
    mLogTable->insertData(0, 0, long(t.getTime().getS()));
    mLogTable->insertData(0, 1, t.getID());
    mLogTable->insertData(0, 2, t.getSubID());
    mLogTable->insertData(0, 3, p.getName());
    mLogTable->insertData(0, 4, t.getIntensity());
    mLogTable->insertData(0, 5, t.getDt());

    //----------------------------------  Set the line color
    genAttr prio("");
    switch (t.getPriority()) {
    case trig::p_info:
        prio.setValue("pr_i");
	break;
    case trig::p_warn:
        prio.setValue("pr_w");
	break;
    case trig::p_error:
        prio.setValue("pr_e");
	break;
    case trig::p_severe:
        prio.setValue("pr_s");
	break;
    default:
	break;
    }
    for (int i=0 ; i<6 ; i++) {
        (mLogTable->refCell(0,i)).addAttr("class", prio);
    }
    mLogUpdate = true;
}

//======================================  Construct the html log document
void
TrigMgrLog::writeLog(void) {
    if (!mLogUpdate || mLogStamp + Interval(15.0) > Now()) return;
    char Date[32];
    mLogStamp = Now();
    LocalStr(mLogStamp, Date, "%M %d, %Y %02H:%02N");
    std::string name = string("Trigger Log:  ") + Date;
    mLogDate->insertText(name);
    std::ofstream out(mFile.c_str(), ios::out);
    writer w(out);
    mTrigLog.write(w);
    mLogUpdate = false;
}

void
TrigMgrLog::setFile(const std::string& file) {
    mFile = file;
}
